/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.JSValidationException;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.common.service.ReportExecutionStatusInformation;
import com.jaspersoft.jasperserver.api.engine.common.service.SchedulerReportExecutionStatusSearchCriteria;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitRequest;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataCacheProvider;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataCacheProvider.SnapshotSaveStatus;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataSnapshotService;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControlsContainer;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.OperationResult;
import com.jaspersoft.jasperserver.remote.ReportExporter;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.ServicesConfiguration;
import com.jaspersoft.jasperserver.remote.ServicesUtils;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.xml.ErrorDescriptor;
import com.jaspersoft.jasperserver.remote.services.RunReportService;
import com.jaspersoft.jasperserver.remote.utils.AuditHelper;
import com.jaspersoft.jasperserver.remote.utils.RepositoryHelper;
import com.jaspersoft.jasperserver.war.cascade.CachedEngineService;
import com.jaspersoft.jasperserver.war.cascade.CascadeResourceNotFoundException;
import com.jaspersoft.jasperserver.war.cascade.InputControlsLogicService;
import com.jaspersoft.jasperserver.war.cascade.InputControlsValidationException;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;
import com.jaspersoft.jasperserver.ws.xml.ByteArrayDataSource;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.ReportContext;
import net.sf.jasperreports.engine.SimpleReportContext;
import net.sf.jasperreports.engine.export.GenericElementReportTransformer;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.util.JRSaver;
import net.sf.jasperreports.engine.util.JRTypeSniffer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;

import javax.activation.DataSource;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;

/**
 * Run Report service
 * Run a report unit using the passing in parameters and options
 *
 * @author gtoffoli
 * @version $Id: RunReportServiceImpl.java 23956 2012-05-30 12:23:37Z lchirita $
 */
@Component
@Scope(value = "session", proxyMode = ScopedProxyMode.TARGET_CLASS)
public class RunReportServiceImpl implements RunReportService {

    private final static Log log = LogFactory.getLog(RunReportServiceImpl.class);
    private static final String RAW_REPORT_PARAMETERS_FOR_CACHE_KEY_SUFFIX = "rawReportParametersForCacheKeySuffix";
    private static final String KEY_JASPER_PRINT = "jasperPrint";
    private static final String KEY_JASPER_PRINT_RESOURCE = "jasperPrint";
    @javax.annotation.Resource
    private AuditHelper auditHelper;
    @javax.annotation.Resource
    private ServicesUtils servicesUtils;
    @javax.annotation.Resource(name = "concreteRepository")
    private RepositoryService repository;
    @javax.annotation.Resource(name = "engineService")
    private EngineService engine;
    @Autowired
    private MessageSource messageSource;
    @javax.annotation.Resource(name = "remoteServiceConfiguration")
    private ServicesConfiguration servicesConfiguration;
    @javax.annotation.Resource
    private Set<String> technicalParameterNames;
    @javax.annotation.Resource
    private InputControlsLogicService inputControlsLogicService;
    @javax.annotation.Resource
    private CachedEngineService cachedEngineService;
    private AutoincrementalIntegerBidirectionalMapping<String> cacheKeysMapping = new HashMapBasedAutoincrementalIntegerBidirectionalMapping<String>();
    
    @javax.annotation.Resource(name = "dataSnapshotService")
    private DataSnapshotService dataSnapshotService;
    
    @javax.annotation.Resource(name = "engineServiceDataCacheProvider")
    private DataCacheProvider dataCacheProvider;

    private final Map<String, Object> attributes = new HashMap<String, Object>();
    private final Map<String, DataSource> outputAttachments = new HashMap<String, DataSource>();
    private final Map<String, DataSource> inputAttachments = new HashMap<String, DataSource>();
    private volatile Map<Integer, Map<String, ReportOutputResource>> reportsOuptutCache = new HashMap<Integer, Map<String, ReportOutputResource>>();
    private volatile Map<Integer, JasperPrint> jasperPrintMap = new HashMap<Integer, JasperPrint>();

    public void setInputControlsLogicService(InputControlsLogicService inputControlsLogicService) {
        this.inputControlsLogicService = inputControlsLogicService;
    }

    public InputControlsLogicService getInputControlsLogicService() {
        return inputControlsLogicService;
    }

    public void setTechnicalParameterNames(Set<String> technicalParameterNames) {
        this.technicalParameterNames = technicalParameterNames;
    }

    public Map<String, DataSource> getInputAttachments() {
        return inputAttachments;
    }

    public Map<String, DataSource> getOutputAttachments() {
        return outputAttachments;
    }

    public Map<String, Object> getAttributes() {
        return attributes;
    }

    public void setServicesConfiguration(ServicesConfiguration servicesConfiguration) {
        this.servicesConfiguration = servicesConfiguration;
    }

    public void setMessageSource(MessageSource messageSource) {
        this.messageSource = messageSource;
    }

    public void setEngine(EngineService engine) {
        this.engine = engine;
    }

    public void setRepository(RepositoryService repository) {
        this.repository = repository;
    }

    public RepositoryService getRepository() {
        return repository;
    }

    public void setServicesUtils(ServicesUtils servicesUtils) {
        this.servicesUtils = servicesUtils;
    }

    public void setAuditHelper(AuditHelper auditHelper) {
        this.auditHelper = auditHelper;
    }

	public DataSnapshotService getDataSnapshotService() {
		return dataSnapshotService;
	}

	public void setDataSnapshotService(DataSnapshotService dataSnapshotService) {
		this.dataSnapshotService = dataSnapshotService;
	}

	public DataCacheProvider getDataCacheProvider() {
		return dataCacheProvider;
	}

	public void setDataCacheProvider(DataCacheProvider dataCacheProvider) {
		this.dataCacheProvider = dataCacheProvider;
	}


    /**
     * Return a response.
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param reportUnitURI
     * @param parameters
     * @param arguments
     * @return
     * @throws ServiceException
     */
    public OperationResult runReport(String reportUnitURI, Map<String, Object> parameters, Map<String, String> arguments) throws ServiceException {

        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");

        OperationResult or = servicesUtils.createOperationResult(OperationResult.SUCCESS, null);

        try {
            Resource reportResource = repository.getResource(null, reportUnitURI);

            RunReportStrategy strategy = getStrategyForReport(reportResource);
            if (strategy == null) {
                throw new ServiceException(HttpServletResponse.SC_BAD_REQUEST,
                        messageSource.getMessage("webservices.error.notValidReportUnit", new Object[]{reportUnitURI}, LocaleContextHolder.getLocale())
                );
            }

            ReportExecutionOptions executionOptions = new ReportExecutionOptions();
        	executionOptions.setFreshData(Boolean.parseBoolean(arguments.get(Argument.FRESH_DATA)));
        	executionOptions.setSaveDataSnapshot(Boolean.parseBoolean(arguments.get(Argument.SAVE_DATA_SNAPSHOT)));
            
            // run the report
            ReportUnitResult result = strategy.runReport(reportResource, parameters, engine, executionOptions);


            if (result == null) {

                throw new ServiceException(ServiceException.FILL_ERROR,
                        messageSource.getMessage("webservices.error.errorExecutingReportUnit",
                                new Object[]{reportUnitURI}, LocaleContextHolder.getLocale())
                );

            } else {

                JasperPrint jasperPrint = result.getJasperPrint();

                or = exportReport(reportUnitURI, jasperPrint, arguments);

                // The jasperprint may have been transformed during export...
                this.getAttributes().put(KEY_JASPER_PRINT_RESOURCE, jasperPrint);
            }

            if (or.getReturnCode() != 0) {
                auditHelper.addExceptionToAllAuditEvents(new Exception(or.getMessage()));
            }

        } catch (ServiceException e) {
            log.error("caught exception: " + e.getMessage(), e);
            or.setReturnCode(e.getErrorCode());
            or.setMessage(e.getMessage());
            auditHelper.addExceptionToAllAuditEvents(e);

        } catch (Throwable e) {
            log.error("caught Throwable exception: " + e.getMessage(), e);
            e.printStackTrace(System.out);
            System.out.flush();
            or.setReturnCode(1);
            or.setMessage(e.getMessage());
            auditHelper.addExceptionToAllAuditEvents(e);
        }

        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
        return or;
    }


    /**
     * Export the report in a specific format using the specified arguments
     * Generated files (one or more) are put in the output attachments map of this context
     *
     * @param jasperPrint JasperPrint
     * @param arguments   - indicates the final file format, starting/ending pages, etc...
     * @return OperationResult
     * @throws ServiceException
     */
    public OperationResult exportReport(String reportUnitURI, JasperPrint jasperPrint, Map<String, String> arguments) throws ServiceException {

        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");

        OperationResult or = servicesUtils.createOperationResult(OperationResult.SUCCESS, null);

        try {
            String format = arguments.get(Argument.RUN_OUTPUT_FORMAT);
            if (format == null) format = Argument.RUN_OUTPUT_FORMAT_PDF;
            format = format.toUpperCase();

            String transformerKey = arguments.get(Argument.RUN_TRANSFORMER_KEY);

            // Export...
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ByteArrayDataSource bads = null;

            if (format.equals(Argument.RUN_OUTPUT_FORMAT_JRPRINT)) {
                if (log.isDebugEnabled()) {
                    log.debug("Returning JasperPrint");
                }
                if (transformerKey != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Transforming JasperPrint generic element for key " + transformerKey);
                    }
                    GenericElementReportTransformer.transformGenericElements(jasperPrint, transformerKey);
                }

                JRSaver.saveObject(jasperPrint, bos);
                bads = new ByteArrayDataSource(bos.toByteArray());
                getOutputAttachments().put(KEY_JASPER_PRINT_RESOURCE, bads);
            } else {
                //List jasperPrintList = new ArrayList();
                //jasperPrintList.add(jasperPrint);

                HashMap exportParameters = new HashMap();

                String value = arguments.get(Argument.RUN_OUTPUT_PAGE);
                if (value != null) {
                    exportParameters.put(Argument.RUN_OUTPUT_PAGE, value);
                }

                value = arguments.get(Argument.RUN_OUTPUT_IMAGES_URI);
                if (value != null) exportParameters.put(Argument.RUN_OUTPUT_IMAGES_URI, value);

                Map exporterParams;
                try {
                    exporterParams = exportReport(reportUnitURI, jasperPrint, format, bos, exportParameters);
                    if (log.isDebugEnabled()) {
                        log.debug("Exporter params: " + Arrays.asList(exporterParams.keySet().toArray()));
                    }
                } catch (Exception e) {
                    log.error("Error exporting report", e);
                    throw new ServiceException(ServiceException.EXPORT_ERROR,
                            messageSource.getMessage("webservices.error.errorExportingReportUnit",
                                    new Object[]{e.getMessage()}, LocaleContextHolder.getLocale()));
                } finally {
                    if (bos != null) {
                        try {
                            bos.close();
                        } catch (IOException ex) {
                            log.error("caught exception: " + ex.getMessage(), ex);
                        }
                    }
                }

                bads = new ByteArrayDataSource(bos.toByteArray(), getContentType(format));
                getOutputAttachments().put("report", bads);
                addAdditionalAttachmentsForReport(jasperPrint, format, exporterParams);
            }

            if (or.getReturnCode() != 0) {
                auditHelper.addExceptionToAllAuditEvents(new Exception(or.getMessage()));
            }

        } catch (ServiceException e) {
            log.error("caught exception: " + e.getMessage(), e);
            or.setReturnCode(e.getErrorCode());
            or.setMessage(e.getMessage());
            auditHelper.addExceptionToAllAuditEvents(e);

        } catch (Throwable e) {
            log.error("caught Throwable exception: " + e.getMessage(), e);
            e.printStackTrace(System.out);
            System.out.flush();
            or.setReturnCode(1);
            or.setMessage(e.getMessage());
            auditHelper.addExceptionToAllAuditEvents(e);
        }

        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
        return or;
    }

    /**
     * Get content type for resource type.
     *
     * @param outputFormat - resource output format
     * @return content type
     */
    protected String getContentType(String outputFormat) {
        return servicesConfiguration.getExporter(outputFormat.toLowerCase()).getContentType();
    }
    
    protected String getFileName(String outputFormat, String reportUnitUri){
        return servicesConfiguration.getExporter(outputFormat.toLowerCase()).getFileName(reportUnitUri);
    }

    /**
     * Create additional Web Services attachments for the content. At this stage, HTML reports
     * have their images as attachments
     *
     * @param jasperPrint
     * @param format
     * @param format
     * @param exportParameters
     * @throws ServiceException
     */
    private void addAdditionalAttachmentsForReport(JasperPrint jasperPrint, String format, Map exportParameters) throws ServiceException {
        if (log.isDebugEnabled()) {
            log.debug("Format requested: " + format + "  " + Argument.RUN_OUTPUT_FORMAT_HTML);
        }
        if (!format.equals(Argument.RUN_OUTPUT_FORMAT_HTML)) {
            return;
        }

        try {
            Map imagesMap = (Map) exportParameters.get(JRHtmlExporterParameter.IMAGES_MAP);
            if (log.isDebugEnabled()) {
                log.debug("imagesMap : " + Arrays.asList(imagesMap.keySet().toArray()));
            }
            for (Iterator it = imagesMap.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry entry = (Map.Entry) it.next();
                String name = (String) entry.getKey();
                byte[] data = (byte[]) entry.getValue();
                byte imageType = JRTypeSniffer.getImageType(data);
                String mimeType = JRTypeSniffer.getImageMimeType(imageType);
                if (log.isDebugEnabled()) {
                    log.debug("Adding image for HTML: " + name + ", type: " + mimeType);
                }

                ByteArrayDataSource bads = new ByteArrayDataSource(data, mimeType);
                getOutputAttachments().put(name, bads);
            }
        } catch (Throwable e) {
            log.error(e);
            throw new ServiceException(ServiceException.EXPORT_ERROR,
                    messageSource.getMessage("webservices.error.errorAddingImage",
                            new Object[]{e.getMessage()}, LocaleContextHolder.getLocale())
            );
        }
    }


    /**
     * Look for the ReportExporter configured for the named format and export
     * the report.
     *
     * @param reportUnitURI
     * @param jasperPrint
     * @param format
     * @param output
     * @param exportParameters
     * @return
     * @throws Exception
     */
    protected Map exportReport(
            String reportUnitURI,
            JasperPrint jasperPrint,
            String format,
            OutputStream output,
            HashMap exportParameters) throws ServiceException, RemoteException {
        ReportExporter exporter = servicesConfiguration.getExporter(format.toLowerCase());

        if (exporter == null) {
            throw new ServiceException(3, "Export format " + format.toLowerCase() + " not supported or misconfigured");
        }

        try {

            return exporter.exportReport(
                    jasperPrint,
                    output,
                    engine,
                    exportParameters,
                    createExecutionContext(),
                    reportUnitURI);

        }catch (RemoteException e){
            throw e;
        }catch (Exception ex) {
            throw new ServiceException(3, ex.getMessage() + " (while exporting the report)");
        }
    }

    protected ExecutionContext createExecutionContext() {
        ExecutionContextImpl ctx = new ExecutionContextImpl();
        ctx.setLocale(LocaleContextHolder.getLocale());
        ctx.setTimeZone(TimeZone.getDefault());
        return ctx;
    }

    public List<InputControlState> getValuesForInputControls(String reportUnitUri, Set<String> inputControlIds, Map<String, String[]> parameters) throws ResourceNotFoundException {
        List<InputControlState> result = null;
        try {
            result = inputControlsLogicService.getValuesForInputControls(reportUnitUri, inputControlIds, parameters);
        } catch (CascadeResourceNotFoundException e) {
            throw new ResourceNotFoundException("URI:" + e.getResourceUri() + " Type:" + e.getResourceType());
        }
        return result;
    }

    /**
     * @param reportUnitUri   - URI of the report
     * @param inputControlIds - list of requested input control ID. Empty list or null means "ALL"
     * @param rawParameters
     * @return list of input controls for the report
     * @throws RemoteException if any error occurs
     */
    public List<ReportInputControl> getInputControlsForReport(String reportUnitUri, Set<String> inputControlIds, Map<String, String[]> rawParameters) throws ResourceNotFoundException {
        List<ReportInputControl> result = null;
        try {
            result = inputControlsLogicService.getInputControlsStructure(reportUnitUri, inputControlIds);
            if (result != null && !result.isEmpty()) {
                final List<InputControlState> states = inputControlsLogicService.getValuesForInputControls(reportUnitUri, inputControlIds, rawParameters);
                if (states != null && !states.isEmpty()) {
                    Map<String, InputControlState> statesMap = new HashMap<String, InputControlState>();
                    for (InputControlState currentState : states)
                        statesMap.put(currentState.getId(), currentState);
                    for (ReportInputControl currentInputControl : result)
                        currentInputControl.setState(statesMap.get(currentInputControl.getId()));
                }
            }
        } catch (CascadeResourceNotFoundException e) {
            throw new ResourceNotFoundException("URI:" + e.getResourceUri() + " Type:" + e.getResourceType());
        }
        return result;
    }

    /**
     * @param itemName - name of the report output item
     * @param cacheKey - cache key of a generated report
     * @return report output resource
     * @throws RemoteException if any error occurs
     */
    public ReportOutputResource getReportItem(String itemName, Integer cacheKey) throws RemoteException {
        if (getReportOutput(cacheKey).get(itemName) == null)
            throw new ResourceNotFoundException(itemName);
        return getReportOutput(cacheKey).get(itemName);
    }

    /**
     *
     * @param reportUnitURI  - URI of the report to run
     * @param outputFormat   - output format
     * @param ignorePagination
     *@param page           - page number, if null - all pages
     * @param transformerKey - transformer key
     * @param rawParameters  - map with parameters in raw state(raw mean, that all parameters are strings or arrays of strings)
     * @param avoidCache     - if true, then report will be regenerated     
     * @param freshData - if true, ignore existing data snapshots
     * @param saveDataSnapshot - if true, save a data snapshot with the current data
     * @return report output resource
     * @throws RemoteException if any error occurs
     */
    public ReportOutputResource getReportOutputFromRawParameters(String reportUnitURI, String outputFormat, Boolean ignorePagination, Integer page, String transformerKey,
                                                                 Map<String, String[]> rawParameters, Boolean avoidCache,
                                                                 Boolean freshData, Boolean saveDataSnapshot) throws RemoteException {
        Resource report = repository.getResource(createExecutionContext(), reportUnitURI);
        if (report == null)
            throw new ResourceNotFoundException(reportUnitURI);
        // convert parameters from raw strings to objects
        Map<String, Object> convertedParameters = null;
        Map<String, String[]> rawReportParametersForCacheKeySuffix = null;
        try {
            convertedParameters = inputControlsLogicService.getTypedParameters(report.getURIString(), rawParameters);
            rawReportParametersForCacheKeySuffix = filterInputControlsRawValues((InputControlsContainer) report, rawParameters);
        } catch (CascadeResourceNotFoundException e) {
            throw new ResourceNotFoundException("URI:" + e.getResourceUri() + " Type:" + e.getResourceType());
        } catch (InputControlsValidationException e) {
            throw new JSValidationException(e.getErrors());
        }
        
        ReportExecutionOptions reportExecutionOptions = new ReportExecutionOptions();
        if (freshData != null) {
        	reportExecutionOptions.setFreshData(freshData);
        }
        if (saveDataSnapshot != null) {
        	reportExecutionOptions.setSaveDataSnapshot(saveDataSnapshot);
        }
        
        return getReportOutputFromConvertedParameters(report, outputFormat, transformerKey, ignorePagination, page,
                convertedParameters, getCacheKey(reportUnitURI, rawReportParametersForCacheKeySuffix, ignorePagination), avoidCache,
                reportExecutionOptions, rawParameters);
    }

    /**
     * Conversion of parameters from strings to objects.
     *
     * @param report        - the report resource
     * @param rawParameters - input parameters in raw state (strings and string arrays)
     * @return converted parameters
     * @throws RemoteException if any error occurs
     */
    protected Map<String, String[]> filterInputControlsRawValues(InputControlsContainer report, Map<String, String[]> rawParameters) throws RemoteException, CascadeResourceNotFoundException {
        Map<String, String[]> result = null;
        if (report != null && rawParameters != null && !rawParameters.isEmpty()) {
            // read input controls for report
            List<InputControl> inputControls = cachedEngineService.getInputControls(report);
            if (inputControls != null && !inputControls.isEmpty()) {
                // rawReportParametersForCacheKeySuffix is needed to build cache key suffix
                // to avoid multiple runs with the same set of parameters and values.
                result = new HashMap<String, String[]>();
                String[] rawValue = null;
                for (InputControl control : inputControls) {
                    final String controlName = control.getName();
                    rawValue = rawParameters.get(controlName);
                    if (rawValue != null) {
                        result.put(controlName, rawValue);

                    }
                }
            }
        }
        return result;
    }

    /**
     * Builds cache key by report unit URI and set of input parameters
     *
     *
     * @param reportUnitURI - URI of the report
     * @param rawReportParametersForCacheKeySuffix
     *                      - set of input parameters and values
     * @param ignorePagination
     * @return unique ID for report unit URI and set of parameters/values
     */
    protected Integer getCacheKey(String reportUnitURI, Map<String, String[]> rawReportParametersForCacheKeySuffix, Boolean ignorePagination) {
        return cacheKeysMapping.put(reportUnitURI + buildCacheKeySuffix(rawReportParametersForCacheKeySuffix) + ";ignorePagination=" + ignorePagination);
    }

    /**
     * Builds string, which contain all input parameters and values.
     *
     * @param rawReportParametersForCacheKeySuffix
     *         - set of input parameters and values
     * @return string, which contain all input parameters and values.
     */
    protected String buildCacheKeySuffix(Map<String, String[]> rawReportParametersForCacheKeySuffix) {
        StringBuilder suffixBuilder = new StringBuilder();
        if (rawReportParametersForCacheKeySuffix != null && !rawReportParametersForCacheKeySuffix.isEmpty()) {
            List<String> sortedNames = new ArrayList<String>();
            sortedNames.addAll(rawReportParametersForCacheKeySuffix.keySet());
            // sort parameter names to have the same sequence
            Collections.sort(sortedNames);
            for (String currentName : sortedNames) {
                // append parameter name and value
                suffixBuilder.append(currentName).append("=");
                String[] currentValue = rawReportParametersForCacheKeySuffix.get(currentName);
                // multiple values
                String[] currentValues = (String[]) currentValue;
                List<String> sortedValues = Arrays.asList(currentValues);
                Collections.sort(sortedValues);
                suffixBuilder.append("[");
                Boolean isFirst = true;
                for (String currentValueItem : sortedValues) {
                    if (isFirst)
                        isFirst = false;
                    else
                        suffixBuilder.append(",");
                    suffixBuilder.append(currentValueItem);
                }
                suffixBuilder.append("]");
                suffixBuilder.append(";");
            }
        }
        return suffixBuilder.toString();
    }

    /**
     * @param cacheKey - key of a specific report output
     * @return report output container, generates new container if no container yet.
     */
    protected Map<String, ReportOutputResource> getReportOutput(Integer cacheKey) {
        if (reportsOuptutCache.get(cacheKey) == null)
            synchronized (reportsOuptutCache) {
                if (reportsOuptutCache.get(cacheKey) == null)
                    reportsOuptutCache.put(cacheKey, new HashMap<String, ReportOutputResource>());
            }
        return reportsOuptutCache.get(cacheKey);
    }

    /**
     * Run report and export it ot specified output format.
     *
     *
     * @param report              - the report resource from a repository
     * @param outputFormat        - output format
     * @param transformerKey      - transformer key
     * @param ignorePagination    - if true, then pagination is ignored
     * @param page                - page number
     * @param convertedParameters - input parameters prepared for report execution
     * @param cacheKey            - cache key for report output container
     * @param avoidCache          - if true, then report is regenerated     @return report output resource in corresponding output format
     * @param reportExecutionOptions - report execution options 
     * @throws RemoteException if any error occurs
     */
    protected ReportOutputResource getReportOutputFromConvertedParameters(
            Resource report, String outputFormat, String transformerKey,
            Boolean ignorePagination, Integer page, Map<String, Object> convertedParameters, Integer cacheKey, Boolean avoidCache, 
            ReportExecutionOptions reportExecutionOptions, Map<String, String[]> rawParameters) throws RemoteException {
        Boolean nullSafeAvoidCache = avoidCache != null ? avoidCache : false;
        Map<String, ReportOutputResource> reportOutput = getReportOutput(cacheKey);
        if (nullSafeAvoidCache || jasperPrintMap.get(cacheKey) == null)
            synchronized (reportOutput) {
                if (nullSafeAvoidCache || jasperPrintMap.get(cacheKey) == null) {
                    // double checked condition: jasperPrint object isn't generated yet or must be regenerated
                    // clean previous generation output
                    reportOutput.clear();
                    convertedParameters.put(JRParameter.IS_IGNORE_PAGINATION, ignorePagination);
                    // generate and cache new jasperPrint object
                    jasperPrintMap.put(cacheKey, runReport(report, convertedParameters, reportExecutionOptions));
                }
            }
        String resourceKey = getResourceKey(outputFormat, page);
        ReportOutputResource result = reportOutput.get(resourceKey);
        if (result == null) {
            synchronized (reportOutput) {
                result = reportOutput.get(resourceKey);
                if (result == null) {
                    // double checked condition: requested output item isn't generated yet
                    JasperPrint jasperPrint = jasperPrintMap.get(cacheKey);
                    // generated requested item and put it to output cache
                    generateReportOutput(getStrategyForReport(report).getConcreteReportURIString(report),
                            jasperPrint, outputFormat, transformerKey, page,
                            report.getURIString().substring(report.getURIString().lastIndexOf("/") + 1) + "/" + cacheKey + "/items/", reportOutput, rawParameters);
                    result = reportOutput.get(resourceKey);
                }
            }
        }
        if (result == null)
            throw new ResourceNotFoundException(report.getURIString() + "." + outputFormat);
        return result;
    }

    /**
     * Report output resource is placed to output map with key generated from output format and page number.
     *
     * @param outputFormat - output format
     * @param page         - number of the page
     * @return resource key, which includes output format and page number
     */
    protected String getResourceKey(String outputFormat, Integer page) {
        return outputFormat + (page != null ? "_page=" + page : "");
    }

    /**
     * Report output generation or export.
     *
     *
     * @param reportUnitURI   - URI of the report
     * @param jasperPrint     - filled with data jasper print object
     * @param rawOutputFormat - output format in raw format
     * @param transformerKey  - transformer key
     * @param page            - number of the page or all pages if null
     * @param imagesURI       - images URI prefix
     * @param ouputContainer  - output map to place output resource    @throws RemoteException - if any error occurs
     */
    protected void generateReportOutput(String reportUnitURI, JasperPrint jasperPrint, String rawOutputFormat, String transformerKey,
                                        Integer page, String imagesURI, Map<String, ReportOutputResource> ouputContainer, Map<String, String[]> rawParameters) throws RemoteException {
        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");
        try {
            String outputFormat = rawOutputFormat != null ? rawOutputFormat.toUpperCase() : Argument.RUN_OUTPUT_FORMAT_PDF;
            // Export...
            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            if (outputFormat.equals(Argument.RUN_OUTPUT_FORMAT_JRPRINT)) {
                if (log.isDebugEnabled())
                    log.debug("Returning JasperPrint");
                if (transformerKey != null) {
                    if (log.isDebugEnabled())
                        log.debug("Transforming JasperPrint generic element for key " + transformerKey);
                    GenericElementReportTransformer.transformGenericElements(jasperPrint, transformerKey);
                }
                JRSaver.saveObject(jasperPrint, bos);
                ouputContainer.put(KEY_JASPER_PRINT_RESOURCE, new ReportOutputResource("application/octet-stream", bos.toByteArray()));
            } else {
                HashMap exportParameters = new HashMap(rawParameters);
                if (page != null)
                    exportParameters.put(Argument.RUN_OUTPUT_PAGE, page.toString());
                if (imagesURI != null)
                    exportParameters.put(Argument.RUN_OUTPUT_IMAGES_URI, imagesURI);
                Map exporterParams;
                try {
                    exporterParams = exportReport(reportUnitURI, jasperPrint, outputFormat, bos, exportParameters);
                    if (log.isDebugEnabled())
                        log.debug("Exporter params: " + Arrays.asList(exporterParams.keySet().toArray()));
                } catch (RemoteException e){
                    throw e;
                }catch (Exception e) {
                    log.error("Error exporting report", e);
                    throw new RemoteException(
                            new ErrorDescriptor.Builder()
                                    .setErrorCode("webservices.error.errorExportingReportUnit").setParameters(e.getMessage())
                                    .getErrorDescriptor(), e);
                } finally {
                    if (bos != null) {
                        try {
                            bos.close();
                        } catch (IOException ex) {
                            log.error("caught exception: " + ex.getMessage(), ex);
                        }
                    }
                }
                ouputContainer.put(getResourceKey(rawOutputFormat, page), new ReportOutputResource(getContentType(outputFormat), bos.toByteArray(), getFileName(outputFormat, reportUnitURI)));
                if (Argument.RUN_OUTPUT_FORMAT_HTML.equals(outputFormat))
                    putImages(exporterParams, ouputContainer);
            }
        } catch (RemoteException e) {
            auditHelper.addExceptionToAllAuditEvents(e);
            throw e;
        } catch (ServiceException e) {
            log.error("caught exception: " + e.getMessage(), e);
            auditHelper.addExceptionToAllAuditEvents(e);
        } catch (Throwable e) {
            log.error("caught Throwable exception: " + e.getMessage(), e);
            auditHelper.addExceptionToAllAuditEvents(e);
        }
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
    }

    /**
     * Running of the report happens here
     *
     * @param reportResource - the report resource from a repository to run
     * @param parameters     - input parameters
     * @param reportExecutionOptions - report execution options
     * @return jasperptint object filled with data
     * @throws RemoteException if any error occurs
     */
    protected JasperPrint runReport(Resource reportResource, Map<String, Object> parameters,
    		ReportExecutionOptions reportExecutionOptions) throws RemoteException {
        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");
        JasperPrint result = null;
        RunReportStrategy strategy = getStrategyForReport(reportResource);
        if (strategy == null) {
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorExecutingReportUnit").setParameters(reportResource.getURI()).getErrorDescriptor());
        }
        // run the report
        ReportUnitResult reportUnitResult = strategy.runReport(reportResource, parameters, engine, reportExecutionOptions);

        if (reportUnitResult == null) {
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorExecutingReportUnit").setParameters(reportResource.getURI()).getErrorDescriptor());
        } else {
            result = reportUnitResult.getJasperPrint();
        }

        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
        return result;
    }

    /**
     * Place images to output container.
     *
     * @param exportParameters - export result, contains images
     * @param outputContainer  - output container to fill with images
     * @throws RemoteException if any error occurs
     */
    protected void putImages(Map exportParameters, Map<String, ReportOutputResource> outputContainer) throws RemoteException {
        try {
            Map imagesMap = (Map) exportParameters.get(JRHtmlExporterParameter.IMAGES_MAP);
            if (imagesMap != null && !imagesMap.isEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("imagesMap : " + Arrays.asList(imagesMap.keySet().toArray()));
                }
                for (Iterator it = imagesMap.entrySet().iterator(); it.hasNext(); ) {
                    Map.Entry entry = (Map.Entry) it.next();
                    String name = (String) entry.getKey();
                    byte[] data = (byte[]) entry.getValue();
                    if (log.isDebugEnabled()) {
                        log.debug("Adding image for HTML: " + name);
                    }
                    outputContainer.put(name, new ReportOutputResource(JRTypeSniffer.getImageTypeValue(data).getMimeType(), data));
                }
            }
        } catch (Throwable e) {
            log.error(e);
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorAddingImage").setParameters(e.getMessage()).getErrorDescriptor(), e);
        }
    }

    /**
     * @param searchCriteria - search criteria
     * @return set of currently running report's information
     */
    public Set<ReportExecutionStatusInformation> getCurrentlyRunningReports(SchedulerReportExecutionStatusSearchCriteria searchCriteria) {
        Set<ReportExecutionStatusInformation> result = null;
        List<ReportExecutionStatusInformation> reportExecutionStatusList = searchCriteria != null ?
                engine.getSchedulerReportExecutionStatusList(searchCriteria) : engine.getReportExecutionStatusList();
        if (reportExecutionStatusList != null && !reportExecutionStatusList.isEmpty()) {
            result = new HashSet<ReportExecutionStatusInformation>();
            result.addAll(reportExecutionStatusList);
        }
        return result;
    }

    public Boolean cancelReportExecution(String requestId) throws RemoteException {
        return engine.cancelExecution(requestId);
    }

    /**
     * @param report - the report resource from a repository
     * @return strategy for running of a report of concrete type
     */
    protected RunReportStrategy getStrategyForReport(Resource report) {
        return report instanceof ReportUnit ? new RunReportUnitStrategy() : null;
    }

    /**
     * Generic run report strategy. Contains generic run report functionality.
     *
     * @param <ReportType> - concrete type of report
     */
    protected abstract class GenericRunReportStrategy<ReportType extends Resource> implements RunReportStrategy {
        /**
         * Runs report of concrete type
         *
         * @param reportResource - the report resource from a repository
         * @param parameters     - input parameters
         * @param engine         - engine service
         * @return result of report execution
         */
        // report type should correspond to concrete type of strategy. getStrategyForReport() assure that.
        // So, unchecked cast is safe
        @SuppressWarnings("unchecked")
        public ReportUnitResult runReport(Resource reportResource, Map<String, Object> parameters, EngineService engine,
        		ReportExecutionOptions options) {
            ReportType report = (ReportType) reportResource;
			Map<String, Object> convertedParameters = parameters != null ? RepositoryHelper.convertParameterValues(getConcreteReportURI(report), parameters, engine) : new HashMap<String, Object>();
            ReportUnitRequest request = getReportUnitRequest(report, convertedParameters, options);
			ExecutionContext executionContext = createExecutionContext();
			ReportUnitResult result = (ReportUnitResult) engine.execute(executionContext, request);
			persistDataSnapshot(executionContext, options, reportResource, request.getReportContext());
			return result;
        }

		protected void persistDataSnapshot(ExecutionContext executionContext, ReportExecutionOptions options, 
				Resource reportResource, ReportContext reportContext) {
			SnapshotSaveStatus snapshotSaveStatus = dataCacheProvider.getSnapshotSaveStatus(reportContext);
			switch (snapshotSaveStatus) {
			case NEW:
				// automatic save
				if (log.isDebugEnabled()) {
					log.debug("saving initial data snapshot for " + reportResource.getURIString());
				}
				
				saveAutoDataSnapshot(executionContext, reportResource, reportContext);
				break;
			case UPDATED:
				if (options.isSaveDataSnapshot()) {
					// requested save
					if (log.isDebugEnabled()) {
						log.debug("saving updated data snapshot for " + reportResource.getURIString());
					}
					
					saveDataSnapshot(executionContext, reportResource, reportContext);
				}
				break;
			case NO_CHANGE:
			default:
				//NOP
				break;
			};
		}
		
		protected void saveAutoDataSnapshot(ExecutionContext executionContext, Resource reportResource, 
				ReportContext reportContext) {
			ReportUnit reportUnit = getReportUnit(reportResource);
			try {
				dataSnapshotService.saveAutoReportDataSnapshot(executionContext, reportContext, reportUnit);
			} catch (Exception e) {
				// catching any exceptions for automatic and requested save
				log.error("Error while saving data snapshot for " + reportUnit.getURIString(), e);
			}
		}
		
		protected void saveDataSnapshot(ExecutionContext executionContext, Resource reportResource, 
				ReportContext reportContext) {
			ReportUnit reportUnit = getReportUnit(reportResource);
			try {
				dataSnapshotService.saveReportDataSnapshot(executionContext, reportContext, reportUnit);
			} catch (Exception e) {
				// catching any exceptions for automatic and requested save
				log.error("Error while saving data snapshot for " + reportUnit.getURIString(), e);
			}
		}

        protected ReportUnitRequest getReportUnitRequest(ReportType reportResource, Map<String, Object> parameters,
        		ReportExecutionOptions options) {
        	Map<String, Object> requestParams = new HashMap<String, Object>();
        	requestParams.putAll(parameters);
        	
            // we need a report context for snapshots
            SimpleReportContext reportContext = new SimpleReportContext();
    		requestParams.put(JRParameter.REPORT_CONTEXT, reportContext);
            
            ReportUnitRequest request = new ReportUnitRequest(getConcreteReportURI(reportResource), requestParams);
            request.setReportContext(reportContext);
            
            // recording is enabled for first-time saves
        	request.setRecordDataSnapshot(dataSnapshotService.isSnapshotPersistenceEnabled());
        	
        	// fresh data if requested or saving
        	request.setUseDataSnapshot(!(options.isFreshData() || options.isSaveDataSnapshot()));
            
			return request;
        }
    }


    /**
     * Strategy to run report of ReportUnit type
     */
    protected class RunReportUnitStrategy extends GenericRunReportStrategy<ReportUnit> {

        public String getConcreteReportURI(Resource reportResource) {
            return reportResource.getURI();
        }

        public String getConcreteReportURIString(Resource reportResource) {
            return reportResource.getURIString();
        }

        // report type is ReportUnit for this strategy. Therefore cast is safe.
        @SuppressWarnings("unchecked")
        public ReportUnit getReportUnit(Resource report) {
            return (ReportUnit) report;
        }
    }

    protected interface RunReportStrategy {
        ReportUnitResult runReport(Resource reportResource, Map<String, Object> parameters, EngineService engine,
        		ReportExecutionOptions options);

        ReportUnit getReportUnit(Resource reportResource);

        String getConcreteReportURI(Resource reportResource);

        String getConcreteReportURIString(Resource reportResource);
    }

    protected static class ReportExecutionOptions {

    	private boolean freshData;
    	private boolean saveDataSnapshot;
    	
		public boolean isFreshData() {
			return freshData;
		}
		
		public void setFreshData(boolean freshData) {
			this.freshData = freshData;
		}
		
		public boolean isSaveDataSnapshot() {
			return saveDataSnapshot;
		}
		
		public void setSaveDataSnapshot(boolean saveDataSnapshot) {
			this.saveDataSnapshot = saveDataSnapshot;
		}
    	
    }

}
