/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.common.service.ResourceFactory;
import com.jaspersoft.jasperserver.api.metadata.security.JasperServerAclEntry;
import com.jaspersoft.jasperserver.api.metadata.user.domain.ObjectPermission;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.ObjectPermissionService;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.services.PermissionsService;
import com.jaspersoft.jasperserver.remote.services.ResourcesManagementRemoteService;
import com.jaspersoft.jasperserver.remote.utils.AuditHelper;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.security.Authentication;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @author Volodya Sabadosh (vsabadosh@jaspersoft.com)
 * @version $Id: PermissionsServiceImpl.java 23756 2012-05-15 14:18:40Z ykovalchyk $
 */
@Component("permissionsService")
public class PermissionsServiceImpl implements PermissionsService {
    @javax.annotation.Resource(name = "concreteObjectPermissionsService")
    protected ObjectPermissionService objectPermissionService;
    @javax.annotation.Resource(name = "concreteRepository")
    protected RepositoryService repositoryService;
    @javax.annotation.Resource
    private AuditHelper auditHelper;
    @javax.annotation.Resource
    private ResourcesManagementRemoteService resourcesManagementRemoteService;
    @javax.annotation.Resource(name = "mappingResourceFactory")
    private ResourceFactory resourceFactory;
    @javax.annotation.Resource(name = "concreteUserAuthorityService")
    protected UserAuthorityService userAuthorityService;


    public void setResourcesManagementRemoteService(ResourcesManagementRemoteService resourcesManagementRemoteService) {
        this.resourcesManagementRemoteService = resourcesManagementRemoteService;
    }

    public void setAuditHelper(AuditHelper auditHelper) {
        this.auditHelper = auditHelper;
    }

    public List<ObjectPermission> getPermissions(String resourceURI) throws RemoteException {
        Resource resource = resourcesManagementRemoteService.locateResource(resourceURI);
        return objectPermissionService.getObjectPermissionsForObject(makeExecutionContext(), resource);
    }

    public List<ObjectPermission> getPermissionsForObject(String targetURI) throws RemoteException {
        if (!objectPermissionService.isObjectAdministrable(makeExecutionContext(), targetURI)) {
            throw new RemoteException("Access is denied");
        }
        Resource res = repositoryService.getResource(makeExecutionContext(), targetURI);
        if (res == null) {
            res = repositoryService.getFolder(makeExecutionContext(), targetURI);

            if (res == null) {
                throw new RemoteException("There is no resource or folder for target URI \"" + targetURI + "\"");
            }
        }

        return objectPermissionService.getObjectPermissionsForObject(makeExecutionContext(), res);
    }

    public ObjectPermission putPermission(ObjectPermission objectPermission) throws RemoteException {
        changePermissionConsistencyCheck(objectPermission);

        ObjectPermission existingObjectPermission =
                objectPermissionService.getObjectPermission(makeExecutionContext(), objectPermission);
        String auditEventType;

        if (existingObjectPermission == null) {
            auditEventType = "createPermission";
        } else {
            auditEventType = "updatePermission";
        }

        auditHelper.createAuditEvent(auditEventType);
        objectPermissionService.putObjectPermission(makeExecutionContext(), objectPermission);
        auditHelper.closeAuditEvent(auditEventType);

        return objectPermissionService.getObjectPermission(makeExecutionContext(), objectPermission);
    }

    public void deletePermission(ObjectPermission objectPermission) throws RemoteException {
        changePermissionConsistencyCheck(objectPermission);

        auditHelper.createAuditEvent("deletePermission");
        objectPermissionService.deleteObjectPermission(makeExecutionContext(), objectPermission);
        auditHelper.closeAuditEvent("deletePermission");
    }

    public ObjectPermission newObjectPermission() {
        return objectPermissionService.newObjectPermission(makeExecutionContext());
    }

    /**
     * Returns permission mask for resource with <strong>targetURI</strong></> for current user.
     *
     * @param targetURI resource URI.
     * @return
     * @throws RemoteException
     */
    public int getAppliedPermissionMaskForObjectAndCurrentUser(String targetURI) throws RemoteException {
        Resource resource = repositoryService.getResource(makeExecutionContext(), targetURI);
        if (resource == null) {
            resource = repositoryService.getFolder(makeExecutionContext(), targetURI);

            if (resource == null) {
                throw new RemoteException("There is no resource or folder for target URI \"" + targetURI + "\"");
            }
        }

        Set<Integer> allUserPermissions = null;

        List<Object> currentUserRecipients = getCurrentUserRecipients();
        if (currentUserRecipients != null && currentUserRecipients.size() > 0) {
            allUserPermissions = new HashSet<Integer>();
            for (Object recipient : getCurrentUserRecipients()) {
                List<ObjectPermission> permissions = objectPermissionService.getObjectPermissionsForObjectAndRecipient(
                        makeExecutionContext(), resource, recipient);
                if (permissions != null && !permissions.isEmpty()) {
                    ObjectPermission permissionObject = permissions.get(0);
                    allUserPermissions.add(permissionObject.getPermissionMask());
                } else {
                    allUserPermissions.add(objectPermissionService.
                            getInheritedObjectPermissionMask(makeExecutionContext(), resource, recipient));
                }
            }
        }
        if (allUserPermissions != null && allUserPermissions.size() > 0) {
            JasperServerAclEntry jasperServerAclEntry = new JasperServerAclEntry();
            // returns permissions mask
            return jasperServerAclEntry.printPermissionsOverlappingBlock(allUserPermissions);
        } else {
            // returns no permissions.
            return 0;
        }
    }
    
    protected List<Object> getCurrentUserRecipients() {
        Authentication authenticationToken = SecurityContextHolder.getContext().getAuthentication();
        List<Object> recipients = new ArrayList<Object>();
        if (authenticationToken.getPrincipal() instanceof User) {
            User user = (User) authenticationToken.getPrincipal();
            recipients.add(user);
            recipients.addAll(user.getRoles());

            return recipients;
        } else {
            return null;
        }
    }

    protected void changePermissionConsistencyCheck(ObjectPermission objectPermission) throws RemoteException {
        if (StringUtils.isBlank(objectPermission.getURI())) {
            throw new RemoteException("URI is blank");
        }

        // Permission recepient check
        if (objectPermission.getPermissionRecipient() == null) {
            throw new RemoteException("Permission recipient is null");
        }

        if (isPermissionForRootRessource(objectPermission)) {
            throw new RemoteException("Setting/changing/deleting permissions for root URI is forbidden");
        }
    }

    public void setObjectPermissionService(ObjectPermissionService objectPermissionService) {
        this.objectPermissionService = objectPermissionService;
    }

    public void setRepositoryService(RepositoryService repositoryService) {
        this.repositoryService = repositoryService;
    }

    public void setUserAuthorityService(UserAuthorityService userAuthorityService) {
        this.userAuthorityService = userAuthorityService;
    }

    protected boolean isPermissionForRootRessource(ObjectPermission objectPermission) {
        return ("repo:/".equals(objectPermission.getURI()) || "/".equals(objectPermission.getURI()));
    }

    protected ExecutionContext makeExecutionContext() {
        return ExecutionContextImpl.getRuntimeExecutionContext();
    }

    protected Locale getLocale() {
        return LocaleContextHolder.getLocale();
    }

}
