/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote.handlers;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResourceData;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.api.search.SearchCriteria;
import com.jaspersoft.jasperserver.api.search.SearchFilter;
import com.jaspersoft.jasperserver.remote.ResourceDataSource;
import com.jaspersoft.jasperserver.remote.ServiceException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;


/**
 * @author gtoffoli
 * @version $Id: FileResourceHandler.java 23457 2012-04-26 07:30:44Z ykovalchyk $
 */
@Service("fileHandler")
public class FileResourceHandler extends RepositoryResourceHandler {
    public static final String MAIN_ATTACHMENT_ID = "attachment";

    private static final Log log = LogFactory.getLog(FileResourceHandler.class);

    public Class getResourceType() {
        return FileResource.class;
    }

    @Override
    public SearchFilter getSearchFilter(String uri, String queryString, final String wsType, boolean recursive, int maxItems, int startIndex) {
        return new SearchFilter() {
            @Override
            public void applyRestrictions(String type, ExecutionContext context, SearchCriteria criteria) {
                criteria.add(Restrictions.eq("fileType", getFileType(wsType)));
            }
        };
    }

    protected void doGet(Resource resource, ResourceDescriptor descriptor, Map options) throws ServiceException {
        FileResource fileResource = (FileResource) resource;
        descriptor.setWsType(fileResource.getFileType());
        descriptor.setHasData(fileResource.hasData());
        descriptor.setIsReference(fileResource.isReference());
        if (descriptor.getIsReference()) {
            descriptor.setReferenceUri(fileResource.getReferenceURI());
        }

        String resourceType = fileResource.getFileType();

        String contentType = "application/octet-stream";

        if (descriptor.getIsReference()) {
            descriptor.setWsType(ResourceDescriptor.TYPE_REFERENCE);
        } else if (resourceType == null)
            descriptor.setWsType(ResourceDescriptor.TYPE_UNKNOW);
        else if (resourceType.equals(FileResource.TYPE_JRXML)) {
            descriptor.setWsType(ResourceDescriptor.TYPE_JRXML);
            contentType = "text/xml";
        } else if (resourceType.equals(FileResource.TYPE_IMAGE)) {
            descriptor.setWsType(ResourceDescriptor.TYPE_IMAGE);
        } else if (resourceType.equals(FileResource.TYPE_FONT))
            descriptor.setWsType(ResourceDescriptor.TYPE_FONT);
        else if (resourceType.equals(FileResource.TYPE_JAR))
            descriptor.setWsType(ResourceDescriptor.TYPE_CLASS_JAR);
        else if (resourceType.equals(FileResource.TYPE_RESOURCE_BUNDLE)) {
            descriptor.setWsType(ResourceDescriptor.TYPE_RESOURCE_BUNDLE);
            contentType = "text/plain";
        } else if (resourceType.equals(FileResource.TYPE_STYLE_TEMPLATE)) {
            descriptor.setWsType(ResourceDescriptor.TYPE_STYLE_TEMPLATE);
            contentType = "text/xml";
        } else if (resourceType.equals(FileResource.TYPE_XML)) {
            descriptor.setWsType(ResourceDescriptor.TYPE_XML_FILE);
            contentType = "text/xml";
        }

        descriptor.setHasData(false);
        if (options == null || !options.containsKey(Argument.NO_RESOURCE_DATA_ATTACHMENT)) {
            //Retrieve resource data...
            descriptor.setHasData(true);
            FileResourceData fileData = getRepository().getResourceData(null, descriptor.getUriString());
            ResourceDataSource attachmentSource = new ResourceDataSource(descriptor.getName(), fileData);
            attachmentSource.setContentType(contentType);
            getRunReportService().getOutputAttachments().put(MAIN_ATTACHMENT_ID, attachmentSource);
            if (log.isWarnEnabled()) {
                log.warn("Placed attachment for " + descriptor.getName() + " in the output attachments list");
            }
            descriptor.setResourceProperty("PROP_ATTACHMENT_ID", MAIN_ATTACHMENT_ID);
        }

    }

    protected Resource createResource(ResourceDescriptor descriptor) throws ServiceException {
        try {

            FileResource resource = (FileResource) super.createResource(descriptor);

            resource.setFileType(descriptor.getWsType());
            if (descriptor.getHasData() && getRunReportService().getInputAttachments().containsKey(descriptor.getUriString())) {
                InputStream ds = getRunReportService().getInputAttachments().get(descriptor.getUriString()).getInputStream();
                resource.setData(IOUtils.toByteArray(ds));
            } else {
                throw new IllegalStateException("could not find the binary content for resource: " + descriptor.getUriString());
            }

            updateResource(resource, descriptor, null);
            return resource;

        } catch (ClassCastException e) {
            throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, e.getLocalizedMessage());
        } catch (IOException e) {
            throw new ServiceException(ServiceException.INTERNAL_SERVER_ERROR, e.getLocalizedMessage());
        }
    }

    @Override
    protected void updateResource(Resource resource, ResourceDescriptor descriptor, Map options) {
        try {
            FileResource fResource;
            if (!(resource instanceof FileResource)) {
                throw new IllegalStateException("resource:" + resource.getName() + " is not a File Resource");
            } else {
                fResource = (FileResource) resource;
            }

            super.updateResource(fResource, descriptor, options);

            fResource.setLabel(descriptor.getLabel());
            fResource.setDescription(descriptor.getDescription());
            fResource.setFileType(descriptor.getWsType());
            InputStream ds = null;

            if (getRunReportService().getInputAttachments().containsKey(descriptor.getUriString())) {
                ds = getRunReportService().getInputAttachments().get(descriptor.getUriString()).getInputStream();
                fResource.setData(IOUtils.toByteArray(ds));
            }
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    protected String getFileType(String wsType) {
        String type = wsType;
        if (wsType.equals(ResourceDescriptor.TYPE_IMAGE))
            type = FileResource.TYPE_IMAGE;
        else if (wsType.equals(ResourceDescriptor.TYPE_FONT))
            type = FileResource.TYPE_FONT;
        else if (wsType.equals(ResourceDescriptor.TYPE_CLASS_JAR))
            type = FileResource.TYPE_JAR;
        else if (wsType.equals(ResourceDescriptor.TYPE_JRXML))
            type = FileResource.TYPE_JRXML;
        else if (wsType.equals(ResourceDescriptor.TYPE_RESOURCE_BUNDLE))
            type = FileResource.TYPE_RESOURCE_BUNDLE;
        else if (wsType.equals(ResourceDescriptor.TYPE_STYLE_TEMPLATE))
            type = FileResource.TYPE_STYLE_TEMPLATE;
        else if (wsType.equals(ResourceDescriptor.TYPE_XML_FILE))
            type = FileResource.TYPE_XML;
        return type;
    }
}
