/* Copyright (C) 2000-2008 by George Williams */
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.

 * The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fontforgevw.h"
#include "ustring.h"
#include <utype.h>

int recognizePUA = false;
NameList *force_names_when_opening=NULL;
NameList *force_names_when_saving=NULL;

static struct psaltnames {
    char *name;
    int unicode;
    int provenance;		/* 1=> Adobe PUA, 2=>AMS PUA, 3=>TeX */
} psaltnames[5488];  /* Explicit dimension specification or initializer 
                        required for an auto or static array. */

static NameList agl_sans, agl, adobepua, greeksc, tex, ams;
NameList *namelist_for_new_fonts = &agl;

	/* Adobe's standard names are wrong for: */
	/* 0x0162 is named Tcommaaccent, 0x21A should be */
	/* 0x0163 is named tcommaaccent, 0x21B should be */
/* The following have been corrected (at least by intent) or removed */
	/* 0xf6be is named dotlessj, 0x237 should be */
	/* 0x2206 is named Delta, 0x394 should be */
	/* 0x2126 is named Omega, 0x3A9 should be */
	/* 0x00b5 is named mu, 0x3BC should be */

static int psnamesinited=false;
#define HASH_SIZE	257
struct psbucket { const char *name; int uni; struct psbucket *prev; } *psbuckets[HASH_SIZE];

static int hashname(const char *_name) {
    const unsigned char *name = (const unsigned char *) _name;
    int hash=0;

    while ( *name ) {
	if ( *name<=' ' )
    break;
	hash = (hash<<3)|((hash>>29)&0x7);
	hash ^= *name++-(' '+1);
    }
    hash ^= (hash>>16);
    hash &= 0xffff;
return( hash%HASH_SIZE );
}

static void psaddbucket(const char *name, int uni) {
    int hash = hashname(name);
    struct psbucket *buck = gcalloc(1,sizeof(struct psbucket));

    buck->name = name;
    buck->uni = uni;
    buck->prev = psbuckets[hash];
    psbuckets[hash] = buck;
}

static void NameListHash(NameList *nl) {
    int i,j,k;
    for ( i=0; i<17; ++i ) if ( nl->unicode[i]!=NULL ) {
	for ( j=0; j<256; ++j ) if ( nl->unicode[i][j]!=NULL ) {
	    for ( k=0; k<256; ++k ) if ( nl->unicode[i][j][k]!=NULL )
		psaddbucket(nl->unicode[i][j][k],(i<<16)|(j<<8)|k );
	}
    }
}

static void psinitnames(void) {
    int i;
    NameList *nl;

    agl.next = &agl_sans;
    agl_sans.next = &adobepua;
    adobepua.next = &greeksc;
    greeksc.next = &tex;
    tex.next = &ams;

    for ( i=0; psaltnames[i].name!=NULL ; ++i )
	psaddbucket(psaltnames[i].name,psaltnames[i].unicode);
    for ( nl=&agl; nl!=NULL; nl=nl->next )
	NameListHash(nl);
    psnamesinited = true;
}

static void psreinitnames(void) {
    /* If we reread a (loaded) namelist file, then we must remove the old defn*/
    /*  which means we must remove all the old hash entries before we can put */
    /*  in the new ones */
    int i;
    struct psbucket *cur, *prev;
    NameList *nl;

    for ( i=0; i<HASH_SIZE; ++i ) {
	for ( cur = psbuckets[i]; cur!=NULL; cur=prev ) {
	    prev = cur->prev;
	    chunkfree(cur,sizeof(struct psbucket));
	}
	psbuckets[i] = NULL;
    }

    for ( i=0; psaltnames[i].name!=NULL ; ++i )
	psaddbucket(psaltnames[i].name,psaltnames[i].unicode);
    for ( nl=&agl; nl!=NULL; nl=nl->next )
	NameListHash(nl);
}

int UniFromName(const char *name,enum uni_interp interp,Encoding *encname) {
    int i = -1;
    char *end;
    struct psbucket *buck;

    if ( strncmp(name,"uni",3)==0 ) {
	i = strtol(name+3,&end,16);
	if ( *end || end-name!=7 )	/* uniXXXXXXXX means a ligature of uniXXXX and uniXXXX */
	    i = -1;
    } else if ( (name[0]=='U' || name[0]=='u') && name[1]=='+' &&
	    (strlen(name)==6 || strlen(name)==7)) {
	/* Unifont uses this convention */
	i = strtol(name+2,&end,16);
	if ( *end )
	    i = -1;
    } else if ( name[0]=='u' && strlen(name)>=5 ) {
	i = strtol(name+1,&end,16);
	if ( *end )
	    i = -1;
	else if ( encname!=NULL && !encname->is_unicodefull &&
		(interp==ui_ams || interp==ui_trad_chinese)) {
	    int j;
	    extern const int cns14pua[], amspua[];
	    const int *pua = interp==ui_ams ? amspua : cns14pua;
	    for ( j=0xf8ff-0xe000; j>=0; --j )
		if ( pua[j]==i ) {
		    i = j+0xe000;
	    break;
		}
	}
    } else if ( name[0]!='\0' && name[1]=='\0' )
	i = ((unsigned char *) name)[0];
    if ( i==-1 ) {
	if ( !psnamesinited )
	    psinitnames();
	for ( buck = psbuckets[hashname(name)]; buck!=NULL; buck=buck->prev )
	    if ( strcmp(buck->name,name)==0 )
	break;
	if ( buck!=NULL )
	    i = buck->uni;
    }
    if ( !recognizePUA && i>=0xe000 && i<=0xf8ff )
	i = -1;
return( i );
}

const char *StdGlyphName(char *buffer, int uni,enum uni_interp interp,NameList *for_this_font) {
    const char *name = NULL;
    NameList *nl;
    int up, ub, uc;

    if ( for_this_font==NULL )
	for_this_font = namelist_for_new_fonts;
    else if ( for_this_font==(NameList *) -1 )
	for_this_font = &agl;
    if ( (uni>=0 && uni<' ') ||
	    (uni>=0x7f && uni<0xa0) )
	/* standard controls */;
    else if ( uni!=-1  ) {
	if ( uni>=0xe000 && uni<=0xf8ff &&
		(interp==ui_trad_chinese || for_this_font==&ams)) {
	    extern const int cns14pua[], amspua[];
	    const int *pua = interp==ui_trad_chinese ? cns14pua : amspua;
	    if ( pua[uni-0xe000]!=0 )
		uni = pua[uni-0xe000];
	}
	up = uni>>16;
	ub = (uni&0xff00)>>8;
	uc = (uni&0xff);
	if ( up<17 )
	    for ( nl=for_this_font; nl!=NULL; nl=nl->basedon ) {
		if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL &&
			(name = nl->unicode[up][ub][uc])!=NULL )
	    break;
	}
    }
    if ( name==NULL ) {
	if ( uni>=0x10000 )
	    sprintf( buffer, "u%04X", uni);
	else
	    sprintf( buffer, "uni%04X", uni);
	name = buffer;
    }
return( name );
}

#define RefMax	40

static int transcmp(RefChar *r1, RefChar *r2) {
    double d1, d2;

    if ( r1->transform[4]<r2->transform[4] )
return( -1 );
    else if ( r1->transform[4]>r2->transform[4] )
return(  1 );
    if ( (d1 = r1->transform[5])<0 ) d1 = -d1;
    if ( (d2 = r2->transform[5])<0 ) d2 = -d2;
    if ( d1<d2 )
return( -1 );
    else if ( d1==d2 )
return( 0 );
    else
return( 1 );
}

static int FindAllRefs(SplineChar *sc,SplineChar *rsc[RefMax], int *au) {
    RefChar *refs[RefMax], *alp[RefMax], *out[RefMax];
    RefChar *ref;
    int layer, last, rcnt, acnt, ocnt, i,j;
    int alluni;
    /* We also order the reference. The order stored in the splinechar doesn't*/
    /*  mean anything, so try and guess what is intended semantically. */

    if ( sc==NULL )
return( 0 );
    last = ly_fore;
    if ( sc->parent->multilayer )
	last = sc->layer_cnt-1;
    for ( layer=ly_fore; layer<=last; ++layer )
	if ( sc->layers[layer].splines!=NULL || sc->layers[layer].images!=NULL )
return( 0 );
    rcnt = 0;
    for ( layer=ly_fore; layer<=last; ++layer ) {
	for ( ref = sc->layers[layer].refs; ref!=NULL; ref = ref->next ) {
	    if ( rcnt>=RefMax )
return( 0 );
	    refs[rcnt++] = ref;
	}
    }
    alluni = true;
    for ( i=0; i<rcnt; ++i ) {
	if ( refs[i]->sc->unicodeenc==-1 ) {
	    alluni = false;
    break;
	}
    }
    if ( !alluni ) {
	/* If not all unicode we can't make any guesses about meaning, so */
	/*  order by transformation */
	for ( i=0; i<rcnt; ++i ) for ( j=i+1; j<rcnt; ++j ) {
	    if ( transcmp(refs[i],refs[j])>0 ) {
		ref = refs[i];
		refs[i] = refs[j];
		refs[j] = ref;
	    }
	}
    } else {
	acnt = 0;
	for ( i=0; i<rcnt; ++i ) {
	    if ( isalpha(refs[i]->sc->unicodeenc )) {
		alp[acnt++] = refs[i];
		--rcnt;
		for ( j=i; j<rcnt; ++j )
		    refs[j] = refs[j+1];
		--i;
	    }
	}
	for ( i=0; i<acnt; ++i ) for ( j=i+1; j<acnt; ++j ) {
	    if ( transcmp(alp[i],alp[j])>0 ) {
		ref = alp[i];
		alp[i] = alp[j];
		alp[j] = ref;
	    }
	}
	for ( i=0; i<rcnt; ++i ) for ( j=i+1; j<rcnt; ++j ) {
	    if ( transcmp(refs[i],refs[j])>0 ) {
		ref = refs[i];
		refs[i] = refs[j];
		refs[j] = ref;
	    }
	}
	if ( acnt!=0 ) {
	    int a=0, r=0;
	    real cutpoint;
	    ocnt = 0;
	    out[ocnt++] = alp[a++];
	    forever {
		if ( a<acnt ) cutpoint = (alp[a]->transform[4]+3*alp[a-1]->transform[4])/4;
		else		cutpoint = 1e30;
		while ( r<rcnt && refs[r]->transform[4]<cutpoint )
		    out[ocnt++] = refs[r++];
		if ( a>=acnt )
	    break;
		out[ocnt++] = alp[a++];
	    }
	    memcpy(refs,out,ocnt*sizeof(RefChar *));
	    rcnt = ocnt;
	}
    }
    for ( i=0; i<rcnt; ++i )
	rsc[i] = refs[i]->sc;
    /* alluni now means can be written as uniXXXX.XXXX.XXXX... */
    for ( i=0; i<rcnt; ++i ) {
	if ( refs[i]->sc->unicodeenc>0x10000 ) {
	    alluni = false;
    break;
	}
    }
    *au = alluni;
return( rcnt );
}

/* Return a list of all alternate or standard glyph names for this encoding */
char **AllGlyphNames(int uni, NameList *for_this_font, SplineChar *sc) {
    int cnt, k, j, i, len;
    NameList *nl, *nl2, *nl3;
    char **names = NULL;
    const char *name;
    int up, ub, uc;
    char buffer[40], *pt;
    SplineChar *refs[RefMax];
    int rcnt, alluni = 0;

    rcnt = FindAllRefs(sc,refs,&alluni);

    up = uni>>16;
    ub = (uni&0xff00)>>8;
    uc = (uni&0xff);

    for ( k=0; k<2; ++k ) {
	cnt = 0;
	/* try the default namelist first to put that at the head of the list */
	name = NULL;
	nl = nl3 = NULL;
	if ( uni>=0 && up<17 ) {
	    if ( for_this_font!=NULL ) {
		for ( nl3=for_this_font; nl3!=NULL; nl3=nl3->basedon ) {
		    if ( nl3->unicode[up]!=NULL && nl3->unicode[up][ub]!=NULL &&
			    (name = nl3->unicode[up][ub][uc])!=NULL )
		break;
		}
		if ( name!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    if ( for_this_font!=namelist_for_new_fonts ) {
		for ( nl=namelist_for_new_fonts; nl!=NULL; nl=nl->basedon ) if ( nl!=nl3 ) {
		    if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL &&
			    (name = nl->unicode[up][ub][uc])!=NULL )
		break;
		}
		if ( name!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    for ( nl2 = &agl; nl2!=NULL; nl2=nl2->next ) if ( nl2!=nl && nl2!=nl3) {
		if ( nl2->unicode[up]!=NULL && nl2->unicode[up][ub]!=NULL &&
			(name = nl2->unicode[up][ub][uc])!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    for ( i=0; psaltnames[i].name!=NULL ; ++i ) {
		if ( psaltnames[i].unicode==uni ) {
		    if ( names )
			names[cnt] = copy(psaltnames[i].name);
		    ++cnt;
		}
	    }
	    if ( uni<0x10000 ) {
		if ( names ) {
		    sprintf( buffer, "uni%04X", uni);
		    names[cnt] = copy(buffer);
		}
		++cnt;
	    }
	    if ( names ) {
		sprintf( buffer, "u%04X", uni);
		names[cnt] = copy(buffer);
	    }
	    ++cnt;
	}
	if ( rcnt>1 && alluni && (uni<0 || (uni>=0xe000 && uni<0xf900) || uni>=0xf0000 ) ) {
	    if ( names ) {
		names[cnt] = galloc(4+4*rcnt);
		strcpy(names[cnt],"uni");
		pt = names[cnt]+3;
		for ( i=0; i<rcnt; ++i ) {
		    if ( refs[i]->unicodeenc==0x131 || refs[i]->unicodeenc==0x237 ||
			    refs[i]->unicodeenc==0xf6be )
			sprintf( pt,"%04X", refs[i]->unicodeenc==0x131?'i':'j' );
		    else
			sprintf( pt,"%04X", CanonicalCombiner(refs[i]->unicodeenc));
		    pt += 4;
		}
	    }
	    ++cnt;
	}
	if ( rcnt>1 ) {
	    if ( names ) {
		for ( i=len=0; i<rcnt; ++i )
		    len += strlen( refs[i]->name )+1;
		names[cnt] = pt = galloc(len);
		for ( i=len=0; i<rcnt; ++i ) {
		    strcpy(pt,refs[i]->name);
		    pt += strlen(pt);
		    *pt++ = '_';
		}
		pt[-1] = '\0';
	    }
	    ++cnt;
	}
	if ( uni<0 || up>=17 ) {
	    if ( names )
		names[cnt] = copy(".notdef");
	    ++cnt;
	}
	if ( k==0 ) {
	    names = galloc((cnt+1)*sizeof(char *));
	    names[cnt] = NULL;
	}
    }
    /* Remove any names from multiiple namelists */
    for ( i=0; i<cnt; ++i ) for ( j=i+1; j<cnt; ++j ) {
	if ( strcmp(names[i],names[j])==0 ) {
	    for ( k=j+1; k<cnt; ++k )
		names[k-1] = names[k];
	    names[--cnt] = NULL;
	    --j;
	}
    }
return( names );
}

char **AllNamelistNames(void) {
    NameList *nl;
    int cnt;
    char **names;

    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt );
    names = galloc((cnt+1) *sizeof(char *));
    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt )
	names[cnt] = copy(_(nl->title));
    names[cnt] = NULL;
return( names );
}

#if 0
uint8 *AllNamelistUnicodes(void) {
    NameList *nl;
    int cnt;
    uint8 *uses;

    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt );
    uses = galloc((cnt+1) *sizeof(uint8));
    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt )
	uses[cnt] = nl->uses_unicode;
    uses[cnt] = 0xff;
return( uses );
}
#endif

NameList *DefaultNameListForNewFonts(void) {
return( namelist_for_new_fonts );
}

NameList *NameListByName(char *name) {
    NameList *nl;
    for ( nl = &agl; nl!=NULL; nl=nl->next ) {
	if ( strcmp(_(nl->title),name)==0 || strcmp(nl->title,name)==0 )
return( nl );
    }
return( NULL );
}

static void NameListFreeContents(NameList *nl) {
    int np, nb, nc, i;

    for ( np=0; np<17; ++np ) if ( nl->unicode[np]!=NULL ) {
	for ( nb=0; nb<256; ++nb ) if ( nl->unicode[np][nb]!=NULL ) {
	    for ( nc=0; nc<256; ++nc ) if ( nl->unicode[np][nb][nc]!=NULL )
		free((char *)nl->unicode[np][nb][nc] );
	    free( (char **) nl->unicode[np][nb]);
	}
	free( (char ***) nl->unicode[np]);
    }
    free( nl->title );
    if ( nl->renames!=NULL ) {
	for ( i=0; nl->renames[i].from!=NULL; ++i ) {
	    free(nl->renames[i].from);
	    free(nl->renames[i].to);
	}
	free(nl->renames);
    }
    free(nl->a_utf8_name);
}

static void NameListFree(NameList *nl) {
    NameListFreeContents(nl);
    chunkfree(nl,sizeof(NameList));
}
/* ************************************************************************** */

#include <sys/types.h>
#include <dirent.h>

NameList *LoadNamelist(char *filename) {
    FILE *file = fopen(filename,"r");
    NameList *nl, *nl2;
    char buffer[400];
    char *pt, *end, *test;
    int uni;
    int len;
    int up, ub, uc;
    int rn_cnt=0, rn_max = 0;
    int uses_unicode = false;

    if ( file==NULL )
return( NULL );

    if ( !psnamesinited )
	psinitnames();

    nl = chunkalloc(sizeof(NameList));
    pt = strrchr(filename,'/');
    if ( pt==NULL ) pt = filename; else ++pt;
    nl->title = def2utf8_copy(pt);
    pt = strrchr(nl->title,'.');
    if ( pt!=NULL ) *pt = '\0';

    while ( fgets(buffer,sizeof(buffer),file)!=NULL ) {
	if ( buffer[0]=='#' || buffer[0]=='\n' || buffer[0]=='\r' )
    continue;
	len = strlen( buffer );
	if ( buffer[len-1]=='\n' ) buffer[--len] = '\0';
	if ( buffer[len-1]=='\r' ) buffer[--len] = '\0';
	if ( strncmp(buffer,"Based:",6)==0 ) {
	    for ( pt=buffer+6; *pt==' ' || *pt=='\t'; ++pt);
	    for ( nl2 = &agl; nl2!=NULL; nl2 = nl2->next )
		if ( strcmp( nl2->title,pt )==0 )
	    break;
	    if ( nl2==NULL ) {
		ff_post_error(_("NameList base missing"),_("NameList %s based on %s which could not be found"), nl->title, pt );
		NameListFree(nl);
return( NULL );
	    } else if ( nl->basedon!=NULL ) {
		ff_post_error(_("NameList based twice"),_("NameList %s based on two NameLists"), nl->title );
		NameListFree(nl);
return( NULL );
	    }
	    nl->basedon = nl2;
	} else if ( strncmp(buffer,"Rename:",7)==0 ) {
	    for ( pt=buffer+7; *pt==' ' || *pt=='\t'; ++pt);
	    for ( test=pt; *test!=' ' && *test!='\t' && *test!='\0'; ++test );
	    if ( *test=='\0' ) {
		ff_post_error(_("NameList parsing error"),_("Missing rename \"to\" name %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    *test='\0';
	    for ( ++test; *test==' ' || *test=='\t'; ++test);
	    if ( (test[0]=='-' || test[0]=='=') && test[1]=='>' )
		for ( test+=2; *test==' ' || *test=='\t'; ++test);
	    if ( *test=='\0' ) {
		ff_post_error(_("NameList parsing error"),_("Missing rename \"to\" name %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    if ( rn_cnt>=rn_max-1 )
		nl->renames = grealloc(nl->renames,(rn_max+=20)*sizeof(struct renames));
	    nl->renames[rn_cnt].from   = copy(pt);
	    nl->renames[rn_cnt].to     = copy(test);
	    nl->renames[++rn_cnt].from = NULL;		/* End mark */
	} else {
	    pt = buffer;
	    if ( *pt=='0' && (pt[1]=='x' || pt[1]=='X'))
		pt += 2;
	    else if (( *pt=='u' || *pt=='U') && pt[1]=='+' )
		pt += 2;
	    uni = strtol(pt,&end,16);
	    if ( end==pt || uni<0 || uni>=unicode4_size ) {
		ff_post_error(_("NameList parsing error"),_("Bad unicode value when parsing %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    pt = end;
	    while ( *pt==' ' || *pt==';' || *pt=='\t' ) ++pt;
	    if ( *pt=='\0' ) {
		ff_post_error(_("NameList parsing error"),_("Missing name when parsing %s for unicode %x"), nl->title, uni );
		NameListFree(nl);
return( NULL );
	    }
	    for ( test=pt; *test; ++test ) {
		if ( (*test<=' ' && *test>=0) ||
		    *test=='(' || *test=='[' || *test=='{' || *test=='<' ||
		    *test==')' || *test==']' || *test=='}' || *test=='>' ||
		    *test=='%' || *test=='/' ) {
		    ff_post_error(_("NameList parsing error"),_("Bad name when parsing %s for unicode %x"), nl->title, uni );
		    NameListFree(nl);
return( NULL );
		}
		if ( *test&0x80 ) {
		    uses_unicode = true;
		    if ( nl->a_utf8_name==NULL )
			nl->a_utf8_name = copy(pt);
		}
	    }
	    up = uni>>16;
	    ub = (uni&0xff00)>>8;
	    uc = uni&0xff;
	    if ( nl->unicode[up]==NULL )
		nl->unicode[up] = gcalloc(256,sizeof(char **));
	    if ( nl->unicode[up][ub]==NULL )
		nl->unicode[up][ub] = gcalloc(256,sizeof(char *));
	    if ( nl->unicode[up][ub][uc]==NULL )
		nl->unicode[up][ub][uc]=copy(pt);
	    else {
		ff_post_error(_("NameList parsing error"),_("Multiple names when parsing %s for unicode %x"), nl->title, uni );
		NameListFree(nl);
return( NULL );
	    }
	}
    }

    nl->uses_unicode = uses_unicode;
    if ( nl->basedon!=NULL && nl->basedon->uses_unicode )
	nl->uses_unicode = true;
    fclose(file);
    for ( nl2 = &agl; nl2->next!=NULL; nl2=nl2->next ) {
	if ( strcmp(nl2->title,nl->title)==0 ) {	/* Replace old version */
	    NameList *next = nl2->next;
	    NameListFreeContents(nl2);
	    *nl2 = *nl;
	    nl2->next = next;
	    chunkfree(nl,sizeof(NameList));
	    psreinitnames();
return( nl2 );
	}
    }
    NameListHash(nl);
    nl2->next = nl;
return( nl );
}


/* ************************************************************************** */
const char *RenameGlyphToNamelist(char *buffer, SplineChar *sc,NameList *old,NameList *new) {
    int i, up, ub, uc, ch;
    char space[80];		/* glyph names are supposed to be less<=31 chars */
    char tempbuf[32];
    char *pt, *start, *opt, *oend; const char *newsubname;
    SplineChar *tempsc;
    NameList *nl;

    if ( sc->unicodeenc!=-1 ) {
	up = sc->unicodeenc>>16;
	ub = (sc->unicodeenc>>8)&0xff;
	uc = (sc->unicodeenc&0xff);
	for ( nl=new; nl!=NULL; nl=nl->basedon )
	    if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL && nl->unicode[up][ub][uc]!=NULL )
return( nl->unicode[up][ub][uc] );
	if ( up==0 )
	    sprintf( buffer, "uni%04X", sc->unicodeenc );
	else
	    sprintf( buffer, "u%04X", sc->unicodeenc );
return( buffer );
    } else {
	if ( old!=NULL && old->renames!=NULL ) {
	    for ( i=0; old->renames[i].from!=NULL; ++i )
		if ( strcmp(sc->name,old->renames[i].from)==0 )
return( old->renames[i].to );
	}
	if ( new->renames!=NULL ) {
	    for ( i=0; new->renames[i].from!=NULL; ++i )
		if ( strcmp(sc->name,new->renames[i].to)==0 )
return( new->renames[i].from );
	}
	if ( strlen(sc->name)>sizeof(space)-1 )
return( sc->name );
	strcpy(space,sc->name);
	opt = buffer; oend = buffer+31;
	start = space;
	/* Check for composite names f_i, A.small */
	while ( *start ) {
	    for ( pt=start; *pt!='\0' && *pt!='.' && *pt!='_'; ++pt );
	    if ( *pt=='\0' && start==space )
return( sc->name );
	    ch = *pt;
	    *pt = '\0';
	    tempsc = SFGetChar(sc->parent,-1,start);
	    if ( tempsc==NULL )
return( sc->name );
	    newsubname = RenameGlyphToNamelist(tempbuf,tempsc,old,new);
	    while ( opt<oend && *newsubname )
		*opt++ = *newsubname++;
	    if ( *newsubname )
return( sc->name );
	    if ( ch=='\0' ) {
		*opt = '\0';
return( buffer );
	    } else if ( ch=='.' ) {
		/* don't attempt to translate anything after a '.' just copy that litterally */
		while ( opt<oend && *pt )
		    *opt++ = *pt++;
		if ( *pt )
return( sc->name );
		*opt = '\0';
return( buffer );
	    } else {		/* _ */
		*opt++ = '_';
		start = pt+1;
	    }
	}
	*opt = '\0';
return( buffer );
    }
}

void SFRenameGlyphsToNamelist(SplineFont *sf,NameList *new) {
    int gid;
    char buffer[40]; const char *name;
    SplineChar *sc;

    if ( new==NULL )
return;

    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	name = RenameGlyphToNamelist(buffer,sc,sf->for_new_glyphs,new);
	if ( name!=sc->name ) {
	    free(sc->name);
	    sc->name = copy(name);
	}
    }
    sf->for_new_glyphs = new;
}

char **SFTemporaryRenameGlyphsToNamelist(SplineFont *sf,NameList *new) {
    int gid;
    char buffer[40]; const char *name;
    SplineChar *sc;
    char **ret;

    if ( new==NULL )
return( NULL );

    ret = gcalloc(sf->glyphcnt,sizeof(char *));
    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	name = RenameGlyphToNamelist(buffer,sc,sf->for_new_glyphs,new);
	if ( name!=sc->name ) {
	    ret[gid] = sc->name;
	    sc->name = copy(name);
	}
    }
return( ret );
}

void SFTemporaryRestoreGlyphNames(SplineFont *sf,char **former) {
    int gid;
    SplineChar *sc;

    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	if ( former[gid]!=NULL ) {
	    free(sc->name);
	    sc->name = former[gid];
	}
    }
    free(former);
}
/* ************************************************************************** */
static const char *agl_sans_p0_b0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"space",
	"exclam",
	"quotedbl",
	"numbersign",
	"dollar",
	"percent",
	"ampersand",
	"quotesingle",
	"parenleft",
	"parenright",
	"asterisk",
	"plus",
	"comma",
	"hyphen",
	"period",
	"slash",
	"zero",
	"one",
	"two",
	"three",
	"four",
	"five",
	"six",
	"seven",
	"eight",
	"nine",
	"colon",
	"semicolon",
	"less",
	"equal",
	"greater",
	"question",
	"at",
	"A",
	"B",
	"C",
	"D",
	"E",
	"F",
	"G",
	"H",
	"I",
	"J",
	"K",
	"L",
	"M",
	"N",
	"O",
	"P",
	"Q",
	"R",
	"S",
	"T",
	"U",
	"V",
	"W",
	"X",
	"Y",
	"Z",
	"bracketleft",
	"backslash",
	"bracketright",
	"asciicircum",
	"underscore",
	"grave",
	"a",
	"b",
	"c",
	"d",
	"e",
	"f",
	"g",
	"h",
	"i",
	"j",
	"k",
	"l",
	"m",
	"n",
	"o",
	"p",
	"q",
	"r",
	"s",
	"t",
	"u",
	"v",
	"w",
	"x",
	"y",
	"z",
	"braceleft",
	"bar",
	"braceright",
	"asciitilde",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclamdown",
	"cent",
	"sterling",
	"currency",
	"yen",
	"brokenbar",
	"section",
	"dieresis",
	"copyright",
	"ordfeminine",
	"guillemotleft",
	"logicalnot",
	NULL,
	"registered",
	"macron",
	"degree",
	"plusminus",
	NULL,
	NULL,
	"acute",
	NULL,
	"paragraph",
	"periodcentered",
	"cedilla",
	NULL,
	"ordmasculine",
	"guillemotright",
	"onequarter",
	"onehalf",
	"threequarters",
	"questiondown",
	"Agrave",
	"Aacute",
	"Acircumflex",
	"Atilde",
	"Adieresis",
	"Aring",
	"AE",
	"Ccedilla",
	"Egrave",
	"Eacute",
	"Ecircumflex",
	"Edieresis",
	"Igrave",
	"Iacute",
	"Icircumflex",
	"Idieresis",
	"Eth",
	"Ntilde",
	"Ograve",
	"Oacute",
	"Ocircumflex",
	"Otilde",
	"Odieresis",
	"multiply",
	"Oslash",
	"Ugrave",
	"Uacute",
	"Ucircumflex",
	"Udieresis",
	"Yacute",
	"Thorn",
	"germandbls",
	"agrave",
	"aacute",
	"acircumflex",
	"atilde",
	"adieresis",
	"aring",
	"ae",
	"ccedilla",
	"egrave",
	"eacute",
	"ecircumflex",
	"edieresis",
	"igrave",
	"iacute",
	"icircumflex",
	"idieresis",
	"eth",
	"ntilde",
	"ograve",
	"oacute",
	"ocircumflex",
	"otilde",
	"odieresis",
	"divide",
	"oslash",
	"ugrave",
	"uacute",
	"ucircumflex",
	"udieresis",
	"yacute",
	"thorn",
	"ydieresis"
};

static const char *agl_sans_p0_b1[] = {
	"Amacron",
	"amacron",
	"Abreve",
	"abreve",
	"Aogonek",
	"aogonek",
	"Cacute",
	"cacute",
	"Ccircumflex",
	"ccircumflex",
	"Cdotaccent",
	"cdotaccent",
	"Ccaron",
	"ccaron",
	"Dcaron",
	"dcaron",
	"Dcroat",
	"dcroat",
	"Emacron",
	"emacron",
	"Ebreve",
	"ebreve",
	"Edotaccent",
	"edotaccent",
	"Eogonek",
	"eogonek",
	"Ecaron",
	"ecaron",
	"Gcircumflex",
	"gcircumflex",
	"Gbreve",
	"gbreve",
	"Gdotaccent",
	"gdotaccent",
	"Gcommaaccent",
	"gcommaaccent",
	"Hcircumflex",
	"hcircumflex",
	"Hbar",
	"hbar",
	"Itilde",
	"itilde",
	"Imacron",
	"imacron",
	"Ibreve",
	"ibreve",
	"Iogonek",
	"iogonek",
	"Idotaccent",
	"dotlessi",
	"IJ",
	"ij",
	"Jcircumflex",
	"jcircumflex",
	"Kcommaaccent",
	"kcommaaccent",
	"kgreenlandic",
	"Lacute",
	"lacute",
	"Lcommaaccent",
	"lcommaaccent",
	"Lcaron",
	"lcaron",
	"Ldot",
	"ldot",
	"Lslash",
	"lslash",
	"Nacute",
	"nacute",
	"Ncommaaccent",
	"ncommaaccent",
	"Ncaron",
	"ncaron",
	"napostrophe",
	"Eng",
	"eng",
	"Omacron",
	"omacron",
	"Obreve",
	"obreve",
	"Ohungarumlaut",
	"ohungarumlaut",
	"OE",
	"oe",
	"Racute",
	"racute",
	"Rcommaaccent",
	"rcommaaccent",
	"Rcaron",
	"rcaron",
	"Sacute",
	"sacute",
	"Scircumflex",
	"scircumflex",
	"Scedilla",
	"scedilla",
	"Scaron",
	"scaron",
	NULL,
	NULL,
	"Tcaron",
	"tcaron",
	"Tbar",
	"tbar",
	"Utilde",
	"utilde",
	"Umacron",
	"umacron",
	"Ubreve",
	"ubreve",
	"Uring",
	"uring",
	"Uhungarumlaut",
	"uhungarumlaut",
	"Uogonek",
	"uogonek",
	"Wcircumflex",
	"wcircumflex",
	"Ycircumflex",
	"ycircumflex",
	"Ydieresis",
	"Zacute",
	"zacute",
	"Zdotaccent",
	"zdotaccent",
	"Zcaron",
	"zcaron",
	"longs",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"florin",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ohorn",
	"ohorn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Uhorn",
	"uhorn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Gcaron",
	"gcaron",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Aringacute",
	"aringacute",
	"AEacute",
	"aeacute",
	"Oslashacute",
	"oslashacute"
};

static const char *agl_sans_p0_b2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Scommaaccent",
	"scommaaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"circumflex",
	"caron",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"breve",
	"dotaccent",
	"ring",
	"ogonek",
	"tilde",
	"hungarumlaut",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b3[] = {
	"gravecomb",
	"acutecomb",
	NULL,
	"tildecomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"hookabovecomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dotbelowcomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"tonos",
	"dieresistonos",
	"Alphatonos",
	"anoteleia",
	"Epsilontonos",
	"Etatonos",
	"Iotatonos",
	NULL,
	"Omicrontonos",
	NULL,
	"Upsilontonos",
	"Omegatonos",
	"iotadieresistonos",
	"Alpha",
	"Beta",
	"Gamma",
	"Delta",
	"Epsilon",
	"Zeta",
	"Eta",
	"Theta",
	"Iota",
	"Kappa",
	"Lambda",
	"Mu",
	"Nu",
	"Xi",
	"Omicron",
	"Pi",
	"Rho",
	NULL,
	"Sigma",
	"Tau",
	"Upsilon",
	"Phi",
	"Chi",
	"Psi",
	"Omega",
	"Iotadieresis",
	"Upsilondieresis",
	"alphatonos",
	"epsilontonos",
	"etatonos",
	"iotatonos",
	"upsilondieresistonos",
	"alpha",
	"beta",
	"gamma",
	"delta",
	"epsilon",
	"zeta",
	"eta",
	"theta",
	"iota",
	"kappa",
	"lambda",
	"mu",
	"nu",
	"xi",
	"omicron",
	"pi",
	"rho",
	"sigma1",
	"sigma",
	"tau",
	"upsilon",
	"phi",
	"chi",
	"psi",
	"omega",
	"iotadieresis",
	"upsilondieresis",
	"omicrontonos",
	"upsilontonos",
	"omegatonos",
	NULL,
	NULL,
	"theta1",
	"Upsilon1",
	NULL,
	NULL,
	"phi1",
	"omega1",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b1e[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Wgrave",
	"wgrave",
	"Wacute",
	"wacute",
	"Wdieresis",
	"wdieresis",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ygrave",
	"ygrave",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"figuredash",
	"endash",
	"emdash",
	NULL,
	NULL,
	"underscoredbl",
	"quoteleft",
	"quoteright",
	"quotesinglbase",
	"quotereversed",
	"quotedblleft",
	"quotedblright",
	"quotedblbase",
	NULL,
	"dagger",
	"daggerdbl",
	"bullet",
	NULL,
	"onedotenleader",
	"twodotenleader",
	"ellipsis",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perthousand",
	NULL,
	"minute",
	"second",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"guilsinglleft",
	"guilsinglright",
	NULL,
	"exclamdbl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"fraction",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"colonmonetary",
	NULL,
	"franc",
	"lira",
	NULL,
	NULL,
	"peseta",
	NULL,
	NULL,
	NULL,
	"dong",
	"Euro",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ifraktur",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"weierstrass",
	NULL,
	NULL,
	NULL,
	"Rfraktur",
	NULL,
	"prescription",
	NULL,
	NULL,
	NULL,
	"trademark",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"estimated",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"aleph",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"onethird",
	"twothirds",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"oneeighth",
	"threeeighths",
	"fiveeighths",
	"seveneighths",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowleft",
	"arrowup",
	"arrowright",
	"arrowdown",
	"arrowboth",
	"arrowupdn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowupdnbse",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"carriagereturn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowdblleft",
	"arrowdblup",
	"arrowdblright",
	"arrowdbldown",
	"arrowdblboth",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b22[] = {
	"universal",
	NULL,
	"partialdiff",
	"existential",
	NULL,
	"emptyset",
	NULL,
	"gradient",
	"element",
	"notelement",
	NULL,
	"suchthat",
	NULL,
	NULL,
	NULL,
	"product",
	NULL,
	"summation",
	"minus",
	NULL,
	NULL,
	NULL,
	NULL,
	"asteriskmath",
	NULL,
	NULL,
	"radical",
	NULL,
	NULL,
	"proportional",
	"infinity",
	"orthogonal",
	"angle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"logicaland",
	"logicalor",
	"intersection",
	"union",
	"integral",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"therefore",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"similar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congruent",
	NULL,
	NULL,
	"approxequal",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"notequal",
	"equivalence",
	NULL,
	NULL,
	"lessequal",
	"greaterequal",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"propersubset",
	"propersuperset",
	"notsubset",
	NULL,
	"reflexsubset",
	"reflexsuperset",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"circleplus",
	NULL,
	"circlemultiply",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perpendicular",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dotmath",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b23[] = {
	NULL,
	NULL,
	"house",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"revlogicalnot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"integraltp",
	"integralbt",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"angleleft",
	"angleright",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b25[] = {
	"SF100000",
	NULL,
	"SF110000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF010000",
	NULL,
	NULL,
	NULL,
	"SF030000",
	NULL,
	NULL,
	NULL,
	"SF020000",
	NULL,
	NULL,
	NULL,
	"SF040000",
	NULL,
	NULL,
	NULL,
	"SF080000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF090000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF060000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF070000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF050000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF430000",
	"SF240000",
	"SF510000",
	"SF520000",
	"SF390000",
	"SF220000",
	"SF210000",
	"SF250000",
	"SF500000",
	"SF490000",
	"SF380000",
	"SF280000",
	"SF270000",
	"SF260000",
	"SF360000",
	"SF370000",
	"SF420000",
	"SF190000",
	"SF200000",
	"SF230000",
	"SF470000",
	"SF480000",
	"SF410000",
	"SF450000",
	"SF460000",
	"SF400000",
	"SF540000",
	"SF530000",
	"SF440000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"upblock",
	NULL,
	NULL,
	NULL,
	"dnblock",
	NULL,
	NULL,
	NULL,
	"block",
	NULL,
	NULL,
	NULL,
	"lfblock",
	NULL,
	NULL,
	NULL,
	"rtblock",
	"ltshade",
	"shade",
	"dkshade",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"filledbox",
	"H22073",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"H18543",
	"H18551",
	"filledrect",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triagup",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triagrt",
	NULL,
	"triagdn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triaglf",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lozenge",
	"circle",
	NULL,
	NULL,
	NULL,
	"H18533",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"invbullet",
	"invcircle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"openbullet",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b26[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"smileface",
	"invsmileface",
	"sun",
	NULL,
	NULL,
	NULL,
	"female",
	NULL,
	"male",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"spade",
	NULL,
	NULL,
	"club",
	NULL,
	"heart",
	"diamond",
	NULL,
	NULL,
	NULL,
	"musicalnote",
	"musicalnotedbl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **agl_sans_p0[] = {
	agl_sans_p0_b0,
	agl_sans_p0_b1,
	agl_sans_p0_b2,
	agl_sans_p0_b3,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	agl_sans_p0_b1e,
	NULL,
	agl_sans_p0_b20,
	agl_sans_p0_b21,
	agl_sans_p0_b22,
	agl_sans_p0_b23,
	NULL,
	agl_sans_p0_b25,
	agl_sans_p0_b26,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList agl_sans = {
	NULL,
	N_("AGL without afii"),
	{ agl_sans_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static const char *agl_p0_b1[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Tcommaaccent",
	"tcommaaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57929",
	"afii64937",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b4[] = {
	NULL,
	"afii10023",
	"afii10051",
	"afii10052",
	"afii10053",
	"afii10054",
	"afii10055",
	"afii10056",
	"afii10057",
	"afii10058",
	"afii10059",
	"afii10060",
	"afii10061",
	NULL,
	"afii10062",
	"afii10145",
	"afii10017",
	"afii10018",
	"afii10019",
	"afii10020",
	"afii10021",
	"afii10022",
	"afii10024",
	"afii10025",
	"afii10026",
	"afii10027",
	"afii10028",
	"afii10029",
	"afii10030",
	"afii10031",
	"afii10032",
	"afii10033",
	"afii10034",
	"afii10035",
	"afii10036",
	"afii10037",
	"afii10038",
	"afii10039",
	"afii10040",
	"afii10041",
	"afii10042",
	"afii10043",
	"afii10044",
	"afii10045",
	"afii10046",
	"afii10047",
	"afii10048",
	"afii10049",
	"afii10065",
	"afii10066",
	"afii10067",
	"afii10068",
	"afii10069",
	"afii10070",
	"afii10072",
	"afii10073",
	"afii10074",
	"afii10075",
	"afii10076",
	"afii10077",
	"afii10078",
	"afii10079",
	"afii10080",
	"afii10081",
	"afii10082",
	"afii10083",
	"afii10084",
	"afii10085",
	"afii10086",
	"afii10087",
	"afii10088",
	"afii10089",
	"afii10090",
	"afii10091",
	"afii10092",
	"afii10093",
	"afii10094",
	"afii10095",
	"afii10096",
	"afii10097",
	NULL,
	"afii10071",
	"afii10099",
	"afii10100",
	"afii10101",
	"afii10102",
	"afii10103",
	"afii10104",
	"afii10105",
	"afii10106",
	"afii10107",
	"afii10108",
	"afii10109",
	NULL,
	"afii10110",
	"afii10193",
	NULL,
	NULL,
	"afii10146",
	"afii10194",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10147",
	"afii10195",
	"afii10148",
	"afii10196",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10050",
	"afii10098",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10846",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57799",
	"afii57801",
	"afii57800",
	"afii57802",
	"afii57793",
	"afii57794",
	"afii57795",
	"afii57798",
	"afii57797",
	"afii57806",
	NULL,
	"afii57796",
	"afii57807",
	"afii57839",
	"afii57645",
	"afii57841",
	"afii57842",
	"afii57804",
	"afii57803",
	"afii57658",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57664",
	"afii57665",
	"afii57666",
	"afii57667",
	"afii57668",
	"afii57669",
	"afii57670",
	"afii57671",
	"afii57672",
	"afii57673",
	"afii57674",
	"afii57675",
	"afii57676",
	"afii57677",
	"afii57678",
	"afii57679",
	"afii57680",
	"afii57681",
	"afii57682",
	"afii57683",
	"afii57684",
	"afii57685",
	"afii57686",
	"afii57687",
	"afii57688",
	"afii57689",
	"afii57690",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57716",
	"afii57717",
	"afii57718",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57388",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57403",
	NULL,
	NULL,
	NULL,
	"afii57407",
	NULL,
	"afii57409",
	"afii57410",
	"afii57411",
	"afii57412",
	"afii57413",
	"afii57414",
	"afii57415",
	"afii57416",
	"afii57417",
	"afii57418",
	"afii57419",
	"afii57420",
	"afii57421",
	"afii57422",
	"afii57423",
	"afii57424",
	"afii57425",
	"afii57426",
	"afii57427",
	"afii57428",
	"afii57429",
	"afii57430",
	"afii57431",
	"afii57432",
	"afii57433",
	"afii57434",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57440",
	"afii57441",
	"afii57442",
	"afii57443",
	"afii57444",
	"afii57445",
	"afii57446",
	"afii57470",
	"afii57448",
	"afii57449",
	"afii57450",
	"afii57451",
	"afii57452",
	"afii57453",
	"afii57454",
	"afii57455",
	"afii57456",
	"afii57457",
	"afii57458",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57392",
	"afii57393",
	"afii57394",
	"afii57395",
	"afii57396",
	"afii57397",
	"afii57398",
	"afii57399",
	"afii57400",
	"afii57401",
	"afii57381",
	NULL,
	NULL,
	"afii63167",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57511",
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57506",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57507",
	NULL,
	"afii57512",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57513",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57508",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57505",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57509",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57514",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57519",
	NULL,
	NULL,
	"afii57534",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61664",
	"afii301",
	"afii299",
	"afii300",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii00208",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61573",
	"afii61574",
	"afii61575",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57636",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61248",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61289",
	NULL,
	NULL,
	"afii61352",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **agl_p0[] = {
	NULL,
	agl_p0_b1,
	agl_p0_b2,
	NULL,
	agl_p0_b4,
	agl_p0_b5,
	agl_p0_b6,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	agl_p0_b20,
	agl_p0_b21,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList agl = {
	&agl_sans,
	N_("Adobe Glyph List"),
	{ agl_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static const char *adobepua_p0_bf6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perthousand.oldstyle",
	"cent.denominator",
	"dollar.denominator",
	"hyphen.denominator",
	"parenleft.denominator",
	"parenright.denominator",
	"cent.numerator",
	"dollar.numerator",
	"hyphen.numerator",
	"parenleft.numerator",
	"parenright.numerator",
	"at.cap",
	"commaaccent.cap",
	"commaaccent.small",
	"pi.math",
	"h.superior",
	"zero.slash",
	"zero.fitted",
	"two.fitted",
	"three.fitted",
	"four.fitted",
	"five.fitted",
	"six.fitted",
	"seven.fitted",
	"eight.fitted",
	"nine.fitted",
	"percent.oldstyle",
	"zero.taboldstyle",
	"one.taboldstyle",
	"two.taboldstyle",
	"three.taboldstyle",
	"four.taboldstyle",
	"five.taboldstyle",
	"six.taboldstyle",
	"seven.taboldstyle",
	"eight.taboldstyle",
	"nine.taboldstyle",
	"colonmonetary.taboldstyle",
	"Euro.taboldstyle",
	"florin.taboldstyle",
	"numbersign.taboldstyle",
	"sterling.taboldstyle",
	"yen.taboldstyle",
	"dollar.taboldstyle",
	"cent.taboldstyle",
	"zero.denominator",
	"one.denominator",
	"two.denominator",
	"three.denominator",
	"four.denominator",
	"five.denominator",
	"six.denominator",
	"seven.denominator",
	"eight.denominator",
	"nine.denominator",
	"comma.denominator",
	"period.denominator",
	"zero.numerator",
	"one.numerator",
	"two.numerator",
	"three.numerator",
	"four.numerator",
	"five.numerator",
	"six.numerator",
	"seven.numerator",
	"eight.numerator",
	"nine.numerator",
	"comma.numerator",
	"period.numerator",
	"Abreve.small",
	"Amacron.small",
	"Aogonek.small",
	"AEacute.small",
	"Cacute.small",
	"Ccaron.small",
	"Ccircumflex.small",
	"Cdotaccent.small",
	"Dcaron.small",
	"Dcroat.small",
	"Ebreve.small",
	"Ecaron.small",
	"Edotaccent.small",
	"Emacron.small",
	"Eng.small",
	"Eogonek.small",
	"Gbreve.small",
	"Gcircumflex.small",
	"Gcommaaccent.small",
	"Gdotaccent.small",
	"Hbar.small",
	"Hcircumflex.small",
	"Ibreve.small",
	"IJ.small",
	"Imacron.small",
	"Iogonek.small",
	"Itilde.small",
	"Jcircumflex.small",
	"Kcommaaccent.small",
	"Lacute.small",
	"Lcaron.small",
	"Lcommaaccent.small",
	"Ldot.small",
	"Nacute.small",
	"Ncaron.small",
	"Ncommaaccent.small",
	"Obreve.small",
	"Ohungarumlaut.small",
	"Omacron.small",
	"Oslashacute.small",
	"Racute.small",
	"Rcaron.small",
	"Rcommaaccent.small",
	"Sacute.small",
	"Scedilla.small",
	"Scircumflex.small",
	"Scommaaccent.small",
	"Tbar.small",
	"Tcaron.small",
	"Tcommaaccent.small",
	"Ubreve.small",
	"Uhungarumlaut.small",
	"Umacron.small",
	"Uogonek.small",
	"Uring.small",
	"Utilde.small",
	"Wacute.small",
	"Wcircumflex.small",
	"Wdieresis.small",
	"Wgrave.small",
	"Ycircumflex.small",
	"Ygrave.small",
	"Zacute.small",
	"Zdotaccent.small",
	"Idotaccent.small",
	"parenleft.cap",
	"parenright.cap",
	"bracketleft.cap",
	"bracketright.cap",
	"braceleft.cap",
	"braceright.cap",
	"exclamdown.cap",
	"questiondown.cap",
	"guillemotleft.cap",
	"guillemotright.cap",
	"guilsinglleft.cap",
	"guilsinglright.cap",
	"hyphen.cap",
	"endash.cap",
	"emdash.cap",
	"periodcentered.cap",
	"j.dotless",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10068.ital",
	"afii10066.ital",
	"afii10069.ital",
	"afii10081.ital",
	"afii10084.ital",
	"acute.cap",
	"caron.cap",
	"dieresis.cap",
	"space_uni0308_uni0301.cap",
	"space_uni0308_uni0300.cap",
	"grave.cap",
	"hungarumlaut.cap",
	"macron.cap",
	"breve.cyrcap",
	"circumflex.cyrcap",
	"space_uni030F.cap",
	"breve.cyr",
	"circumflex.cyr",
	"space_uni030F",
	"space_uni0308_uni0301",
	"space_uni0308_uni0300",
	"copyright.serif",
	"registered.serif",
	"trademark.serif",
	"one.fitted",
	"R_p",
	"uni2014.alt",
	"cent.inferior",
	"cent.superior",
	"comma.inferior",
	"comma.superior",
	"dollar.inferior",
	"dollar.superior",
	"hyphen.inferior",
	"hyphen.superior",
	"period.inferior",
	"period.superior",
	"a.superior",
	"b.superior",
	"d.superior",
	"e.superior",
	"i.superior",
	"l.superior",
	"m.superior",
	"o.superior",
	"r.superior",
	"s.superior",
	"t.superior",
	"breve.sc",
	"caron.sc",
	"circumflex.sc",
	"dotaccent.sc",
	"hungarumlaut.sc",
	"lslash.sc",
	"oe.sc",
	"ogonek.sc",
	"ring.sc",
	"scaron.sc",
	"tilde.sc",
	"zcaron.sc"
};

static const char *adobepua_p0_bf7[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclam.sc",
	NULL,
	NULL,
	"dollar.oldstyle",
	NULL,
	"ampersand.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"zero.oldstyle",
	"one.oldstyle",
	"two.oldstyle",
	"three.oldstyle",
	"four.oldstyle",
	"five.oldstyle",
	"six.oldstyle",
	"seven.oldstyle",
	"eight.oldstyle",
	"nine.oldstyle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"question.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"grave.sc",
	"a.sc",
	"b.sc",
	"c.sc",
	"d.sc",
	"e.sc",
	"f.sc",
	"g.sc",
	"h.sc",
	"i.sc",
	"j.sc",
	"k.sc",
	"l.sc",
	"m.sc",
	"n.sc",
	"o.sc",
	"p.sc",
	"q.sc",
	"r.sc",
	"s.sc",
	"t.sc",
	"u.sc",
	"v.sc",
	"w.sc",
	"x.sc",
	"y.sc",
	"z.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclamdown.sc",
	"cent.oldstyle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dieresis.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"macron.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	"acute.sc",
	NULL,
	NULL,
	NULL,
	"cedilla.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"questiondown.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"agrave.sc",
	"aacute.sc",
	"acircumflex.sc",
	"atilde.sc",
	"adieresis.sc",
	"aring.sc",
	"ae.sc",
	"ccedilla.sc",
	"egrave.sc",
	"eacute.sc",
	"ecircumflex.sc",
	"edieresis.sc",
	"igrave.sc",
	"iacute.sc",
	"icircumflex.sc",
	"idieresis.sc",
	"eth.sc",
	"ntilde.sc",
	"ograve.sc",
	"oacute.sc",
	"ocircumflex.sc",
	"otilde.sc",
	"odieresis.sc",
	NULL,
	"oslash.sc",
	"ugrave.sc",
	"uacute.sc",
	"ucircumflex.sc",
	"udieresis.sc",
	"yacute.sc",
	"thorn.sc",
	"ydieresis.sc"
};

static const char *adobepua_p0_bf8[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"registered.sans",
	"copyright.sans",
	"trademark.sans",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *adobepua_p0_bfb[] = {
	"f_f",
	"f_i",
	"f_l",
	"f_f_i",
	"f_f_l",
	"longs_t",
	"s_t",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **adobepua_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	adobepua_p0_bf6,
	adobepua_p0_bf7,
	adobepua_p0_bf8,
	NULL,
	NULL,
	adobepua_p0_bfb,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList adobepua = {
	&agl,
	N_("AGL with PUA"),
	{ adobepua_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static const char *greeksc_p0_bf5[] = {
	"alpha.sc",
	"beta.sc",
	"gamma.sc",
	"delta.sc",
	"epsilon.sc",
	"zeta.sc",
	"eta.sc",
	"theta.sc",
	"iota.sc",
	"kappa.sc",
	"lambda.sc",
	"mu.sc",
	"nu.sc",
	"xi.sc",
	"omicron.sc",
	"pi.sc",
	"rho.sc",
	NULL,
	"sigma.sc",
	"tau.sc",
	"upsilon.sc",
	"phi.sc",
	"chi.sc",
	"psi.sc",
	"omega.sc",
	"iotadieresis.sc",
	"upsilondieresis.sc",
	"alphaiota.sc",
	"etaiota.sc",
	"omegaiota.sc",
	"prosgegrammeni.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **greeksc_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	greeksc_p0_bf5,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList greeksc = {
	&adobepua,
	N_("Greek small caps"),
	{ greeksc_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static const char *tex_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"closure",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"leftharpoonaccent",
	"rightharpoonaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"shortleftarrow",
	"shortuparrow",
	"shortrightarrow",
	"shortdownarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"leftwavyarrow",
	"rightwavyarrow",
	NULL,
	"twoheaduparrow",
	NULL,
	"twoheaddownarrow",
	NULL,
	NULL,
	NULL,
	"mapsup",
	NULL,
	"mapsdown",
	"updownarrowbar",
	NULL,
	NULL,
	NULL,
	NULL,
	"leftrightwavyarrow",
	NULL,
	"downzigzagarrow",
	NULL,
	NULL,
	"Ldsh",
	"Rdsh",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"acwopencirclearrow",
	"cwopencirclearrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"updownarrows",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Nwarrow",
	"Nearrow",
	"Searrow",
	"Swarrow",
	NULL,
	NULL,
	"longleftzigzagarrow",
	"rightzigzagarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"barleftarrow",
	"rightarrowbar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"downuparrows",
	"rightthreearrows",
	"nvleftarrow",
	"nvrightarrow",
	"nvleftrightarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b22[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"ltrpar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"difference",
	NULL,
	"ac",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"approxident",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arceq",
	NULL,
	"veeeq",
	"stareq",
	NULL,
	"eqdef",
	"measeq",
	NULL,
	NULL,
	NULL,
	NULL,
	"Equiv",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"prurel",
	"scurel",
	NULL,
	NULL,
	NULL,
	NULL,
	"origof",
	"imageof",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"eqless",
	"eqgtr",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"disin",
	NULL,
	"isins",
	"isindot",
	NULL,
	NULL,
	"isinvb",
	"isinE",
	"nisd",
	NULL,
	"nis",
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b23[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"invnot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"turnednot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"solbar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rangledownzigzagarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b27[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"longrightzigzagarrow"
};

static const char *tex_p0_b29[] = {
	NULL,
	NULL,
	"nvLeftarrow",
	"nvRightarrow",
	"nvLeftrightarrow",
	"twoheadmapsto",
	NULL,
	NULL,
	"downarrowbarred",
	"uparrowbarred",
	"Uuparrow",
	"Ddownarrow",
	"leftbkarrow",
	"rightbkarrow",
	"leftdbkarrow",
	NULL,
	NULL,
	"rightdotarrow",
	"baruparrow",
	"downarrowbar",
	NULL,
	NULL,
	"twoheadrightarrowtail",
	NULL,
	NULL,
	"lefttail",
	"righttail",
	"leftdbltail",
	"rightdbltail",
	"diamondleftarrow",
	"rightarrowdiamond",
	"diamondleftarrowbar",
	"barrightarrowdiamond",
	"nwsearrow",
	"neswarrow",
	"hknwarrow",
	"hknearrow",
	"hksearrow",
	"hkswarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	"rdiagovfdiag",
	"fdiagovrdiag",
	"seovnearrow",
	"neovsearrow",
	"fdiagovnearrow",
	"rdiagovsearrow",
	"neovnwarrow",
	"nwovnearrow",
	"rightcurvedarrow",
	NULL,
	NULL,
	"leftdowncurvedarrow",
	"rightdowncurvedarrow",
	"cwhalfcirclearrow",
	"acwhalfcirclearrow",
	NULL,
	"acwundercurvearrow",
	"curvearrowrightminus",
	"curvearrowleftplus",
	"cwundercurvearrow",
	NULL,
	NULL,
	NULL,
	"rightarrowshortleftarrow",
	"leftarrowshortrightarrow",
	"shortrightarrowleftarrow",
	"rightarrowplus",
	"leftarrowplus",
	"rightarrowx",
	"leftrightarrowcircle",
	"twoheaduparrowcircle",
	"leftrightharpoonupdown",
	"leftrightharpoondownup",
	"updownharpoonrightleft",
	"updownharpoonleftright",
	"leftrightharpoonupup",
	"updownharpoonrightright",
	"leftrightharpoondowndown",
	"updownharpoonleftleft",
	"barleftharpoonup",
	"rightharpoonupbar",
	"barupharpoonright",
	"downharpoonrightbar",
	"barleftharpoondown",
	"rightharpoondownbar",
	"barupharpoonleft",
	"downharpoonleftbar",
	"leftharpoonupbar",
	"barrightharpoonup",
	"upharpoonrightbar",
	"bardownharpoonright",
	"leftharpoondownbar",
	"barrightharpoondown",
	"upharpoonleftbar",
	"bardownharpoonleft",
	"leftharpoonsupdown",
	"upharpoonsleftright",
	"rightharpoonsupdown",
	"downharpoonsleftright",
	"leftrightharpoonsup",
	"leftrightharpoonsdown",
	"rightleftharpoonsup",
	"rightleftharpoonsdown",
	"leftharpoonupdash",
	"dashleftharpoondown",
	"rightharpoonupdash",
	"dashrightharpoondown",
	"updownharpoonsleftright",
	"downupharpoonsleftright",
	"rightimply",
	"equalrightarrow",
	"similarrightarrow",
	"leftarrowsimilar",
	"rightarrowsimilar",
	"rightarrowapprox",
	"ltlarr",
	"leftarrowless",
	"gtrarr",
	"subrarr",
	"leftarrowsubset",
	"suplarr",
	"leftfishtail",
	"rightfishtail",
	"upfishtail",
	"downfishtail",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rtriltri",
	"ltrivb",
	"vbrtri",
	"lfbowtie",
	"rfbowtie",
	"fbowtie",
	"lftimes",
	"rftimes",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"eqvparsl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"ruledelayed",
	NULL,
	"dsol",
	"rsolbar",
	"xsol",
	"xbsol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b2a[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congdot",
	NULL,
	NULL,
	"approxeqq",
	NULL,
	NULL,
	"eqqsim",
	"Coloneq",
	"eqeq",
	NULL,
	NULL,
	"equivDD",
	"ltcir",
	"gtcir",
	"ltquest",
	"gtquest",
	NULL,
	NULL,
	"lesdot",
	"gesdot",
	"lesdoto",
	"gesdoto",
	"lesdotor",
	"gesdotol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lsime",
	"gsime",
	"lsimg",
	"gsiml",
	"lgE",
	"glE",
	"lesges",
	"gesles",
	NULL,
	NULL,
	"elsdot",
	"egsdot",
	NULL,
	NULL,
	NULL,
	NULL,
	"simless",
	"simgtr",
	"simlE",
	"simgE",
	"Lt",
	"Gt",
	NULL,
	"glj",
	"gla",
	"ltcc",
	"gtcc",
	"lescc",
	"gescc",
	"smt",
	"lat",
	"smte",
	"late",
	"bumpeqq",
	NULL,
	NULL,
	NULL,
	NULL,
	"prE",
	"scE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Pr",
	"Sc",
	"subsetdot",
	"supsetdot",
	"subsetplus",
	"supsetplus",
	"submult",
	"supmult",
	"subedot",
	"supedot",
	NULL,
	NULL,
	"subsim",
	"supsim",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"csub",
	"csup",
	"csube",
	"csupe",
	"subsup",
	"supsub",
	"subsub",
	"supsup",
	"suphsub",
	"supdsub",
	"forkv",
	"topfork",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Barv",
	"vBar",
	"vBarv",
	"barV",
	"Vbar",
	"Not",
	"bNot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rightdbkarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"nrightwavyarrow",
	NULL,
	"nrightcurvedarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"race",
	"acE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be3[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lesg",
	"gesl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"smtes",
	"lates",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"bsolhsub",
	"suphsol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congruence",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"mostpos",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"varrightfishtail",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **tex_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tex_p0_b20,
	tex_p0_b21,
	tex_p0_b22,
	tex_p0_b23,
	NULL,
	NULL,
	NULL,
	tex_p0_b27,
	NULL,
	tex_p0_b29,
	tex_p0_b2a,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tex_p0_be2,
	tex_p0_be3,
	tex_p0_be4,
	NULL,
	tex_p0_be6,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList tex = {
	&agl,
	NU_("ΤεΧ Names"),
	{ tex_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static const char *ams_p0_be2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni219D0338",
	NULL,
	"uni29330338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2192.short",
	"uni2190.short",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni222AFE00",
	"uni2229FE00",
	"uni2294FE00",
	"uni2293FE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E0332.reversed",
	"uni223E0333",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2269FE00",
	NULL,
	NULL,
	"uni2268FE00",
	"uni22670338",
	"uni2A7E0338",
	"uni2A7D0338",
	"uni22660338",
	NULL,
	"uni2224.short",
	"uni2226.short",
	"uni228220D2",
	"uni228620D2",
	"uni2AC50338",
	"uni228320D2",
	"uni2AC60338",
	"uni228720D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2ACBFE00",
	"uni228AFE00",
	"uni228BFE00",
	"uni2ACCFE00",
	"uni224B0338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni226420D2",
	"uni226520D2",
	"uni003C20D2",
	"uni003E20D2",
	"uni2A6D0338",
	"uni224820D2",
	"uni2A700338",
	NULL,
	"uni2AA120D2",
	"uni2AA220D2",
	"uni226A0338",
	"uni226B0338",
	"uni22D80338",
	"uni22D90338",
	"uni22B520D2",
	"uni22B420D2",
	NULL,
	NULL,
	NULL,
	"uni006A.dotless",
	"uni210F.var",
	NULL,
	NULL,
	"uni222020D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be3[] = {
	NULL,
	"uni2223.short",
	"uni2225.short",
	NULL,
	NULL,
	NULL,
	"uni2248.bold",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22DAFE00",
	"uni22DBFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AACFE00",
	"uni2AADFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220B20D2",
	"uni220D0338",
	NULL,
	NULL,
	"uni220820D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A15.up",
	"uni2A16.up",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni226120E5",
	"uni003D20E5",
	"uni2AFD20E5",
	"uni22500338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22020338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A10.up",
	NULL,
	"uni2A12.up",
	"uni2A13.up",
	"uni2A14.up",
	"uni2A17.up",
	"uni2A11.up",
	NULL,
	"uni22F50338",
	NULL,
	"uni22F90338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2026.em",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A0338",
	"uni227320D2",
	"uni227220D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E.reversed",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni0265.superscript",
	"uni0252.superscript",
	NULL,
	NULL,
	NULL,
	"uni2A0F.up",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A9B20D2",
	"uni2A9C20D2",
	"uni2A9B0338",
	"uni2A9C0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223C20D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A3CFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AAF0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224220D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AB00338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	"uni228F0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22900338",
	"uni224E0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2205.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni0077.subscript",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni1D0B.reversed",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni025102DE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni025B02DE",
	"uni025402DE",
	NULL,
	"uni03C9.inverted",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni297D.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223C.bold",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be8[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni227E0338",
	"uni227F0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2216.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224F0338",
	"uni22420338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bea[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni00730336",
	NULL,
	"uni006C0321",
	NULL,
	NULL,
	"uni03040304",
	NULL,
	NULL,
	"uni229CFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03020302",
	NULL,
	"uni03030303",
	"uni033103310331",
	"uni0331033103310331",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03070302",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03040303",
	"uni03310330",
	"uni03040308",
	"uni03300331",
	NULL,
	"uni00640321",
	"uni00680321",
	"uni006B0321",
	"uni00780321",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni00730321",
	"uni007A0321",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2034.notsup",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A0B.up",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni227620D2",
	"uni227720D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bec[] = {
	NULL,
	NULL,
	"uni2A3DFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bed[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A20D2",
	"uni220D20D2",
	"uni22F60338",
	"uni22FD0338",
	"uni22F620D2",
	"uni22FD20D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2295FE00",
	"uni2297FE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224320D2",
	"uni224520D2",
	NULL,
	"uni2A6C0338",
	"uni2A6C20D2",
	"uni2A7020D2",
	"uni224D20D2",
	"uni223F.reversed",
	"uni003D20D2",
	NULL,
	"uni226120D2",
	"uni22630338",
	"uni226320D2",
	NULL,
	NULL,
	"uni2A7D20D2",
	"uni2A7E20D2",
	"uni226620D2",
	"uni226720D2",
	"uni2A950338",
	"uni2A960338",
	"uni2A9520D2",
	"uni2A9620D2",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A990338",
	"uni2A9A0338",
	"uni2A9920D2",
	"uni2A9A20D2",
	"uni2272FE00",
	"uni2273FE00",
	"uni2A9DFE00",
	"uni2A9EFE00",
	NULL,
	NULL,
	"uni227A20D2",
	"uni227B20D2",
	"uni227C20D2",
	"uni227D20D2",
	"uni22DE20D2",
	"uni22DF20D2",
	"uni22DE0338",
	"uni22DF0338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AC520D2",
	"uni2AC620D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AF40338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A0A.lgdisplay",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A.narrow",
	"uni220D.narrow",
	NULL,
	"u1D6FB.narrow",
	"uni002820090029",
	"uni002822C50029",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bee[] = {
	"stixEE00",
	"stixEE01",
	"stixEE02",
	"stixEE03",
	"stixEE04",
	"stixEE05",
	"stixEE06",
	"stixEE07",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE10",
	"stixEE11",
	"stixEE12",
	"stixEE13",
	"stixEE14",
	"stixEE15",
	"stixEE16",
	"stixEE17",
	"stixEE18",
	"stixEE19",
	"stixEE1A",
	"stixEE1B",
	"stixEE1C",
	"stixEE1D",
	"stixEE1E",
	"stixEE1F",
	"stixEE20",
	"stixEE21",
	"stixEE22",
	"stixEE23",
	"stixEE24",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE30",
	"stixEE31",
	"stixEE32",
	"stixEE33",
	"stixEE34",
	"stixEE35",
	"stixEE36",
	"stixEE37",
	"stixEE38",
	"stixEE39",
	"stixEE3A",
	"stixEE3B",
	"stixEE3C",
	"stixEE3D",
	"stixEE3E",
	NULL,
	"stixEE40",
	"stixEE41",
	"stixEE42",
	"stixEE43",
	"stixEE44",
	"stixEE45",
	"stixEE46",
	"stixEE47",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE50",
	"stixEE51",
	"stixEE52",
	"stixEE53",
	"stixEE54",
	"stixEE55",
	"stixEE56",
	"stixEE57",
	"stixEE58",
	"stixEE59",
	"stixEE5A",
	"stixEE5B",
	"stixEE5C",
	"stixEE5D",
	"stixEE5E",
	"stixEE5F",
	"stixEE60",
	"stixEE61",
	"stixEE62",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE70",
	"stixEE71",
	"stixEE72",
	"stixEE73",
	"stixEE74",
	"stixEE75",
	"stixEE76",
	"stixEE77",
	"stixEE78",
	"stixEE79",
	"stixEE7A",
	"stixEE7B",
	"stixEE7C",
	"stixEE7D",
	"stixEE7E",
	"stixEE7F",
	"stixEE80",
	"stixEE81",
	"stixEE82",
	"stixEE83",
	"stixEE84",
	"stixEE85",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE90",
	"stixEE91",
	"stixEE92",
	"stixEE93",
	"stixEE94",
	"stixEE95",
	"stixEE96",
	"stixEE97",
	"stixEE98",
	"stixEE99",
	"stixEE9A",
	"stixEE9B",
	"stixEE9C",
	"stixEE9D",
	"stixEE9E",
	"stixEE9F",
	"stixEEA0",
	"stixEEA1",
	"stixEEA2",
	"stixEEA3",
	"stixEEA4",
	"stixEEA5",
	"stixEEA6",
	"stixEEA7",
	"stixEEA8",
	"stixEEA9",
	"stixEEAA",
	"stixEEAB",
	"stixEEAC",
	NULL,
	NULL,
	NULL,
	"stixEEB0",
	"stixEEB1",
	"stixEEB2",
	"stixEEB3",
	"stixEEB4",
	"stixEEB5",
	"stixEEB6",
	"stixEEB7",
	"stixEEB8",
	"stixEEB9",
	"stixEEBA",
	"stixEEBB",
	"stixEEBC",
	"stixEEBD",
	"stixEEBE",
	"stixEEBF",
	"stixEEC0",
	"stixEEC1",
	"stixEEC2",
	"stixEEC3",
	"stixEEC4",
	"stixEEC5",
	"stixEEC6",
	"stixEEC7",
	"stixEEC8",
	"stixEEC9",
	"stixEECA",
	"stixEECB",
	NULL,
	"stixEECD",
	"stixEECE",
	"stixEECF",
	"stixEED0",
	"stixEED1",
	"stixEED2",
	"stixEED3",
	"stixEED4",
	"stixEED5",
	"stixEED6",
	"stixEED7",
	"stixEED8",
	"stixEED9",
	"stixEEDA",
	"stixEEDB",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bf4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni29CF0338",
	"uni29D00338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AA10338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AA20338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bf5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2191.short",
	"uni2193.short",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **ams_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	ams_p0_be2,
	ams_p0_be3,
	ams_p0_be4,
	ams_p0_be5,
	ams_p0_be6,
	NULL,
	ams_p0_be8,
	NULL,
	ams_p0_bea,
	NULL,
	ams_p0_bec,
	ams_p0_bed,
	ams_p0_bee,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	ams_p0_bf4,
	ams_p0_bf5,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList ams = {
	&tex,
	N_("AMS Names"),
	{ ams_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	NULL, 0, 0, NULL
};
/* ************************************************************************** */
static struct psaltnames psaltnames[] = {
	{ "AEmacron", 0x01e2 , 0 },
	{ "AEsmall", 0xf7e6 , 0 },
	{ "Aacutesmall", 0xf7e1 , 0 },
	{ "Abreveacute", 0x1eae , 0 },
	{ "Abrevecyrillic", 0x04d0 , 0 },
	{ "Abrevedotbelow", 0x1eb6 , 0 },
	{ "Abrevegrave", 0x1eb0 , 0 },
	{ "Abrevehookabove", 0x1eb2 , 0 },
	{ "Abrevetilde", 0x1eb4 , 0 },
	{ "Acaron", 0x01cd , 0 },
	{ "Acircle", 0x24b6 , 0 },
	{ "Acircumflexacute", 0x1ea4 , 0 },
	{ "Acircumflexdotbelow", 0x1eac , 0 },
	{ "Acircumflexgrave", 0x1ea6 , 0 },
	{ "Acircumflexhookabove", 0x1ea8 , 0 },
	{ "Acircumflexsmall", 0xf7e2 , 0 },
	{ "Acircumflextilde", 0x1eaa , 0 },
	{ "Acute", 0xf6c9 , 0 },
	{ "Acutesmall", 0xf7b4 , 0 },
	{ "Acyrillic", 0x0410 , 0 },
	{ "Adblgrave", 0x0200 , 0 },
	{ "Adieresiscyrillic", 0x04d2 , 0 },
	{ "Adieresismacron", 0x01de , 0 },
	{ "Adieresissmall", 0xf7e4 , 0 },
	{ "Adotbelow", 0x1ea0 , 0 },
	{ "Adotmacron", 0x01e0 , 0 },
	{ "Agravesmall", 0xf7e0 , 0 },
	{ "Ahookabove", 0x1ea2 , 0 },
	{ "Aiecyrillic", 0x04d4 , 0 },
	{ "Ainvertedbreve", 0x0202 , 0 },
	{ "Amonospace", 0xff21 , 0 },
	{ "Aringbelow", 0x1e00 , 0 },
	{ "Aringsmall", 0xf7e5 , 0 },
	{ "Asmall", 0xf761 , 0 },
	{ "Atildesmall", 0xf7e3 , 0 },
	{ "Aybarmenian", 0x0531 , 0 },
	{ "Bcircle", 0x24b7 , 0 },
	{ "Bdotaccent", 0x1e02 , 0 },
	{ "Bdotbelow", 0x1e04 , 0 },
	{ "Becyrillic", 0x0411 , 0 },
	{ "Benarmenian", 0x0532 , 0 },
	{ "Bhook", 0x0181 , 0 },
	{ "Blinebelow", 0x1e06 , 0 },
	{ "Bmonospace", 0xff22 , 0 },
	{ "Brevesmall", 0xf6f4 , 0 },
	{ "Bsmall", 0xf762 , 0 },
	{ "Btopbar", 0x0182 , 0 },
	{ "Caarmenian", 0x053e , 0 },
	{ "Caron", 0xf6ca , 0 },
	{ "Caronsmall", 0xf6f5 , 0 },
	{ "Ccedillaacute", 0x1e08 , 0 },
	{ "Ccedillasmall", 0xf7e7 , 0 },
	{ "Ccircle", 0x24b8 , 0 },
	{ "Cdot", 0x010a , 0 },
	{ "Cedillasmall", 0xf7b8 , 0 },
	{ "Chaarmenian", 0x0549 , 0 },
	{ "Cheabkhasiancyrillic", 0x04bc , 0 },
	{ "Checyrillic", 0x0427 , 0 },
	{ "Chedescenderabkhasiancyrillic", 0x04be , 0 },
	{ "Chedescendercyrillic", 0x04b6 , 0 },
	{ "Chedieresiscyrillic", 0x04f4 , 0 },
	{ "Cheharmenian", 0x0543 , 0 },
	{ "Chekhakassiancyrillic", 0x04cb , 0 },
	{ "Cheverticalstrokecyrillic", 0x04b8 , 0 },
	{ "Chook", 0x0187 , 0 },
	{ "Circumflexsmall", 0xf6f6 , 0 },
	{ "Cmonospace", 0xff23 , 0 },
	{ "Coarmenian", 0x0551 , 0 },
	{ "Csmall", 0xf763 , 0 },
	{ "DZ", 0x01f1 , 0 },
	{ "DZcaron", 0x01c4 , 0 },
	{ "Daarmenian", 0x0534 , 0 },
	{ "Dafrican", 0x0189 , 0 },
	{ "Dcedilla", 0x1e10 , 0 },
	{ "Dcircle", 0x24b9 , 0 },
	{ "Dcircumflexbelow", 0x1e12 , 0 },
	{ "Ddotaccent", 0x1e0a , 0 },
	{ "Ddotbelow", 0x1e0c , 0 },
	{ "Decyrillic", 0x0414 , 0 },
	{ "Deicoptic", 0x03ee , 0 },
	{ "Deltagreek", 0x0394 , 0 },
	{ "Dhook", 0x018a , 0 },
	{ "Dieresis", 0xf6cb , 0 },
	{ "diaeresis", 0x00a8 , 0 },
	{ "Diaeresis", 0xf6cb , 0 },
	{ "DieresisAcute", 0xf6cc , 0 },
	{ "DieresisGrave", 0xf6cd , 0 },
	{ "Dieresissmall", 0xf7a8 , 0 },
	{ "Digammagreek", 0x03dc , 0 },
	{ "Djecyrillic", 0x0402 , 0 },
	{ "Dlinebelow", 0x1e0e , 0 },
	{ "Dmonospace", 0xff24 , 0 },
	{ "Dotaccentsmall", 0xf6f7 , 0 },
	{ "Dslash", 0x0110 , 0 },
	{ "Dsmall", 0xf764 , 0 },
	{ "Dtopbar", 0x018b , 0 },
	{ "Dz", 0x01f2 , 0 },
	{ "Dzcaron", 0x01c5 , 0 },
	{ "Dzeabkhasiancyrillic", 0x04e0 , 0 },
	{ "Dzecyrillic", 0x0405 , 0 },
	{ "Dzhecyrillic", 0x040f , 0 },
	{ "Eacutesmall", 0xf7e9 , 0 },
	{ "Ecedillabreve", 0x1e1c , 0 },
	{ "Echarmenian", 0x0535 , 0 },
	{ "Ecircle", 0x24ba , 0 },
	{ "Ecircumflexacute", 0x1ebe , 0 },
	{ "Ecircumflexbelow", 0x1e18 , 0 },
	{ "Ecircumflexdotbelow", 0x1ec6 , 0 },
	{ "Ecircumflexgrave", 0x1ec0 , 0 },
	{ "Ecircumflexhookabove", 0x1ec2 , 0 },
	{ "Ecircumflexsmall", 0xf7ea , 0 },
	{ "Ecircumflextilde", 0x1ec4 , 0 },
	{ "Ecyrillic", 0x0404 , 0 },
	{ "Edblgrave", 0x0204 , 0 },
	{ "Edieresissmall", 0xf7eb , 0 },
	{ "Edot", 0x0116 , 0 },
	{ "Edotbelow", 0x1eb8 , 0 },
	{ "Efcyrillic", 0x0424 , 0 },
	{ "Egravesmall", 0xf7e8 , 0 },
	{ "Eharmenian", 0x0537 , 0 },
	{ "Ehookabove", 0x1eba , 0 },
	{ "Eightroman", 0x2167 , 0 },
	{ "Einvertedbreve", 0x0206 , 0 },
	{ "Eiotifiedcyrillic", 0x0464 , 0 },
	{ "Elcyrillic", 0x041b , 0 },
	{ "Elevenroman", 0x216a , 0 },
	{ "Emacronacute", 0x1e16 , 0 },
	{ "Emacrongrave", 0x1e14 , 0 },
	{ "Emcyrillic", 0x041c , 0 },
	{ "Emonospace", 0xff25 , 0 },
	{ "Encyrillic", 0x041d , 0 },
	{ "Endescendercyrillic", 0x04a2 , 0 },
	{ "Enghecyrillic", 0x04a4 , 0 },
	{ "Enhookcyrillic", 0x04c7 , 0 },
	{ "Eopen", 0x0190 , 0 },
	{ "Ercyrillic", 0x0420 , 0 },
	{ "Ereversed", 0x018e , 0 },
	{ "Ereversedcyrillic", 0x042d , 0 },
	{ "Escyrillic", 0x0421 , 0 },
	{ "Esdescendercyrillic", 0x04aa , 0 },
	{ "Esh", 0x01a9 , 0 },
	{ "Esmall", 0xf765 , 0 },
	{ "Etarmenian", 0x0538 , 0 },
	{ "Ethsmall", 0xf7f0 , 0 },
	{ "Etilde", 0x1ebc , 0 },
	{ "Etildebelow", 0x1e1a , 0 },
	{ "Ezh", 0x01b7 , 0 },
	{ "Ezhcaron", 0x01ee , 0 },
	{ "Ezhreversed", 0x01b8 , 0 },
	{ "Fcircle", 0x24bb , 0 },
	{ "Fdotaccent", 0x1e1e , 0 },
	{ "Feharmenian", 0x0556 , 0 },
	{ "Feicoptic", 0x03e4 , 0 },
	{ "Fhook", 0x0191 , 0 },
	{ "Fitacyrillic", 0x0472 , 0 },
	{ "Fiveroman", 0x2164 , 0 },
	{ "Fmonospace", 0xff26 , 0 },
	{ "Fourroman", 0x2163 , 0 },
	{ "Fsmall", 0xf766 , 0 },
	{ "GBsquare", 0x3387 , 0 },
	{ "Gacute", 0x01f4 , 0 },
	{ "Gammaafrican", 0x0194 , 0 },
	{ "Gangiacoptic", 0x03ea , 0 },
	{ "Gcedilla", 0x0122 , 0 },
	{ "Gcircle", 0x24bc , 0 },
	{ "Gdot", 0x0120 , 0 },
	{ "Gecyrillic", 0x0413 , 0 },
	{ "Ghadarmenian", 0x0542 , 0 },
	{ "Ghemiddlehookcyrillic", 0x0494 , 0 },
	{ "Ghestrokecyrillic", 0x0492 , 0 },
	{ "Gheupturncyrillic", 0x0490 , 0 },
	{ "Ghook", 0x0193 , 0 },
	{ "Gimarmenian", 0x0533 , 0 },
	{ "Gjecyrillic", 0x0403 , 0 },
	{ "Gmacron", 0x1e20 , 0 },
	{ "Gmonospace", 0xff27 , 0 },
	{ "Grave", 0xf6ce , 0 },
	{ "Gravesmall", 0xf760 , 0 },
	{ "Gsmall", 0xf767 , 0 },
	{ "Gsmallhook", 0x029b , 0 },
	{ "Gstroke", 0x01e4 , 0 },
	{ "HPsquare", 0x33cb , 0 },
	{ "Haabkhasiancyrillic", 0x04a8 , 0 },
	{ "Hadescendercyrillic", 0x04b2 , 0 },
	{ "Hardsigncyrillic", 0x042a , 0 },
	{ "Hbrevebelow", 0x1e2a , 0 },
	{ "Hcedilla", 0x1e28 , 0 },
	{ "Hcircle", 0x24bd , 0 },
	{ "Hdieresis", 0x1e26 , 0 },
	{ "Hdotaccent", 0x1e22 , 0 },
	{ "Hdotbelow", 0x1e24 , 0 },
	{ "Hmonospace", 0xff28 , 0 },
	{ "Hoarmenian", 0x0540 , 0 },
	{ "Horicoptic", 0x03e8 , 0 },
	{ "Hsmall", 0xf768 , 0 },
	{ "Hungarumlaut", 0xf6cf , 0 },
	{ "Hungarumlautsmall", 0xf6f8 , 0 },
	{ "Hzsquare", 0x3390 , 0 },
	{ "IAcyrillic", 0x042f , 0 },
	{ "IUcyrillic", 0x042e , 0 },
	{ "Iacutesmall", 0xf7ed , 0 },
	{ "Icaron", 0x01cf , 0 },
	{ "Icircle", 0x24be , 0 },
	{ "Icircumflexsmall", 0xf7ee , 0 },
	{ "Icyrillic", 0x0406 , 0 },
	{ "Idblgrave", 0x0208 , 0 },
	{ "Idieresisacute", 0x1e2e , 0 },
	{ "Idieresiscyrillic", 0x04e4 , 0 },
	{ "Idieresissmall", 0xf7ef , 0 },
	{ "Idot", 0x0130 , 0 },
	{ "Idotbelow", 0x1eca , 0 },
	{ "Iebrevecyrillic", 0x04d6 , 0 },
	{ "Iecyrillic", 0x0415 , 0 },
	{ "Igravesmall", 0xf7ec , 0 },
	{ "Ihookabove", 0x1ec8 , 0 },
	{ "Iicyrillic", 0x0418 , 0 },
	{ "Iinvertedbreve", 0x020a , 0 },
	{ "Iishortcyrillic", 0x0419 , 0 },
	{ "Imacroncyrillic", 0x04e2 , 0 },
	{ "Imonospace", 0xff29 , 0 },
	{ "Iniarmenian", 0x053b , 0 },
	{ "Iocyrillic", 0x0401 , 0 },
	{ "Iotaafrican", 0x0196 , 0 },
	{ "Ismall", 0xf769 , 0 },
	{ "Istroke", 0x0197 , 0 },
	{ "Itildebelow", 0x1e2c , 0 },
	{ "Izhitsacyrillic", 0x0474 , 0 },
	{ "Izhitsadblgravecyrillic", 0x0476 , 0 },
	{ "Jaarmenian", 0x0541 , 0 },
	{ "Jcircle", 0x24bf , 0 },
	{ "Jecyrillic", 0x0408 , 0 },
	{ "Jheharmenian", 0x054b , 0 },
	{ "Jmonospace", 0xff2a , 0 },
	{ "Jsmall", 0xf76a , 0 },
	{ "KBsquare", 0x3385 , 0 },
	{ "KKsquare", 0x33cd , 0 },
	{ "Kabashkircyrillic", 0x04a0 , 0 },
	{ "Kacute", 0x1e30 , 0 },
	{ "Kacyrillic", 0x041a , 0 },
	{ "Kadescendercyrillic", 0x049a , 0 },
	{ "Kahookcyrillic", 0x04c3 , 0 },
	{ "Kastrokecyrillic", 0x049e , 0 },
	{ "Kaverticalstrokecyrillic", 0x049c , 0 },
	{ "Kcaron", 0x01e8 , 0 },
	{ "Kcedilla", 0x0136 , 0 },
	{ "Kcircle", 0x24c0 , 0 },
	{ "Kdotbelow", 0x1e32 , 0 },
	{ "Keharmenian", 0x0554 , 0 },
	{ "Kenarmenian", 0x053f , 0 },
	{ "Khacyrillic", 0x0425 , 0 },
	{ "Kheicoptic", 0x03e6 , 0 },
	{ "Khook", 0x0198 , 0 },
	{ "Kjecyrillic", 0x040c , 0 },
	{ "Klinebelow", 0x1e34 , 0 },
	{ "Kmonospace", 0xff2b , 0 },
	{ "Koppacyrillic", 0x0480 , 0 },
	{ "Koppagreek", 0x03de , 0 },
	{ "Ksicyrillic", 0x046e , 0 },
	{ "Ksmall", 0xf76b , 0 },
	{ "LJ", 0x01c7 , 0 },
	{ "LL", 0xf6bf , 0 },
	{ "Lcedilla", 0x013b , 0 },
	{ "Lcircle", 0x24c1 , 0 },
	{ "Lcircumflexbelow", 0x1e3c , 0 },
	{ "Ldotaccent", 0x013f , 0 },
	{ "Ldotbelow", 0x1e36 , 0 },
	{ "Ldotbelowmacron", 0x1e38 , 0 },
	{ "Liwnarmenian", 0x053c , 0 },
	{ "Lj", 0x01c8 , 0 },
	{ "Ljecyrillic", 0x0409 , 0 },
	{ "Llinebelow", 0x1e3a , 0 },
	{ "Lmonospace", 0xff2c , 0 },
	{ "Lslashsmall", 0xf6f9 , 0 },
	{ "Lsmall", 0xf76c , 0 },
	{ "MBsquare", 0x3386 , 0 },
	{ "Macron", 0xf6d0 , 0 },
	{ "Macronsmall", 0xf7af , 0 },
	{ "Macute", 0x1e3e , 0 },
	{ "Mcircle", 0x24c2 , 0 },
	{ "Mdotaccent", 0x1e40 , 0 },
	{ "Mdotbelow", 0x1e42 , 0 },
	{ "Menarmenian", 0x0544 , 0 },
	{ "Mmonospace", 0xff2d , 0 },
	{ "Msmall", 0xf76d , 0 },
	{ "Mturned", 0x019c , 0 },
	{ "NJ", 0x01ca , 0 },
	{ "Ncedilla", 0x0145 , 0 },
	{ "Ncircle", 0x24c3 , 0 },
	{ "Ncircumflexbelow", 0x1e4a , 0 },
	{ "Ndotaccent", 0x1e44 , 0 },
	{ "Ndotbelow", 0x1e46 , 0 },
	{ "Nhookleft", 0x019d , 0 },
	{ "Nineroman", 0x2168 , 0 },
	{ "Nj", 0x01cb , 0 },
	{ "Njecyrillic", 0x040a , 0 },
	{ "Nlinebelow", 0x1e48 , 0 },
	{ "Nmonospace", 0xff2e , 0 },
	{ "Nowarmenian", 0x0546 , 0 },
	{ "Nsmall", 0xf76e , 0 },
	{ "Ntildesmall", 0xf7f1 , 0 },
	{ "OEsmall", 0xf6fa , 0 },
	{ "Oacutesmall", 0xf7f3 , 0 },
	{ "Obarredcyrillic", 0x04e8 , 0 },
	{ "Obarreddieresiscyrillic", 0x04ea , 0 },
	{ "Ocaron", 0x01d1 , 0 },
	{ "Ocenteredtilde", 0x019f , 0 },
	{ "Ocircle", 0x24c4 , 0 },
	{ "Ocircumflexacute", 0x1ed0 , 0 },
	{ "Ocircumflexdotbelow", 0x1ed8 , 0 },
	{ "Ocircumflexgrave", 0x1ed2 , 0 },
	{ "Ocircumflexhookabove", 0x1ed4 , 0 },
	{ "Ocircumflexsmall", 0xf7f4 , 0 },
	{ "Ocircumflextilde", 0x1ed6 , 0 },
	{ "Ocyrillic", 0x041e , 0 },
	{ "Odblacute", 0x0150 , 0 },
	{ "Odblgrave", 0x020c , 0 },
	{ "Odieresiscyrillic", 0x04e6 , 0 },
	{ "Odieresissmall", 0xf7f6 , 0 },
	{ "Odotbelow", 0x1ecc , 0 },
	{ "Ogoneksmall", 0xf6fb , 0 },
	{ "Ogravesmall", 0xf7f2 , 0 },
	{ "Oharmenian", 0x0555 , 0 },
	{ "Ohm", 0x2126 , 0 },
	{ "Ohookabove", 0x1ece , 0 },
	{ "Ohornacute", 0x1eda , 0 },
	{ "Ohorndotbelow", 0x1ee2 , 0 },
	{ "Ohorngrave", 0x1edc , 0 },
	{ "Ohornhookabove", 0x1ede , 0 },
	{ "Ohorntilde", 0x1ee0 , 0 },
	{ "Oi", 0x01a2 , 0 },
	{ "Oinvertedbreve", 0x020e , 0 },
	{ "Omacronacute", 0x1e52 , 0 },
	{ "Omacrongrave", 0x1e50 , 0 },
	{ "Omegacyrillic", 0x0460 , 0 },
	{ "Omegagreek", 0x03a9 , 0 },
	{ "Omegaroundcyrillic", 0x047a , 0 },
	{ "Omegatitlocyrillic", 0x047c , 0 },
	{ "Omonospace", 0xff2f , 0 },
	{ "Oneroman", 0x2160 , 0 },
	{ "Oogonek", 0x01ea , 0 },
	{ "Oogonekmacron", 0x01ec , 0 },
	{ "Oopen", 0x0186 , 0 },
	{ "Oslashsmall", 0xf7f8 , 0 },
	{ "Osmall", 0xf76f , 0 },
	{ "Ostrokeacute", 0x01fe , 0 },
	{ "Otcyrillic", 0x047e , 0 },
	{ "Otildeacute", 0x1e4c , 0 },
	{ "Otildedieresis", 0x1e4e , 0 },
	{ "Otildesmall", 0xf7f5 , 0 },
	{ "Pacute", 0x1e54 , 0 },
	{ "Pcircle", 0x24c5 , 0 },
	{ "Pdotaccent", 0x1e56 , 0 },
	{ "Pecyrillic", 0x041f , 0 },
	{ "Peharmenian", 0x054a , 0 },
	{ "Pemiddlehookcyrillic", 0x04a6 , 0 },
	{ "Phook", 0x01a4 , 0 },
	{ "Piwrarmenian", 0x0553 , 0 },
	{ "Pmonospace", 0xff30 , 0 },
	{ "Psicyrillic", 0x0470 , 0 },
	{ "Psmall", 0xf770 , 0 },
	{ "Qcircle", 0x24c6 , 0 },
	{ "Qmonospace", 0xff31 , 0 },
	{ "Qsmall", 0xf771 , 0 },
	{ "Raarmenian", 0x054c , 0 },
	{ "Rcedilla", 0x0156 , 0 },
	{ "Rcircle", 0x24c7 , 0 },
	{ "Rdblgrave", 0x0210 , 0 },
	{ "Rdotaccent", 0x1e58 , 0 },
	{ "Rdotbelow", 0x1e5a , 0 },
	{ "Rdotbelowmacron", 0x1e5c , 0 },
	{ "Reharmenian", 0x0550 , 0 },
	{ "Ringsmall", 0xf6fc , 0 },
	{ "Rinvertedbreve", 0x0212 , 0 },
	{ "Rlinebelow", 0x1e5e , 0 },
	{ "Rmonospace", 0xff32 , 0 },
	{ "Rsmall", 0xf772 , 0 },
	{ "Rsmallinverted", 0x0281 , 0 },
	{ "Rsmallinvertedsuperior", 0x02b6 , 0 },
	{ "Sacutedotaccent", 0x1e64 , 0 },
	{ "Sampigreek", 0x03e0 , 0 },
	{ "Scarondotaccent", 0x1e66 , 0 },
	{ "Scaronsmall", 0xf6fd , 0 },
	{ "Schwa", 0x018f , 0 },
	{ "Schwacyrillic", 0x04d8 , 0 },
	{ "Schwadieresiscyrillic", 0x04da , 0 },
	{ "Scircle", 0x24c8 , 0 },
	{ "Sdotaccent", 0x1e60 , 0 },
	{ "Sdotbelow", 0x1e62 , 0 },
	{ "Sdotbelowdotaccent", 0x1e68 , 0 },
	{ "Seharmenian", 0x054d , 0 },
	{ "Sevenroman", 0x2166 , 0 },
	{ "Shaarmenian", 0x0547 , 0 },
	{ "Shacyrillic", 0x0428 , 0 },
	{ "Shchacyrillic", 0x0429 , 0 },
	{ "Sheicoptic", 0x03e2 , 0 },
	{ "Shhacyrillic", 0x04ba , 0 },
	{ "Shimacoptic", 0x03ec , 0 },
	{ "Sixroman", 0x2165 , 0 },
	{ "Smonospace", 0xff33 , 0 },
	{ "Softsigncyrillic", 0x042c , 0 },
	{ "Ssmall", 0xf773 , 0 },
	{ "Stigmagreek", 0x03da , 0 },
	{ "Tcedilla", 0x0162 , 0 },
	{ "Tcircle", 0x24c9 , 0 },
	{ "Tcircumflexbelow", 0x1e70 , 0 },
	{ "Tdotaccent", 0x1e6a , 0 },
	{ "Tdotbelow", 0x1e6c , 0 },
	{ "Tecyrillic", 0x0422 , 0 },
	{ "Tedescendercyrillic", 0x04ac , 0 },
	{ "Tenroman", 0x2169 , 0 },
	{ "Tetsecyrillic", 0x04b4 , 0 },
	{ "Thook", 0x01ac , 0 },
	{ "Thornsmall", 0xf7fe , 0 },
	{ "Threeroman", 0x2162 , 0 },
	{ "Tildesmall", 0xf6fe , 0 },
	{ "Tiwnarmenian", 0x054f , 0 },
	{ "Tlinebelow", 0x1e6e , 0 },
	{ "Tmonospace", 0xff34 , 0 },
	{ "Toarmenian", 0x0539 , 0 },
	{ "Tonefive", 0x01bc , 0 },
	{ "Tonesix", 0x0184 , 0 },
	{ "Tonetwo", 0x01a7 , 0 },
	{ "Tretroflexhook", 0x01ae , 0 },
	{ "Tsecyrillic", 0x0426 , 0 },
	{ "Tshecyrillic", 0x040b , 0 },
	{ "Tsmall", 0xf774 , 0 },
	{ "Twelveroman", 0x216b , 0 },
	{ "Tworoman", 0x2161 , 0 },
	{ "Uacutesmall", 0xf7fa , 0 },
	{ "Ucaron", 0x01d3 , 0 },
	{ "Ucircle", 0x24ca , 0 },
	{ "Ucircumflexbelow", 0x1e76 , 0 },
	{ "Ucircumflexsmall", 0xf7fb , 0 },
	{ "Ucyrillic", 0x0423 , 0 },
	{ "Udblacute", 0x0170 , 0 },
	{ "Udblgrave", 0x0214 , 0 },
	{ "Udieresisacute", 0x01d7 , 0 },
	{ "Udieresisbelow", 0x1e72 , 0 },
	{ "Udieresiscaron", 0x01d9 , 0 },
	{ "Udieresiscyrillic", 0x04f0 , 0 },
	{ "Udieresisgrave", 0x01db , 0 },
	{ "Udieresismacron", 0x01d5 , 0 },
	{ "Udieresissmall", 0xf7fc , 0 },
	{ "Udotbelow", 0x1ee4 , 0 },
	{ "Ugravesmall", 0xf7f9 , 0 },
	{ "Uhookabove", 0x1ee6 , 0 },
	{ "Uhornacute", 0x1ee8 , 0 },
	{ "Uhorndotbelow", 0x1ef0 , 0 },
	{ "Uhorngrave", 0x1eea , 0 },
	{ "Uhornhookabove", 0x1eec , 0 },
	{ "Uhorntilde", 0x1eee , 0 },
	{ "Uhungarumlautcyrillic", 0x04f2 , 0 },
	{ "Uinvertedbreve", 0x0216 , 0 },
	{ "Ukcyrillic", 0x0478 , 0 },
	{ "Umacroncyrillic", 0x04ee , 0 },
	{ "Umacrondieresis", 0x1e7a , 0 },
	{ "Umonospace", 0xff35 , 0 },
	{ "Upsilonacutehooksymbolgreek", 0x03d3 , 0 },
	{ "Upsilonafrican", 0x01b1 , 0 },
	{ "Upsilondieresishooksymbolgreek", 0x03d4 , 0 },
	{ "Upsilonhooksymbol", 0x03d2 , 0 },
	{ "Ushortcyrillic", 0x040e , 0 },
	{ "Usmall", 0xf775 , 0 },
	{ "Ustraightcyrillic", 0x04ae , 0 },
	{ "Ustraightstrokecyrillic", 0x04b0 , 0 },
	{ "Utildeacute", 0x1e78 , 0 },
	{ "Utildebelow", 0x1e74 , 0 },
	{ "Vcircle", 0x24cb , 0 },
	{ "Vdotbelow", 0x1e7e , 0 },
	{ "Vecyrillic", 0x0412 , 0 },
	{ "Vewarmenian", 0x054e , 0 },
	{ "Vhook", 0x01b2 , 0 },
	{ "Vmonospace", 0xff36 , 0 },
	{ "Voarmenian", 0x0548 , 0 },
	{ "Vsmall", 0xf776 , 0 },
	{ "Vtilde", 0x1e7c , 0 },
	{ "Wcircle", 0x24cc , 0 },
	{ "Wdotaccent", 0x1e86 , 0 },
	{ "Wdotbelow", 0x1e88 , 0 },
	{ "Wmonospace", 0xff37 , 0 },
	{ "Wsmall", 0xf777 , 0 },
	{ "Xcircle", 0x24cd , 0 },
	{ "Xdieresis", 0x1e8c , 0 },
	{ "Xdotaccent", 0x1e8a , 0 },
	{ "Xeharmenian", 0x053d , 0 },
	{ "Xmonospace", 0xff38 , 0 },
	{ "Xsmall", 0xf778 , 0 },
	{ "Yacutesmall", 0xf7fd , 0 },
	{ "Yatcyrillic", 0x0462 , 0 },
	{ "Ycircle", 0x24ce , 0 },
	{ "Ydieresissmall", 0xf7ff , 0 },
	{ "Ydotaccent", 0x1e8e , 0 },
	{ "Ydotbelow", 0x1ef4 , 0 },
	{ "Yericyrillic", 0x042b , 0 },
	{ "Yerudieresiscyrillic", 0x04f8 , 0 },
	{ "Yhook", 0x01b3 , 0 },
	{ "Yhookabove", 0x1ef6 , 0 },
	{ "Yiarmenian", 0x0545 , 0 },
	{ "Yicyrillic", 0x0407 , 0 },
	{ "Yiwnarmenian", 0x0552 , 0 },
	{ "Ymonospace", 0xff39 , 0 },
	{ "Ysmall", 0xf779 , 0 },
	{ "Ytilde", 0x1ef8 , 0 },
	{ "Yusbigcyrillic", 0x046a , 0 },
	{ "Yusbigiotifiedcyrillic", 0x046c , 0 },
	{ "Yuslittlecyrillic", 0x0466 , 0 },
	{ "Yuslittleiotifiedcyrillic", 0x0468 , 0 },
	{ "Zaarmenian", 0x0536 , 0 },
	{ "Zcaronsmall", 0xf6ff , 0 },
	{ "Zcircle", 0x24cf , 0 },
	{ "Zcircumflex", 0x1e90 , 0 },
	{ "Zdot", 0x017b , 0 },
	{ "Zdotbelow", 0x1e92 , 0 },
	{ "Zecyrillic", 0x0417 , 0 },
	{ "Zedescendercyrillic", 0x0498 , 0 },
	{ "Zedieresiscyrillic", 0x04de , 0 },
	{ "Zhearmenian", 0x053a , 0 },
	{ "Zhebrevecyrillic", 0x04c1 , 0 },
	{ "Zhecyrillic", 0x0416 , 0 },
	{ "Zhedescendercyrillic", 0x0496 , 0 },
	{ "Zhedieresiscyrillic", 0x04dc , 0 },
	{ "Zlinebelow", 0x1e94 , 0 },
	{ "Zmonospace", 0xff3a , 0 },
	{ "Zsmall", 0xf77a , 0 },
	{ "Zstroke", 0x01b5 , 0 },
	{ "aabengali", 0x0986 , 0 },
	{ "aadeva", 0x0906 , 0 },
	{ "aagujarati", 0x0a86 , 0 },
	{ "aagurmukhi", 0x0a06 , 0 },
	{ "aamatragurmukhi", 0x0a3e , 0 },
	{ "aarusquare", 0x3303 , 0 },
	{ "aavowelsignbengali", 0x09be , 0 },
	{ "aavowelsigndeva", 0x093e , 0 },
	{ "aavowelsigngujarati", 0x0abe , 0 },
	{ "abbreviationmarkarmenian", 0x055f , 0 },
	{ "abbreviationsigndeva", 0x0970 , 0 },
	{ "abengali", 0x0985 , 0 },
	{ "abopomofo", 0x311a , 0 },
	{ "abreveacute", 0x1eaf , 0 },
	{ "abrevecyrillic", 0x04d1 , 0 },
	{ "abrevedotbelow", 0x1eb7 , 0 },
	{ "abrevegrave", 0x1eb1 , 0 },
	{ "abrevehookabove", 0x1eb3 , 0 },
	{ "abrevetilde", 0x1eb5 , 0 },
	{ "acaron", 0x01ce , 0 },
	{ "acircle", 0x24d0 , 0 },
	{ "acircumflexacute", 0x1ea5 , 0 },
	{ "acircumflexdotbelow", 0x1ead , 0 },
	{ "acircumflexgrave", 0x1ea7 , 0 },
	{ "acircumflexhookabove", 0x1ea9 , 0 },
	{ "acircumflextilde", 0x1eab , 0 },
	{ "acutebelowcmb", 0x0317 , 0 },
	{ "acutecmb", 0x0301 , 0 },
	{ "acutedeva", 0x0954 , 0 },
	{ "acutelowmod", 0x02cf , 0 },
	{ "acutetonecmb", 0x0341 , 0 },
	{ "acyrillic", 0x0430 , 0 },
	{ "adblgrave", 0x0201 , 0 },
	{ "addakgurmukhi", 0x0a71 , 0 },
	{ "adeva", 0x0905 , 0 },
	{ "adieresiscyrillic", 0x04d3 , 0 },
	{ "adieresismacron", 0x01df , 0 },
	{ "adotbelow", 0x1ea1 , 0 },
	{ "adotmacron", 0x01e1 , 0 },
	{ "aekorean", 0x3150 , 0 },
	{ "aemacron", 0x01e3 , 0 },
	{ "afii08941", 0x20a4 , 0 },
	{ "afii10063", 0xf6c4 , 0 },
	{ "afii10064", 0xf6c5 , 0 },
	{ "afii10192", 0xf6c6 , 0 },
	{ "afii10831", 0xf6c7 , 0 },
	{ "afii10832", 0xf6c8 , 0 },
	{ "afii57694", 0xfb2a , 0 },
	{ "afii57695", 0xfb2b , 0 },
	{ "afii57700", 0xfb4b , 0 },
	{ "afii57705", 0xfb1f , 0 },
	{ "afii57723", 0xfb35 , 0 },
	{ "agujarati", 0x0a85 , 0 },
	{ "agurmukhi", 0x0a05 , 0 },
	{ "ahiragana", 0x3042 , 0 },
	{ "ahookabove", 0x1ea3 , 0 },
	{ "aibengali", 0x0990 , 0 },
	{ "aibopomofo", 0x311e , 0 },
	{ "aideva", 0x0910 , 0 },
	{ "aiecyrillic", 0x04d5 , 0 },
	{ "aigujarati", 0x0a90 , 0 },
	{ "aigurmukhi", 0x0a10 , 0 },
	{ "aimatragurmukhi", 0x0a48 , 0 },
	{ "ainarabic", 0x0639 , 0 },
	{ "ainfinalarabic", 0xfeca , 0 },
	{ "aininitialarabic", 0xfecb , 0 },
	{ "ainmedialarabic", 0xfecc , 0 },
	{ "ainvertedbreve", 0x0203 , 0 },
	{ "aivowelsignbengali", 0x09c8 , 0 },
	{ "aivowelsigndeva", 0x0948 , 0 },
	{ "aivowelsigngujarati", 0x0ac8 , 0 },
	{ "akatakana", 0x30a2 , 0 },
	{ "akatakanahalfwidth", 0xff71 , 0 },
	{ "akorean", 0x314f , 0 },
	{ "alef", 0x05d0 , 0 },
	{ "alefarabic", 0x0627 , 0 },
	{ "alefdageshhebrew", 0xfb30 , 0 },
	{ "aleffinalarabic", 0xfe8e , 0 },
	{ "alefhamzaabovearabic", 0x0623 , 0 },
	{ "alefhamzaabovefinalarabic", 0xfe84 , 0 },
	{ "alefhamzabelowarabic", 0x0625 , 0 },
	{ "alefhamzabelowfinalarabic", 0xfe88 , 0 },
	{ "alefhebrew", 0x05d0 , 0 },
	{ "aleflamedhebrew", 0xfb4f , 0 },
	{ "alefmaddaabovearabic", 0x0622 , 0 },
	{ "alefmaddaabovefinalarabic", 0xfe82 , 0 },
	{ "alefmaksuraarabic", 0x0649 , 0 },
	{ "alefmaksurafinalarabic", 0xfef0 , 0 },
	{ "alefmaksurainitialarabic", 0xfef3 , 0 },
	{ "alefmaksuramedialarabic", 0xfef4 , 0 },
	{ "alefpatahhebrew", 0xfb2e , 0 },
	{ "alefqamatshebrew", 0xfb2f , 0 },
	{ "allequal", 0x224c , 0 },
	{ "amonospace", 0xff41 , 0 },
	{ "ampersandmonospace", 0xff06 , 0 },
	{ "ampersandsmall", 0xf726 , 0 },
	{ "amsquare", 0x33c2 , 0 },
	{ "anbopomofo", 0x3122 , 0 },
	{ "angbopomofo", 0x3124 , 0 },
	{ "angkhankhuthai", 0x0e5a , 0 },
	{ "anglebracketleft", 0x3008 , 0 },
	{ "anglebracketleftvertical", 0xfe3f , 0 },
	{ "anglebracketright", 0x3009 , 0 },
	{ "anglebracketrightvertical", 0xfe40 , 0 },
	{ "angstrom", 0x212b , 0 },
	{ "anudattadeva", 0x0952 , 0 },
	{ "anusvarabengali", 0x0982 , 0 },
	{ "anusvaradeva", 0x0902 , 0 },
	{ "anusvaragujarati", 0x0a82 , 0 },
	{ "apaatosquare", 0x3300 , 0 },
	{ "aparen", 0x249c , 0 },
	{ "apostrophearmenian", 0x055a , 0 },
	{ "apostrophemod", 0x02bc , 0 },
	{ "apple", 0xf8ff , 0 },
	{ "approaches", 0x2250 , 0 },
	{ "approxequalorimage", 0x2252 , 0 },
	{ "approximatelyequal", 0x2245 , 0 },
	{ "araeaekorean", 0x318e , 0 },
	{ "araeakorean", 0x318d , 0 },
	{ "arc", 0x2312 , 0 },
	{ "arighthalfring", 0x1e9a , 0 },
	{ "aringbelow", 0x1e01 , 0 },
	{ "arrowdashdown", 0x21e3 , 0 },
	{ "arrowdashleft", 0x21e0 , 0 },
	{ "arrowdashright", 0x21e2 , 0 },
	{ "arrowdashup", 0x21e1 , 0 },
	{ "arrowdownleft", 0x2199 , 0 },
	{ "arrowdownright", 0x2198 , 0 },
	{ "arrowdownwhite", 0x21e9 , 0 },
	{ "arrowheaddownmod", 0x02c5 , 0 },
	{ "arrowheadleftmod", 0x02c2 , 0 },
	{ "arrowheadrightmod", 0x02c3 , 0 },
	{ "arrowheadupmod", 0x02c4 , 0 },
	{ "arrowhorizex", 0xf8e7 , 0 },
	{ "arrowleftdbl", 0x21d0 , 0 },
	{ "arrowleftdblstroke", 0x21cd , 0 },
	{ "arrowleftoverright", 0x21c6 , 0 },
	{ "arrowleftwhite", 0x21e6 , 0 },
	{ "arrowrightdblstroke", 0x21cf , 0 },
	{ "arrowrightheavy", 0x279e , 0 },
	{ "arrowrightoverleft", 0x21c4 , 0 },
	{ "arrowrightwhite", 0x21e8 , 0 },
	{ "arrowtableft", 0x21e4 , 0 },
	{ "arrowtabright", 0x21e5 , 0 },
	{ "arrowupdownbase", 0x21a8 , 0 },
	{ "arrowupleft", 0x2196 , 0 },
	{ "arrowupleftofdown", 0x21c5 , 0 },
	{ "arrowupright", 0x2197 , 0 },
	{ "arrowupwhite", 0x21e7 , 0 },
	{ "arrowvertex", 0xf8e6 , 0 },
	{ "asciicircummonospace", 0xff3e , 0 },
	{ "asciitildemonospace", 0xff5e , 0 },
	{ "ascript", 0x0251 , 0 },
	{ "ascriptturned", 0x0252 , 0 },
	{ "asmallhiragana", 0x3041 , 0 },
	{ "asmallkatakana", 0x30a1 , 0 },
	{ "asmallkatakanahalfwidth", 0xff67 , 0 },
	{ "asteriskaltonearabic", 0x066d , 0 },
	{ "asteriskarabic", 0x066d , 0 },
	{ "asteriskmonospace", 0xff0a , 0 },
	{ "asterisksmall", 0xfe61 , 0 },
	{ "asterism", 0x2042 , 0 },
	{ "asuperior", 0xf6e9 , 0 },
	{ "asymptoticallyequal", 0x2243 , 0 },
	{ "atmonospace", 0xff20 , 0 },
	{ "atsmall", 0xfe6b , 0 },
	{ "aturned", 0x0250 , 0 },
	{ "aubengali", 0x0994 , 0 },
	{ "aubopomofo", 0x3120 , 0 },
	{ "audeva", 0x0914 , 0 },
	{ "augujarati", 0x0a94 , 0 },
	{ "augurmukhi", 0x0a14 , 0 },
	{ "aulengthmarkbengali", 0x09d7 , 0 },
	{ "aumatragurmukhi", 0x0a4c , 0 },
	{ "auvowelsignbengali", 0x09cc , 0 },
	{ "auvowelsigndeva", 0x094c , 0 },
	{ "auvowelsigngujarati", 0x0acc , 0 },
	{ "avagrahadeva", 0x093d , 0 },
	{ "aybarmenian", 0x0561 , 0 },
	{ "ayin", 0x05e2 , 0 },
	{ "ayinaltonehebrew", 0xfb20 , 0 },
	{ "ayinhebrew", 0x05e2 , 0 },
	{ "babengali", 0x09ac , 0 },
	{ "backslashmonospace", 0xff3c , 0 },
	{ "badeva", 0x092c , 0 },
	{ "bagujarati", 0x0aac , 0 },
	{ "bagurmukhi", 0x0a2c , 0 },
	{ "bahiragana", 0x3070 , 0 },
	{ "bahtthai", 0x0e3f , 0 },
	{ "bakatakana", 0x30d0 , 0 },
	{ "barmonospace", 0xff5c , 0 },
	{ "bbopomofo", 0x3105 , 0 },
	{ "bcircle", 0x24d1 , 0 },
	{ "bdotaccent", 0x1e03 , 0 },
	{ "bdotbelow", 0x1e05 , 0 },
	{ "beamedsixteenthnotes", 0x266c , 0 },
	{ "because", 0x2235 , 0 },
	{ "becyrillic", 0x0431 , 0 },
	{ "beharabic", 0x0628 , 0 },
	{ "behfinalarabic", 0xfe90 , 0 },
	{ "behinitialarabic", 0xfe91 , 0 },
	{ "behiragana", 0x3079 , 0 },
	{ "behmedialarabic", 0xfe92 , 0 },
	{ "behmeeminitialarabic", 0xfc9f , 0 },
	{ "behmeemisolatedarabic", 0xfc08 , 0 },
	{ "behnoonfinalarabic", 0xfc6d , 0 },
	{ "bekatakana", 0x30d9 , 0 },
	{ "benarmenian", 0x0562 , 0 },
	{ "bet", 0x05d1 , 0 },
	{ "betasymbolgreek", 0x03d0 , 0 },
	{ "betdagesh", 0xfb31 , 0 },
	{ "betdageshhebrew", 0xfb31 , 0 },
	{ "bethebrew", 0x05d1 , 0 },
	{ "betrafehebrew", 0xfb4c , 0 },
	{ "bhabengali", 0x09ad , 0 },
	{ "bhadeva", 0x092d , 0 },
	{ "bhagujarati", 0x0aad , 0 },
	{ "bhagurmukhi", 0x0a2d , 0 },
	{ "bhook", 0x0253 , 0 },
	{ "bihiragana", 0x3073 , 0 },
	{ "bikatakana", 0x30d3 , 0 },
	{ "bilabialclick", 0x0298 , 0 },
	{ "bindigurmukhi", 0x0a02 , 0 },
	{ "birusquare", 0x3331 , 0 },
	{ "blackcircle", 0x25cf , 0 },
	{ "blackdiamond", 0x25c6 , 0 },
	{ "blackdownpointingtriangle", 0x25bc , 0 },
	{ "blackleftpointingpointer", 0x25c4 , 0 },
	{ "blackleftpointingtriangle", 0x25c0 , 0 },
	{ "blacklenticularbracketleft", 0x3010 , 0 },
	{ "blacklenticularbracketleftvertical", 0xfe3b , 0 },
	{ "blacklenticularbracketright", 0x3011 , 0 },
	{ "blacklenticularbracketrightvertical", 0xfe3c , 0 },
	{ "blacklowerlefttriangle", 0x25e3 , 0 },
	{ "blacklowerrighttriangle", 0x25e2 , 0 },
	{ "blackrectangle", 0x25ac , 0 },
	{ "blackrightpointingpointer", 0x25ba , 0 },
	{ "blackrightpointingtriangle", 0x25b6 , 0 },
	{ "blacksmallsquare", 0x25aa , 0 },
	{ "blacksmilingface", 0x263b , 0 },
	{ "blacksquare", 0x25a0 , 0 },
	{ "blackstar", 0x2605 , 0 },
	{ "blackupperlefttriangle", 0x25e4 , 0 },
	{ "blackupperrighttriangle", 0x25e5 , 0 },
	{ "blackuppointingsmalltriangle", 0x25b4 , 0 },
	{ "blackuppointingtriangle", 0x25b2 , 0 },
	{ "blank", 0x2423 , 0 },
	{ "blinebelow", 0x1e07 , 0 },
	{ "bmonospace", 0xff42 , 0 },
	{ "bobaimaithai", 0x0e1a , 0 },
	{ "bohiragana", 0x307c , 0 },
	{ "bokatakana", 0x30dc , 0 },
	{ "bparen", 0x249d , 0 },
	{ "bqsquare", 0x33c3 , 0 },
	{ "braceex", 0xf8f4 , 0 },
	{ "braceleftbt", 0xf8f3 , 0 },
	{ "braceleftmid", 0xf8f2 , 0 },
	{ "braceleftmonospace", 0xff5b , 0 },
	{ "braceleftsmall", 0xfe5b , 0 },
	{ "bracelefttp", 0xf8f1 , 0 },
	{ "braceleftvertical", 0xfe37 , 0 },
	{ "bracerightbt", 0xf8fe , 0 },
	{ "bracerightmid", 0xf8fd , 0 },
	{ "bracerightmonospace", 0xff5d , 0 },
	{ "bracerightsmall", 0xfe5c , 0 },
	{ "bracerighttp", 0xf8fc , 0 },
	{ "bracerightvertical", 0xfe38 , 0 },
	{ "bracketleftbt", 0xf8f0 , 0 },
	{ "bracketleftex", 0xf8ef , 0 },
	{ "bracketleftmonospace", 0xff3b , 0 },
	{ "bracketlefttp", 0xf8ee , 0 },
	{ "bracketrightbt", 0xf8fb , 0 },
	{ "bracketrightex", 0xf8fa , 0 },
	{ "bracketrightmonospace", 0xff3d , 0 },
	{ "bracketrighttp", 0xf8f9 , 0 },
	{ "brevebelowcmb", 0x032e , 0 },
	{ "brevecmb", 0x0306 , 0 },
	{ "breveinvertedbelowcmb", 0x032f , 0 },
	{ "breveinvertedcmb", 0x0311 , 0 },
	{ "breveinverteddoublecmb", 0x0361 , 0 },
	{ "bridgebelowcmb", 0x032a , 0 },
	{ "bridgeinvertedbelowcmb", 0x033a , 0 },
	{ "bstroke", 0x0180 , 0 },
	{ "bsuperior", 0xf6ea , 0 },
	{ "btopbar", 0x0183 , 0 },
	{ "buhiragana", 0x3076 , 0 },
	{ "bukatakana", 0x30d6 , 0 },
	{ "bulletinverse", 0x25d8 , 0 },
	{ "bulletoperator", 0x2219 , 0 },
	{ "bullseye", 0x25ce , 0 },
	{ "caarmenian", 0x056e , 0 },
	{ "cabengali", 0x099a , 0 },
	{ "cadeva", 0x091a , 0 },
	{ "cagujarati", 0x0a9a , 0 },
	{ "cagurmukhi", 0x0a1a , 0 },
	{ "calsquare", 0x3388 , 0 },
	{ "candrabindubengali", 0x0981 , 0 },
	{ "candrabinducmb", 0x0310 , 0 },
	{ "candrabindudeva", 0x0901 , 0 },
	{ "candrabindugujarati", 0x0a81 , 0 },
	{ "capslock", 0x21ea , 0 },
	{ "careof", 0x2105 , 0 },
	{ "caronbelowcmb", 0x032c , 0 },
	{ "caroncmb", 0x030c , 0 },
	{ "cbopomofo", 0x3118 , 0 },
	{ "ccedillaacute", 0x1e09 , 0 },
	{ "ccircle", 0x24d2 , 0 },
	{ "ccurl", 0x0255 , 0 },
	{ "cdot", 0x010b , 0 },
	{ "cdsquare", 0x33c5 , 0 },
	{ "cedillacmb", 0x0327 , 0 },
	{ "centigrade", 0x2103 , 0 },
	{ "centinferior", 0xf6df , 0 },
	{ "centmonospace", 0xffe0 , 0 },
	{ "centoldstyle", 0xf7a2 , 0 },
	{ "centsuperior", 0xf6e0 , 0 },
	{ "chaarmenian", 0x0579 , 0 },
	{ "chabengali", 0x099b , 0 },
	{ "chadeva", 0x091b , 0 },
	{ "chagujarati", 0x0a9b , 0 },
	{ "chagurmukhi", 0x0a1b , 0 },
	{ "chbopomofo", 0x3114 , 0 },
	{ "cheabkhasiancyrillic", 0x04bd , 0 },
	{ "checkmark", 0x2713 , 0 },
	{ "checyrillic", 0x0447 , 0 },
	{ "chedescenderabkhasiancyrillic", 0x04bf , 0 },
	{ "chedescendercyrillic", 0x04b7 , 0 },
	{ "chedieresiscyrillic", 0x04f5 , 0 },
	{ "cheharmenian", 0x0573 , 0 },
	{ "chekhakassiancyrillic", 0x04cc , 0 },
	{ "cheverticalstrokecyrillic", 0x04b9 , 0 },
	{ "chieuchacirclekorean", 0x3277 , 0 },
	{ "chieuchaparenkorean", 0x3217 , 0 },
	{ "chieuchcirclekorean", 0x3269 , 0 },
	{ "chieuchkorean", 0x314a , 0 },
	{ "chieuchparenkorean", 0x3209 , 0 },
	{ "chochangthai", 0x0e0a , 0 },
	{ "chochanthai", 0x0e08 , 0 },
	{ "chochingthai", 0x0e09 , 0 },
	{ "chochoethai", 0x0e0c , 0 },
	{ "chook", 0x0188 , 0 },
	{ "cieucacirclekorean", 0x3276 , 0 },
	{ "cieucaparenkorean", 0x3216 , 0 },
	{ "cieuccirclekorean", 0x3268 , 0 },
	{ "cieuckorean", 0x3148 , 0 },
	{ "cieucparenkorean", 0x3208 , 0 },
	{ "cieucuparenkorean", 0x321c , 0 },
	{ "circleot", 0x2299 , 0 },			/* Typo in Adobe's glyphlist */
	{ "circledot", 0x2299 , 0 },		/* But same typo exists in acrobat */
	{ "circlepostalmark", 0x3036 , 0 },
	{ "circlewithlefthalfblack", 0x25d0 , 0 },
	{ "circlewithrighthalfblack", 0x25d1 , 0 },
	{ "circumflexbelowcmb", 0x032d , 0 },
	{ "circumflexcmb", 0x0302 , 0 },
	{ "clear", 0x2327 , 0 },
	{ "clickalveolar", 0x01c2 , 0 },
	{ "clickdental", 0x01c0 , 0 },
	{ "clicklateral", 0x01c1 , 0 },
	{ "clickretroflex", 0x01c3 , 0 },
	{ "clubsuitblack", 0x2663 , 0 },
	{ "clubsuitwhite", 0x2667 , 0 },
	{ "cmcubedsquare", 0x33a4 , 0 },
	{ "cmonospace", 0xff43 , 0 },
	{ "cmsquaredsquare", 0x33a0 , 0 },
	{ "coarmenian", 0x0581 , 0 },
	{ "colonmonospace", 0xff1a , 0 },
	{ "colonsign", 0x20a1 , 0 },
	{ "colonsmall", 0xfe55 , 0 },
	{ "colontriangularhalfmod", 0x02d1 , 0 },
	{ "colontriangularmod", 0x02d0 , 0 },
	{ "commaabovecmb", 0x0313 , 0 },
	{ "commaaboverightcmb", 0x0315 , 0 },
	{ "commaaccent", 0xf6c3 , 0 },
	{ "commaarabic", 0x060c , 0 },
	{ "commaarmenian", 0x055d , 0 },
	{ "commainferior", 0xf6e1 , 0 },
	{ "commamonospace", 0xff0c , 0 },
	{ "commareversedabovecmb", 0x0314 , 0 },
	{ "commareversedmod", 0x02bd , 0 },
	{ "commasmall", 0xfe50 , 0 },
	{ "commasuperior", 0xf6e2 , 0 },
	{ "commaturnedabovecmb", 0x0312 , 0 },
	{ "commaturnedmod", 0x02bb , 0 },
	{ "compass", 0x263c , 0 },
	{ "contourintegral", 0x222e , 0 },
	{ "control", 0x2303 , 0 },
	{ "controlACK", 0x0006 , 0 },
	{ "controlBEL", 0x0007 , 0 },
	{ "controlBS", 0x0008 , 0 },
	{ "controlCAN", 0x0018 , 0 },
	{ "controlCR", 0x000d , 0 },
	{ "controlDC1", 0x0011 , 0 },
	{ "controlDC2", 0x0012 , 0 },
	{ "controlDC3", 0x0013 , 0 },
	{ "controlDC4", 0x0014 , 0 },
	{ "controlDEL", 0x007f , 0 },
	{ "controlDLE", 0x0010 , 0 },
	{ "controlEM", 0x0019 , 0 },
	{ "controlENQ", 0x0005 , 0 },
	{ "controlEOT", 0x0004 , 0 },
	{ "controlESC", 0x001b , 0 },
	{ "controlETB", 0x0017 , 0 },
	{ "controlETX", 0x0003 , 0 },
	{ "controlFF", 0x000c , 0 },
	{ "controlFS", 0x001c , 0 },
	{ "controlGS", 0x001d , 0 },
	{ "controlHT", 0x0009 , 0 },
	{ "controlLF", 0x000a , 0 },
	{ "controlNAK", 0x0015 , 0 },
	{ "controlRS", 0x001e , 0 },
	{ "controlSI", 0x000f , 0 },
	{ "controlSO", 0x000e , 0 },
	{ "controlSOT", 0x0002 , 0 },
	{ "controlSTX", 0x0001 , 0 },
	{ "controlSUB", 0x001a , 0 },
	{ "controlSYN", 0x0016 , 0 },
	{ "controlUS", 0x001f , 0 },
	{ "controlVT", 0x000b , 0 },
	{ "copyrightsans", 0xf8e9 , 0 },
	{ "copyrightserif", 0xf6d9 , 0 },
	{ "cornerbracketleft", 0x300c , 0 },
	{ "cornerbracketlefthalfwidth", 0xff62 , 0 },
	{ "cornerbracketleftvertical", 0xfe41 , 0 },
	{ "cornerbracketright", 0x300d , 0 },
	{ "cornerbracketrighthalfwidth", 0xff63 , 0 },
	{ "cornerbracketrightvertical", 0xfe42 , 0 },
	{ "corporationsquare", 0x337f , 0 },
	{ "cosquare", 0x33c7 , 0 },
	{ "coverkgsquare", 0x33c6 , 0 },
	{ "cparen", 0x249e , 0 },
	{ "cruzeiro", 0x20a2 , 0 },
	{ "cstretched", 0x0297 , 0 },
	{ "curlyand", 0x22cf , 0 },
	{ "curlyor", 0x22ce , 0 },
	{ "cyrBreve", 0xf6d1 , 0 },
	{ "cyrFlex", 0xf6d2 , 0 },
	{ "cyrbreve", 0xf6d4 , 0 },
	{ "cyrflex", 0xf6d5 , 0 },
	{ "daarmenian", 0x0564 , 0 },
	{ "dabengali", 0x09a6 , 0 },
	{ "dadarabic", 0x0636 , 0 },
	{ "dadeva", 0x0926 , 0 },
	{ "dadfinalarabic", 0xfebe , 0 },
	{ "dadinitialarabic", 0xfebf , 0 },
	{ "dadmedialarabic", 0xfec0 , 0 },
	{ "dagesh", 0x05bc , 0 },
	{ "dageshhebrew", 0x05bc , 0 },
	{ "dagujarati", 0x0aa6 , 0 },
	{ "dagurmukhi", 0x0a26 , 0 },
	{ "dahiragana", 0x3060 , 0 },
	{ "dakatakana", 0x30c0 , 0 },
	{ "dalarabic", 0x062f , 0 },
	{ "dalet", 0x05d3 , 0 },
	{ "daletdagesh", 0xfb33 , 0 },
	{ "daletdageshhebrew", 0xfb33 , 0 },
	{ "dalethatafpatah", 0x05d3 , 0 },
	{ "dalethatafpatahhebrew", 0x05d3 , 0 },
	{ "dalethatafsegol", 0x05d3 , 0 },
	{ "dalethatafsegolhebrew", 0x05d3 , 0 },
	{ "dalethebrew", 0x05d3 , 0 },
	{ "dalethiriq", 0x05d3 , 0 },
	{ "dalethiriqhebrew", 0x05d3 , 0 },
	{ "daletholam", 0x05d3 , 0 },
	{ "daletholamhebrew", 0x05d3 , 0 },
	{ "daletpatah", 0x05d3 , 0 },
	{ "daletpatahhebrew", 0x05d3 , 0 },
	{ "daletqamats", 0x05d3 , 0 },
	{ "daletqamatshebrew", 0x05d3 , 0 },
	{ "daletqubuts", 0x05d3 , 0 },
	{ "daletqubutshebrew", 0x05d3 , 0 },
	{ "daletsegol", 0x05d3 , 0 },
	{ "daletsegolhebrew", 0x05d3 , 0 },
	{ "daletsheva", 0x05d3 , 0 },
	{ "daletshevahebrew", 0x05d3 , 0 },
	{ "dalettsere", 0x05d3 , 0 },
	{ "dalettserehebrew", 0x05d3 , 0 },
	{ "dalfinalarabic", 0xfeaa , 0 },
	{ "dammaarabic", 0x064f , 0 },
	{ "dammalowarabic", 0x064f , 0 },
	{ "dammatanaltonearabic", 0x064c , 0 },
	{ "dammatanarabic", 0x064c , 0 },
	{ "danda", 0x0964 , 0 },
	{ "dargahebrew", 0x05a7 , 0 },
	{ "dargalefthebrew", 0x05a7 , 0 },
	{ "dasiapneumatacyrilliccmb", 0x0485 , 0 },
	{ "dblGrave", 0xf6d3 , 0 },
	{ "dblanglebracketleft", 0x300a , 0 },
	{ "dblanglebracketleftvertical", 0xfe3d , 0 },
	{ "dblanglebracketright", 0x300b , 0 },
	{ "dblanglebracketrightvertical", 0xfe3e , 0 },
	{ "dblarchinvertedbelowcmb", 0x032b , 0 },
	{ "dblarrowleft", 0x21d4 , 0 },
	{ "dblarrowright", 0x21d2 , 0 },
	{ "dbldanda", 0x0965 , 0 },
	{ "dblgrave", 0xf6d6 , 0 },
	{ "dblgravecmb", 0x030f , 0 },
	{ "dblintegral", 0x222c , 0 },
	{ "dbllowline", 0x2017 , 0 },
	{ "dbllowlinecmb", 0x0333 , 0 },
	{ "dbloverlinecmb", 0x033f , 0 },
	{ "dblprimemod", 0x02ba , 0 },
	{ "dblverticalbar", 0x2016 , 0 },
	{ "dblverticallineabovecmb", 0x030e , 0 },
	{ "dbopomofo", 0x3109 , 0 },
	{ "dbsquare", 0x33c8 , 0 },
	{ "dcedilla", 0x1e11 , 0 },
	{ "dcircle", 0x24d3 , 0 },
	{ "dcircumflexbelow", 0x1e13 , 0 },
	{ "ddabengali", 0x09a1 , 0 },
	{ "ddadeva", 0x0921 , 0 },
	{ "ddagujarati", 0x0aa1 , 0 },
	{ "ddagurmukhi", 0x0a21 , 0 },
	{ "ddalarabic", 0x0688 , 0 },
	{ "ddalfinalarabic", 0xfb89 , 0 },
	{ "dddhadeva", 0x095c , 0 },
	{ "ddhabengali", 0x09a2 , 0 },
	{ "ddhadeva", 0x0922 , 0 },
	{ "ddhagujarati", 0x0aa2 , 0 },
	{ "ddhagurmukhi", 0x0a22 , 0 },
	{ "ddotaccent", 0x1e0b , 0 },
	{ "ddotbelow", 0x1e0d , 0 },
	{ "decimalseparatorarabic", 0x066b , 0 },
	{ "decimalseparatorpersian", 0x066b , 0 },
	{ "decyrillic", 0x0434 , 0 },
	{ "dehihebrew", 0x05ad , 0 },
	{ "dehiragana", 0x3067 , 0 },
	{ "deicoptic", 0x03ef , 0 },
	{ "dekatakana", 0x30c7 , 0 },
	{ "deleteleft", 0x232b , 0 },
	{ "deleteright", 0x2326 , 0 },
	{ "deltaturned", 0x018d , 0 },
	{ "denominatorminusonenumeratorbengali", 0x09f8 , 0 },
	{ "dezh", 0x02a4 , 0 },
	{ "dhabengali", 0x09a7 , 0 },
	{ "dhadeva", 0x0927 , 0 },
	{ "dhagujarati", 0x0aa7 , 0 },
	{ "dhagurmukhi", 0x0a27 , 0 },
	{ "dhook", 0x0257 , 0 },
	{ "dialytikatonos", 0x0385 , 0 },
	{ "dialytikatonoscmb", 0x0344 , 0 },
	{ "diamondsuitwhite", 0x2662 , 0 },
	{ "dieresisacute", 0xf6d7 , 0 },
	{ "dieresisbelowcmb", 0x0324 , 0 },
	{ "dieresiscmb", 0x0308 , 0 },
	{ "dieresisgrave", 0xf6d8 , 0 },
	{ "dihiragana", 0x3062 , 0 },
	{ "dikatakana", 0x30c2 , 0 },
	{ "dittomark", 0x3003 , 0 },
	{ "divides", 0x2223 , 0 },
	{ "divisionslash", 0x2215 , 0 },
	{ "djecyrillic", 0x0452 , 0 },
	{ "dlinebelow", 0x1e0f , 0 },
	{ "dlsquare", 0x3397 , 0 },
	{ "dmacron", 0x0111 , 0 },
	{ "dmonospace", 0xff44 , 0 },
	{ "dochadathai", 0x0e0e , 0 },
	{ "dodekthai", 0x0e14 , 0 },
	{ "dohiragana", 0x3069 , 0 },
	{ "dokatakana", 0x30c9 , 0 },
	{ "dollarinferior", 0xf6e3 , 0 },
	{ "dollarmonospace", 0xff04 , 0 },
	{ "dollaroldstyle", 0xf724 , 0 },
	{ "dollarsmall", 0xfe69 , 0 },
	{ "dollarsuperior", 0xf6e4 , 0 },
	{ "dorusquare", 0x3326 , 0 },
	{ "dotaccentcmb", 0x0307 , 0 },
	{ "dotbelowcmb", 0x0323 , 0 },
	{ "dotkatakana", 0x30fb , 0 },
	{ "dotlessj", 0x0237 , 0 },			/* !!!! AGL Still says this is 0xf6be */
	{ "dotlessjstrokehook", 0x0284 , 0 },
	{ "dottedcircle", 0x25cc , 0 },
	{ "doubleyodpatah", 0xfb1f , 0 },
	{ "doubleyodpatahhebrew", 0xfb1f , 0 },
	{ "downtackbelowcmb", 0x031e , 0 },
	{ "downtackmod", 0x02d5 , 0 },
	{ "dparen", 0x249f , 0 },
	{ "dsuperior", 0xf6eb , 0 },
	{ "dtail", 0x0256 , 0 },
	{ "dtopbar", 0x018c , 0 },
	{ "duhiragana", 0x3065 , 0 },
	{ "dukatakana", 0x30c5 , 0 },
	{ "dz", 0x01f3 , 0 },
	{ "dzaltone", 0x02a3 , 0 },
	{ "dzcaron", 0x01c6 , 0 },
	{ "dzcurl", 0x02a5 , 0 },
	{ "dzeabkhasiancyrillic", 0x04e1 , 0 },
	{ "dzecyrillic", 0x0455 , 0 },
	{ "dzhecyrillic", 0x045f , 0 },
	{ "earth", 0x2641 , 0 },
	{ "ebengali", 0x098f , 0 },
	{ "ebopomofo", 0x311c , 0 },
	{ "ecandradeva", 0x090d , 0 },
	{ "ecandragujarati", 0x0a8d , 0 },
	{ "ecandravowelsigndeva", 0x0945 , 0 },
	{ "ecandravowelsigngujarati", 0x0ac5 , 0 },
	{ "ecedillabreve", 0x1e1d , 0 },
	{ "echarmenian", 0x0565 , 0 },
	{ "echyiwnarmenian", 0x0587 , 0 },
	{ "ecircle", 0x24d4 , 0 },
	{ "ecircumflexacute", 0x1ebf , 0 },
	{ "ecircumflexbelow", 0x1e19 , 0 },
	{ "ecircumflexdotbelow", 0x1ec7 , 0 },
	{ "ecircumflexgrave", 0x1ec1 , 0 },
	{ "ecircumflexhookabove", 0x1ec3 , 0 },
	{ "ecircumflextilde", 0x1ec5 , 0 },
	{ "ecyrillic", 0x0454 , 0 },
	{ "edblgrave", 0x0205 , 0 },
	{ "edeva", 0x090f , 0 },
	{ "edot", 0x0117 , 0 },
	{ "edotbelow", 0x1eb9 , 0 },
	{ "eegurmukhi", 0x0a0f , 0 },
	{ "eematragurmukhi", 0x0a47 , 0 },
	{ "efcyrillic", 0x0444 , 0 },
	{ "egujarati", 0x0a8f , 0 },
	{ "eharmenian", 0x0567 , 0 },
	{ "ehbopomofo", 0x311d , 0 },
	{ "ehiragana", 0x3048 , 0 },
	{ "ehookabove", 0x1ebb , 0 },
	{ "eibopomofo", 0x311f , 0 },
	{ "eightarabic", 0x0668 , 0 },
	{ "eightbengali", 0x09ee , 0 },
	{ "eightcircle", 0x2467 , 0 },
	{ "eightcircleinversesansserif", 0x2791 , 0 },
	{ "eightdeva", 0x096e , 0 },
	{ "eighteencircle", 0x2471 , 0 },
	{ "eighteenparen", 0x2485 , 0 },
	{ "eighteenperiod", 0x2499 , 0 },
	{ "eightgujarati", 0x0aee , 0 },
	{ "eightgurmukhi", 0x0a6e , 0 },
	{ "eighthackarabic", 0x0668 , 0 },
	{ "eighthangzhou", 0x3028 , 0 },
	{ "eighthnotebeamed", 0x266b , 0 },
	{ "eightideographicparen", 0x3227 , 0 },
	{ "eightinferior", 0x2088 , 0 },
	{ "eightmonospace", 0xff18 , 0 },
	{ "eightoldstyle", 0xf738 , 0 },
	{ "eightparen", 0x247b , 0 },
	{ "eightperiod", 0x248f , 0 },
	{ "eightpersian", 0x06f8 , 0 },
	{ "eightroman", 0x2177 , 0 },
	{ "eightsuperior", 0x2078 , 0 },
	{ "eightthai", 0x0e58 , 0 },
	{ "einvertedbreve", 0x0207 , 0 },
	{ "eiotifiedcyrillic", 0x0465 , 0 },
	{ "ekatakana", 0x30a8 , 0 },
	{ "ekatakanahalfwidth", 0xff74 , 0 },
	{ "ekonkargurmukhi", 0x0a74 , 0 },
	{ "ekorean", 0x3154 , 0 },
	{ "elcyrillic", 0x043b , 0 },
	{ "elevencircle", 0x246a , 0 },
	{ "elevenparen", 0x247e , 0 },
	{ "elevenperiod", 0x2492 , 0 },
	{ "elevenroman", 0x217a , 0 },
	{ "ellipsisvertical", 0x22ee , 0 },
	{ "emacronacute", 0x1e17 , 0 },
	{ "emacrongrave", 0x1e15 , 0 },
	{ "emcyrillic", 0x043c , 0 },
	{ "emdashvertical", 0xfe31 , 0 },
	{ "emonospace", 0xff45 , 0 },
	{ "emphasismarkarmenian", 0x055b , 0 },
	{ "enbopomofo", 0x3123 , 0 },
	{ "encyrillic", 0x043d , 0 },
	{ "endashvertical", 0xfe32 , 0 },
	{ "endescendercyrillic", 0x04a3 , 0 },
	{ "engbopomofo", 0x3125 , 0 },
	{ "enghecyrillic", 0x04a5 , 0 },
	{ "enhookcyrillic", 0x04c8 , 0 },
	{ "enspace", 0x2002 , 0 },
	{ "eokorean", 0x3153 , 0 },
	{ "eopen", 0x025b , 0 },
	{ "eopenclosed", 0x029a , 0 },
	{ "eopenreversed", 0x025c , 0 },
	{ "eopenreversedclosed", 0x025e , 0 },
	{ "eopenreversedhook", 0x025d , 0 },
	{ "eparen", 0x24a0 , 0 },
	{ "equalmonospace", 0xff1d , 0 },
	{ "equalsmall", 0xfe66 , 0 },
	{ "equalsuperior", 0x207c , 0 },
	{ "erbopomofo", 0x3126 , 0 },
	{ "ercyrillic", 0x0440 , 0 },
	{ "ereversed", 0x0258 , 0 },
	{ "ereversedcyrillic", 0x044d , 0 },
	{ "escyrillic", 0x0441 , 0 },
	{ "esdescendercyrillic", 0x04ab , 0 },
	{ "esh", 0x0283 , 0 },
	{ "eshcurl", 0x0286 , 0 },
	{ "eshortdeva", 0x090e , 0 },
	{ "eshortvowelsigndeva", 0x0946 , 0 },
	{ "eshreversedloop", 0x01aa , 0 },
	{ "eshsquatreversed", 0x0285 , 0 },
	{ "esmallhiragana", 0x3047 , 0 },
	{ "esmallkatakana", 0x30a7 , 0 },
	{ "esmallkatakanahalfwidth", 0xff6a , 0 },
	{ "esuperior", 0xf6ec , 0 },
	{ "etarmenian", 0x0568 , 0 },
	{ "etilde", 0x1ebd , 0 },
	{ "etildebelow", 0x1e1b , 0 },
	{ "etnahtafoukhhebrew", 0x0591 , 0 },
	{ "etnahtafoukhlefthebrew", 0x0591 , 0 },
	{ "etnahtahebrew", 0x0591 , 0 },
	{ "etnahtalefthebrew", 0x0591 , 0 },
	{ "eturned", 0x01dd , 0 },
	{ "eukorean", 0x3161 , 0 },
	{ "euro", 0x20ac , 0 },
	{ "evowelsignbengali", 0x09c7 , 0 },
	{ "evowelsigndeva", 0x0947 , 0 },
	{ "evowelsigngujarati", 0x0ac7 , 0 },
	{ "exclamarmenian", 0x055c , 0 },
	{ "exclamdownsmall", 0xf7a1 , 0 },
	{ "exclammonospace", 0xff01 , 0 },
	{ "exclamsmall", 0xf721 , 0 },
	{ "ezh", 0x0292 , 0 },
	{ "ezhcaron", 0x01ef , 0 },
	{ "ezhcurl", 0x0293 , 0 },
	{ "ezhreversed", 0x01b9 , 0 },
	{ "ezhtail", 0x01ba , 0 },
	{ "fadeva", 0x095e , 0 },
	{ "fagurmukhi", 0x0a5e , 0 },
	{ "fahrenheit", 0x2109 , 0 },
	{ "fathaarabic", 0x064e , 0 },
	{ "fathalowarabic", 0x064e , 0 },
	{ "fathatanarabic", 0x064b , 0 },
	{ "fbopomofo", 0x3108 , 0 },
	{ "fcircle", 0x24d5 , 0 },
	{ "fdotaccent", 0x1e1f , 0 },
	{ "feharabic", 0x0641 , 0 },
	{ "feharmenian", 0x0586 , 0 },
	{ "fehfinalarabic", 0xfed2 , 0 },
	{ "fehinitialarabic", 0xfed3 , 0 },
	{ "fehmedialarabic", 0xfed4 , 0 },
	{ "feicoptic", 0x03e5 , 0 },
	{ "ff", 0xfb00 , 0 },
	{ "ffi", 0xfb03 , 0 },
	{ "ffl", 0xfb04 , 0 },
	{ "fi", 0xfb01 , 0 },
	{ "fifteencircle", 0x246e , 0 },
	{ "fifteenparen", 0x2482 , 0 },
	{ "fifteenperiod", 0x2496 , 0 },
	{ "finalkaf", 0x05da , 0 },
	{ "finalkafdagesh", 0xfb3a , 0 },
	{ "finalkafdageshhebrew", 0xfb3a , 0 },
	{ "finalkafhebrew", 0x05da , 0 },
	{ "finalkafqamats", 0x05da , 0 },
	{ "finalkafqamatshebrew", 0x05da , 0 },
	{ "finalkafsheva", 0x05da , 0 },
	{ "finalkafshevahebrew", 0x05da , 0 },
	{ "finalmem", 0x05dd , 0 },
	{ "finalmemhebrew", 0x05dd , 0 },
	{ "finalnun", 0x05df , 0 },
	{ "finalnunhebrew", 0x05df , 0 },
	{ "finalpe", 0x05e3 , 0 },
	{ "finalpehebrew", 0x05e3 , 0 },
	{ "finaltsadi", 0x05e5 , 0 },
	{ "finaltsadihebrew", 0x05e5 , 0 },
	{ "firsttonechinese", 0x02c9 , 0 },
	{ "fisheye", 0x25c9 , 0 },
	{ "fitacyrillic", 0x0473 , 0 },
	{ "fivearabic", 0x0665 , 0 },
	{ "fivebengali", 0x09eb , 0 },
	{ "fivecircle", 0x2464 , 0 },
	{ "fivecircleinversesansserif", 0x278e , 0 },
	{ "fivedeva", 0x096b , 0 },
	{ "fivegujarati", 0x0aeb , 0 },
	{ "fivegurmukhi", 0x0a6b , 0 },
	{ "fivehackarabic", 0x0665 , 0 },
	{ "fivehangzhou", 0x3025 , 0 },
	{ "fiveideographicparen", 0x3224 , 0 },
	{ "fiveinferior", 0x2085 , 0 },
	{ "fivemonospace", 0xff15 , 0 },
	{ "fiveoldstyle", 0xf735 , 0 },
	{ "fiveparen", 0x2478 , 0 },
	{ "fiveperiod", 0x248c , 0 },
	{ "fivepersian", 0x06f5 , 0 },
	{ "fiveroman", 0x2174 , 0 },
	{ "fivesuperior", 0x2075 , 0 },
	{ "fivethai", 0x0e55 , 0 },
	{ "fl", 0xfb02 , 0 },
	{ "fmonospace", 0xff46 , 0 },
	{ "fmsquare", 0x3399 , 0 },
	{ "fofanthai", 0x0e1f , 0 },
	{ "fofathai", 0x0e1d , 0 },
	{ "fongmanthai", 0x0e4f , 0 },
	{ "forall", 0x2200 , 0 },
	{ "fourarabic", 0x0664 , 0 },
	{ "fourbengali", 0x09ea , 0 },
	{ "fourcircle", 0x2463 , 0 },
	{ "fourcircleinversesansserif", 0x278d , 0 },
	{ "fourdeva", 0x096a , 0 },
	{ "fourgujarati", 0x0aea , 0 },
	{ "fourgurmukhi", 0x0a6a , 0 },
	{ "fourhackarabic", 0x0664 , 0 },
	{ "fourhangzhou", 0x3024 , 0 },
	{ "fourideographicparen", 0x3223 , 0 },
	{ "fourinferior", 0x2084 , 0 },
	{ "fourmonospace", 0xff14 , 0 },
	{ "fournumeratorbengali", 0x09f7 , 0 },
	{ "fouroldstyle", 0xf734 , 0 },
	{ "fourparen", 0x2477 , 0 },
	{ "fourperiod", 0x248b , 0 },
	{ "fourpersian", 0x06f4 , 0 },
	{ "fourroman", 0x2173 , 0 },
	{ "foursuperior", 0x2074 , 0 },
	{ "fourteencircle", 0x246d , 0 },
	{ "fourteenparen", 0x2481 , 0 },
	{ "fourteenperiod", 0x2495 , 0 },
	{ "fourthai", 0x0e54 , 0 },
	{ "fourthtonechinese", 0x02cb , 0 },
	{ "fparen", 0x24a1 , 0 },
	{ "gabengali", 0x0997 , 0 },
	{ "gacute", 0x01f5 , 0 },
	{ "gadeva", 0x0917 , 0 },
	{ "gafarabic", 0x06af , 0 },
	{ "gaffinalarabic", 0xfb93 , 0 },
	{ "gafinitialarabic", 0xfb94 , 0 },
	{ "gafmedialarabic", 0xfb95 , 0 },
	{ "gagujarati", 0x0a97 , 0 },
	{ "gagurmukhi", 0x0a17 , 0 },
	{ "gahiragana", 0x304c , 0 },
	{ "gakatakana", 0x30ac , 0 },
	{ "gammalatinsmall", 0x0263 , 0 },
	{ "gammasuperior", 0x02e0 , 0 },
	{ "gangiacoptic", 0x03eb , 0 },
	{ "gbopomofo", 0x310d , 0 },
	{ "gcedilla", 0x0123 , 0 },
	{ "gcircle", 0x24d6 , 0 },
	{ "gdot", 0x0121 , 0 },
	{ "gecyrillic", 0x0433 , 0 },
	{ "gehiragana", 0x3052 , 0 },
	{ "gekatakana", 0x30b2 , 0 },
	{ "geometricallyequal", 0x2251 , 0 },
	{ "gereshaccenthebrew", 0x059c , 0 },
	{ "gereshhebrew", 0x05f3 , 0 },
	{ "gereshmuqdamhebrew", 0x059d , 0 },
	{ "gershayimaccenthebrew", 0x059e , 0 },
	{ "gershayimhebrew", 0x05f4 , 0 },
	{ "getamark", 0x3013 , 0 },
	{ "ghabengali", 0x0998 , 0 },
	{ "ghadarmenian", 0x0572 , 0 },
	{ "ghadeva", 0x0918 , 0 },
	{ "ghagujarati", 0x0a98 , 0 },
	{ "ghagurmukhi", 0x0a18 , 0 },
	{ "ghainarabic", 0x063a , 0 },
	{ "ghainfinalarabic", 0xfece , 0 },
	{ "ghaininitialarabic", 0xfecf , 0 },
	{ "ghainmedialarabic", 0xfed0 , 0 },
	{ "ghemiddlehookcyrillic", 0x0495 , 0 },
	{ "ghestrokecyrillic", 0x0493 , 0 },
	{ "gheupturncyrillic", 0x0491 , 0 },
	{ "ghhadeva", 0x095a , 0 },
	{ "ghhagurmukhi", 0x0a5a , 0 },
	{ "ghook", 0x0260 , 0 },
	{ "ghzsquare", 0x3393 , 0 },
	{ "gihiragana", 0x304e , 0 },
	{ "gikatakana", 0x30ae , 0 },
	{ "gimarmenian", 0x0563 , 0 },
	{ "gimel", 0x05d2 , 0 },
	{ "gimeldagesh", 0xfb32 , 0 },
	{ "gimeldageshhebrew", 0xfb32 , 0 },
	{ "gimelhebrew", 0x05d2 , 0 },
	{ "gjecyrillic", 0x0453 , 0 },
	{ "glottalinvertedstroke", 0x01be , 0 },
	{ "glottalstop", 0x0294 , 0 },
	{ "glottalstopinverted", 0x0296 , 0 },
	{ "glottalstopmod", 0x02c0 , 0 },
	{ "glottalstopreversed", 0x0295 , 0 },
	{ "glottalstopreversedmod", 0x02c1 , 0 },
	{ "glottalstopreversedsuperior", 0x02e4 , 0 },
	{ "glottalstopstroke", 0x02a1 , 0 },
	{ "glottalstopstrokereversed", 0x02a2 , 0 },
	{ "gmacron", 0x1e21 , 0 },
	{ "gmonospace", 0xff47 , 0 },
	{ "gohiragana", 0x3054 , 0 },
	{ "gokatakana", 0x30b4 , 0 },
	{ "gparen", 0x24a2 , 0 },
	{ "gpasquare", 0x33ac , 0 },
	{ "gravebelowcmb", 0x0316 , 0 },
	{ "gravecmb", 0x0300 , 0 },
	{ "gravedeva", 0x0953 , 0 },
	{ "gravelowmod", 0x02ce , 0 },
	{ "gravemonospace", 0xff40 , 0 },
	{ "gravetonecmb", 0x0340 , 0 },
	{ "greaterequalorless", 0x22db , 0 },
	{ "greatermonospace", 0xff1e , 0 },
	{ "greaterorequivalent", 0x2273 , 0 },
	{ "greaterorless", 0x2277 , 0 },
	{ "greateroverequal", 0x2267 , 0 },
	{ "greatersmall", 0xfe65 , 0 },
	{ "gscript", 0x0261 , 0 },
	{ "gstroke", 0x01e5 , 0 },
	{ "guhiragana", 0x3050 , 0 },
	{ "gukatakana", 0x30b0 , 0 },
	{ "guramusquare", 0x3318 , 0 },
	{ "gysquare", 0x33c9 , 0 },
	{ "haabkhasiancyrillic", 0x04a9 , 0 },
	{ "haaltonearabic", 0x06c1 , 0 },
	{ "habengali", 0x09b9 , 0 },
	{ "hadescendercyrillic", 0x04b3 , 0 },
	{ "hadeva", 0x0939 , 0 },
	{ "hagujarati", 0x0ab9 , 0 },
	{ "hagurmukhi", 0x0a39 , 0 },
	{ "haharabic", 0x062d , 0 },
	{ "hahfinalarabic", 0xfea2 , 0 },
	{ "hahinitialarabic", 0xfea3 , 0 },
	{ "hahiragana", 0x306f , 0 },
	{ "hahmedialarabic", 0xfea4 , 0 },
	{ "haitusquare", 0x332a , 0 },
	{ "hakatakana", 0x30cf , 0 },
	{ "hakatakanahalfwidth", 0xff8a , 0 },
	{ "halantgurmukhi", 0x0a4d , 0 },
	{ "hamzaarabic", 0x0621 , 0 },
	{ "hamzadammaarabic", 0x0621 , 0 },
	{ "hamzadammatanarabic", 0x0621 , 0 },
	{ "hamzafathaarabic", 0x0621 , 0 },
	{ "hamzafathatanarabic", 0x0621 , 0 },
	{ "hamzalowarabic", 0x0621 , 0 },
	{ "hamzalowkasraarabic", 0x0621 , 0 },
	{ "hamzalowkasratanarabic", 0x0621 , 0 },
	{ "hamzasukunarabic", 0x0621 , 0 },
	{ "hangulfiller", 0x3164 , 0 },
	{ "hardsigncyrillic", 0x044a , 0 },
	{ "harpoonleftbarbup", 0x21bc , 0 },
	{ "harpoonrightbarbup", 0x21c0 , 0 },
	{ "hasquare", 0x33ca , 0 },
	{ "hatafpatah", 0x05b2 , 0 },
	{ "hatafpatah16", 0x05b2 , 0 },
	{ "hatafpatah23", 0x05b2 , 0 },
	{ "hatafpatah2f", 0x05b2 , 0 },
	{ "hatafpatahhebrew", 0x05b2 , 0 },
	{ "hatafpatahnarrowhebrew", 0x05b2 , 0 },
	{ "hatafpatahquarterhebrew", 0x05b2 , 0 },
	{ "hatafpatahwidehebrew", 0x05b2 , 0 },
	{ "hatafqamats", 0x05b3 , 0 },
	{ "hatafqamats1b", 0x05b3 , 0 },
	{ "hatafqamats28", 0x05b3 , 0 },
	{ "hatafqamats34", 0x05b3 , 0 },
	{ "hatafqamatshebrew", 0x05b3 , 0 },
	{ "hatafqamatsnarrowhebrew", 0x05b3 , 0 },
	{ "hatafqamatsquarterhebrew", 0x05b3 , 0 },
	{ "hatafqamatswidehebrew", 0x05b3 , 0 },
	{ "hatafsegol", 0x05b1 , 0 },
	{ "hatafsegol17", 0x05b1 , 0 },
	{ "hatafsegol24", 0x05b1 , 0 },
	{ "hatafsegol30", 0x05b1 , 0 },
	{ "hatafsegolhebrew", 0x05b1 , 0 },
	{ "hatafsegolnarrowhebrew", 0x05b1 , 0 },
	{ "hatafsegolquarterhebrew", 0x05b1 , 0 },
	{ "hatafsegolwidehebrew", 0x05b1 , 0 },
	{ "hbopomofo", 0x310f , 0 },
	{ "hbrevebelow", 0x1e2b , 0 },
	{ "hcedilla", 0x1e29 , 0 },
	{ "hcircle", 0x24d7 , 0 },
	{ "hdieresis", 0x1e27 , 0 },
	{ "hdotaccent", 0x1e23 , 0 },
	{ "hdotbelow", 0x1e25 , 0 },
	{ "he", 0x05d4 , 0 },
	{ "heartsuitblack", 0x2665 , 0 },
	{ "heartsuitwhite", 0x2661 , 0 },
	{ "hedagesh", 0xfb34 , 0 },
	{ "hedageshhebrew", 0xfb34 , 0 },
	{ "hehaltonearabic", 0x06c1 , 0 },
	{ "heharabic", 0x0647 , 0 },
	{ "hehebrew", 0x05d4 , 0 },
	{ "hehfinalaltonearabic", 0xfba7 , 0 },
	{ "hehfinalalttwoarabic", 0xfeea , 0 },
	{ "hehfinalarabic", 0xfeea , 0 },
	{ "hehhamzaabovefinalarabic", 0xfba5 , 0 },
	{ "hehhamzaaboveisolatedarabic", 0xfba4 , 0 },
	{ "hehinitialaltonearabic", 0xfba8 , 0 },
	{ "hehinitialarabic", 0xfeeb , 0 },
	{ "hehiragana", 0x3078 , 0 },
	{ "hehmedialaltonearabic", 0xfba9 , 0 },
	{ "hehmedialarabic", 0xfeec , 0 },
	{ "heiseierasquare", 0x337b , 0 },
	{ "hekatakana", 0x30d8 , 0 },
	{ "hekatakanahalfwidth", 0xff8d , 0 },
	{ "hekutaarusquare", 0x3336 , 0 },
	{ "henghook", 0x0267 , 0 },
	{ "herutusquare", 0x3339 , 0 },
	{ "het", 0x05d7 , 0 },
	{ "hethebrew", 0x05d7 , 0 },
	{ "hhook", 0x0266 , 0 },
	{ "hhooksuperior", 0x02b1 , 0 },
	{ "hieuhacirclekorean", 0x327b , 0 },
	{ "hieuhaparenkorean", 0x321b , 0 },
	{ "hieuhcirclekorean", 0x326d , 0 },
	{ "hieuhkorean", 0x314e , 0 },
	{ "hieuhparenkorean", 0x320d , 0 },
	{ "hihiragana", 0x3072 , 0 },
	{ "hikatakana", 0x30d2 , 0 },
	{ "hikatakanahalfwidth", 0xff8b , 0 },
	{ "hiriq", 0x05b4 , 0 },
	{ "hiriq14", 0x05b4 , 0 },
	{ "hiriq21", 0x05b4 , 0 },
	{ "hiriq2d", 0x05b4 , 0 },
	{ "hiriqhebrew", 0x05b4 , 0 },
	{ "hiriqnarrowhebrew", 0x05b4 , 0 },
	{ "hiriqquarterhebrew", 0x05b4 , 0 },
	{ "hiriqwidehebrew", 0x05b4 , 0 },
	{ "hlinebelow", 0x1e96 , 0 },
	{ "hmonospace", 0xff48 , 0 },
	{ "hoarmenian", 0x0570 , 0 },
	{ "hohipthai", 0x0e2b , 0 },
	{ "hohiragana", 0x307b , 0 },
	{ "hokatakana", 0x30db , 0 },
	{ "hokatakanahalfwidth", 0xff8e , 0 },
	{ "holam", 0x05b9 , 0 },
	{ "holam19", 0x05b9 , 0 },
	{ "holam26", 0x05b9 , 0 },
	{ "holam32", 0x05b9 , 0 },
	{ "holamhebrew", 0x05b9 , 0 },
	{ "holamnarrowhebrew", 0x05b9 , 0 },
	{ "holamquarterhebrew", 0x05b9 , 0 },
	{ "holamwidehebrew", 0x05b9 , 0 },
	{ "honokhukthai", 0x0e2e , 0 },
	{ "hookcmb", 0x0309 , 0 },
	{ "hookpalatalizedbelowcmb", 0x0321 , 0 },
	{ "hookretroflexbelowcmb", 0x0322 , 0 },
	{ "hoonsquare", 0x3342 , 0 },
	{ "horicoptic", 0x03e9 , 0 },
	{ "horizontalbar", 0x2015 , 0 },
	{ "horncmb", 0x031b , 0 },
	{ "hotsprings", 0x2668 , 0 },
	{ "hparen", 0x24a3 , 0 },
	{ "hsuperior", 0x02b0 , 0 },
	{ "hturned", 0x0265 , 0 },
	{ "huhiragana", 0x3075 , 0 },
	{ "huiitosquare", 0x3333 , 0 },
	{ "hukatakana", 0x30d5 , 0 },
	{ "hukatakanahalfwidth", 0xff8c , 0 },
	{ "hungarumlautcmb", 0x030b , 0 },
	{ "hv", 0x0195 , 0 },
	{ "hypheninferior", 0xf6e5 , 0 },
	{ "hyphenmonospace", 0xff0d , 0 },
	{ "hyphensmall", 0xfe63 , 0 },
	{ "hyphensuperior", 0xf6e6 , 0 },
	{ "hyphentwo", 0x2010 , 0 },
	{ "iacyrillic", 0x044f , 0 },
	{ "ibengali", 0x0987 , 0 },
	{ "ibopomofo", 0x3127 , 0 },
	{ "icaron", 0x01d0 , 0 },
	{ "icircle", 0x24d8 , 0 },
	{ "icyrillic", 0x0456 , 0 },
	{ "idblgrave", 0x0209 , 0 },
	{ "ideographearthcircle", 0x328f , 0 },
	{ "ideographfirecircle", 0x328b , 0 },
	{ "ideographicallianceparen", 0x323f , 0 },
	{ "ideographiccallparen", 0x323a , 0 },
	{ "ideographiccentrecircle", 0x32a5 , 0 },
	{ "ideographicclose", 0x3006 , 0 },
	{ "ideographiccomma", 0x3001 , 0 },
	{ "ideographiccommaleft", 0xff64 , 0 },
	{ "ideographiccongratulationparen", 0x3237 , 0 },
	{ "ideographiccorrectcircle", 0x32a3 , 0 },
	{ "ideographicearthparen", 0x322f , 0 },
	{ "ideographicenterpriseparen", 0x323d , 0 },
	{ "ideographicexcellentcircle", 0x329d , 0 },
	{ "ideographicfestivalparen", 0x3240 , 0 },
	{ "ideographicfinancialcircle", 0x3296 , 0 },
	{ "ideographicfinancialparen", 0x3236 , 0 },
	{ "ideographicfireparen", 0x322b , 0 },
	{ "ideographichaveparen", 0x3232 , 0 },
	{ "ideographichighcircle", 0x32a4 , 0 },
	{ "ideographiciterationmark", 0x3005 , 0 },
	{ "ideographiclaborcircle", 0x3298 , 0 },
	{ "ideographiclaborparen", 0x3238 , 0 },
	{ "ideographicleftcircle", 0x32a7 , 0 },
	{ "ideographiclowcircle", 0x32a6 , 0 },
	{ "ideographicmedicinecircle", 0x32a9 , 0 },
	{ "ideographicmetalparen", 0x322e , 0 },
	{ "ideographicmoonparen", 0x322a , 0 },
	{ "ideographicnameparen", 0x3234 , 0 },
	{ "ideographicperiod", 0x3002 , 0 },
	{ "ideographicprintcircle", 0x329e , 0 },
	{ "ideographicreachparen", 0x3243 , 0 },
	{ "ideographicrepresentparen", 0x3239 , 0 },
	{ "ideographicresourceparen", 0x323e , 0 },
	{ "ideographicrightcircle", 0x32a8 , 0 },
	{ "ideographicsecretcircle", 0x3299 , 0 },
	{ "ideographicselfparen", 0x3242 , 0 },
	{ "ideographicsocietyparen", 0x3233 , 0 },
	{ "ideographicspace", 0x3000 , 0 },
	{ "ideographicspecialparen", 0x3235 , 0 },
	{ "ideographicstockparen", 0x3231 , 0 },
	{ "ideographicstudyparen", 0x323b , 0 },
	{ "ideographicsunparen", 0x3230 , 0 },
	{ "ideographicsuperviseparen", 0x323c , 0 },
	{ "ideographicwaterparen", 0x322c , 0 },
	{ "ideographicwoodparen", 0x322d , 0 },
	{ "ideographiczero", 0x3007 , 0 },
	{ "ideographmetalcircle", 0x328e , 0 },
	{ "ideographmooncircle", 0x328a , 0 },
	{ "ideographnamecircle", 0x3294 , 0 },
	{ "ideographsuncircle", 0x3290 , 0 },
	{ "ideographwatercircle", 0x328c , 0 },
	{ "ideographwoodcircle", 0x328d , 0 },
	{ "ideva", 0x0907 , 0 },
	{ "idieresisacute", 0x1e2f , 0 },
	{ "idieresiscyrillic", 0x04e5 , 0 },
	{ "idotbelow", 0x1ecb , 0 },
	{ "iebrevecyrillic", 0x04d7 , 0 },
	{ "iecyrillic", 0x0435 , 0 },
	{ "ieungacirclekorean", 0x3275 , 0 },
	{ "ieungaparenkorean", 0x3215 , 0 },
	{ "ieungcirclekorean", 0x3267 , 0 },
	{ "ieungkorean", 0x3147 , 0 },
	{ "ieungparenkorean", 0x3207 , 0 },
	{ "igujarati", 0x0a87 , 0 },
	{ "igurmukhi", 0x0a07 , 0 },
	{ "ihiragana", 0x3044 , 0 },
	{ "ihookabove", 0x1ec9 , 0 },
	{ "iibengali", 0x0988 , 0 },
	{ "iicyrillic", 0x0438 , 0 },
	{ "iideva", 0x0908 , 0 },
	{ "iigujarati", 0x0a88 , 0 },
	{ "iigurmukhi", 0x0a08 , 0 },
	{ "iimatragurmukhi", 0x0a40 , 0 },
	{ "iinvertedbreve", 0x020b , 0 },
	{ "iishortcyrillic", 0x0439 , 0 },
	{ "iivowelsignbengali", 0x09c0 , 0 },
	{ "iivowelsigndeva", 0x0940 , 0 },
	{ "iivowelsigngujarati", 0x0ac0 , 0 },
	{ "ikatakana", 0x30a4 , 0 },
	{ "ikatakanahalfwidth", 0xff72 , 0 },
	{ "ikorean", 0x3163 , 0 },
	{ "ilde", 0x02dc , 0 },
	{ "iluyhebrew", 0x05ac , 0 },
	{ "imacroncyrillic", 0x04e3 , 0 },
	{ "imageorapproximatelyequal", 0x2253 , 0 },
	{ "imatragurmukhi", 0x0a3f , 0 },
	{ "imonospace", 0xff49 , 0 },
	{ "increment", 0x2206 , 0 },
	{ "iniarmenian", 0x056b , 0 },
	{ "integralbottom", 0x2321 , 0 },
	{ "integralex", 0xf8f5 , 0 },
	{ "integraltop", 0x2320 , 0 },
	{ "intisquare", 0x3305 , 0 },
	{ "iocyrillic", 0x0451 , 0 },
	{ "iotalatin", 0x0269 , 0 },
	{ "iparen", 0x24a4 , 0 },
	{ "irigurmukhi", 0x0a72 , 0 },
	{ "ismallhiragana", 0x3043 , 0 },
	{ "ismallkatakana", 0x30a3 , 0 },
	{ "ismallkatakanahalfwidth", 0xff68 , 0 },
	{ "issharbengali", 0x09fa , 0 },
	{ "istroke", 0x0268 , 0 },
	{ "isuperior", 0xf6ed , 0 },
	{ "iterationhiragana", 0x309d , 0 },
	{ "iterationkatakana", 0x30fd , 0 },
	{ "itildebelow", 0x1e2d , 0 },
	{ "iubopomofo", 0x3129 , 0 },
	{ "iucyrillic", 0x044e , 0 },
	{ "ivowelsignbengali", 0x09bf , 0 },
	{ "ivowelsigndeva", 0x093f , 0 },
	{ "ivowelsigngujarati", 0x0abf , 0 },
	{ "izhitsacyrillic", 0x0475 , 0 },
	{ "izhitsadblgravecyrillic", 0x0477 , 0 },
	{ "jaarmenian", 0x0571 , 0 },
	{ "jabengali", 0x099c , 0 },
	{ "jadeva", 0x091c , 0 },
	{ "jagujarati", 0x0a9c , 0 },
	{ "jagurmukhi", 0x0a1c , 0 },
	{ "jbopomofo", 0x3110 , 0 },
	{ "jcaron", 0x01f0 , 0 },
	{ "jcircle", 0x24d9 , 0 },
	{ "jcrossedtail", 0x029d , 0 },
	{ "jdotlessstroke", 0x025f , 0 },
	{ "jecyrillic", 0x0458 , 0 },
	{ "jeemarabic", 0x062c , 0 },
	{ "jeemfinalarabic", 0xfe9e , 0 },
	{ "jeeminitialarabic", 0xfe9f , 0 },
	{ "jeemmedialarabic", 0xfea0 , 0 },
	{ "jeharabic", 0x0698 , 0 },
	{ "jehfinalarabic", 0xfb8b , 0 },
	{ "jhabengali", 0x099d , 0 },
	{ "jhadeva", 0x091d , 0 },
	{ "jhagujarati", 0x0a9d , 0 },
	{ "jhagurmukhi", 0x0a1d , 0 },
	{ "jheharmenian", 0x057b , 0 },
	{ "jis", 0x3004 , 0 },
	{ "jmonospace", 0xff4a , 0 },
	{ "jparen", 0x24a5 , 0 },
	{ "jsuperior", 0x02b2 , 0 },
	{ "kabashkircyrillic", 0x04a1 , 0 },
	{ "kabengali", 0x0995 , 0 },
	{ "kacute", 0x1e31 , 0 },
	{ "kacyrillic", 0x043a , 0 },
	{ "kadescendercyrillic", 0x049b , 0 },
	{ "kadeva", 0x0915 , 0 },
	{ "kaf", 0x05db , 0 },
	{ "kafarabic", 0x0643 , 0 },
	{ "kafdagesh", 0xfb3b , 0 },
	{ "kafdageshhebrew", 0xfb3b , 0 },
	{ "kaffinalarabic", 0xfeda , 0 },
	{ "kafhebrew", 0x05db , 0 },
	{ "kafinitialarabic", 0xfedb , 0 },
	{ "kafmedialarabic", 0xfedc , 0 },
	{ "kafrafehebrew", 0xfb4d , 0 },
	{ "kagujarati", 0x0a95 , 0 },
	{ "kagurmukhi", 0x0a15 , 0 },
	{ "kahiragana", 0x304b , 0 },
	{ "kahookcyrillic", 0x04c4 , 0 },
	{ "kakatakana", 0x30ab , 0 },
	{ "kakatakanahalfwidth", 0xff76 , 0 },
	{ "kappasymbolgreek", 0x03f0 , 0 },
	{ "kapyeounmieumkorean", 0x3171 , 0 },
	{ "kapyeounphieuphkorean", 0x3184 , 0 },
	{ "kapyeounpieupkorean", 0x3178 , 0 },
	{ "kapyeounssangpieupkorean", 0x3179 , 0 },
	{ "karoriisquare", 0x330d , 0 },
	{ "kashidaautoarabic", 0x0640 , 0 },
	{ "kashidaautonosidebearingarabic", 0x0640 , 0 },
	{ "kasmallkatakana", 0x30f5 , 0 },
	{ "kasquare", 0x3384 , 0 },
	{ "kasraarabic", 0x0650 , 0 },
	{ "kasratanarabic", 0x064d , 0 },
	{ "kastrokecyrillic", 0x049f , 0 },
	{ "katahiraprolongmarkhalfwidth", 0xff70 , 0 },
	{ "kaverticalstrokecyrillic", 0x049d , 0 },
	{ "kbopomofo", 0x310e , 0 },
	{ "kcalsquare", 0x3389 , 0 },
	{ "kcaron", 0x01e9 , 0 },
	{ "kcedilla", 0x0137 , 0 },
	{ "kcircle", 0x24da , 0 },
	{ "kdotbelow", 0x1e33 , 0 },
	{ "keharmenian", 0x0584 , 0 },
	{ "kehiragana", 0x3051 , 0 },
	{ "kekatakana", 0x30b1 , 0 },
	{ "kekatakanahalfwidth", 0xff79 , 0 },
	{ "kenarmenian", 0x056f , 0 },
	{ "kesmallkatakana", 0x30f6 , 0 },
	{ "khabengali", 0x0996 , 0 },
	{ "khacyrillic", 0x0445 , 0 },
	{ "khadeva", 0x0916 , 0 },
	{ "khagujarati", 0x0a96 , 0 },
	{ "khagurmukhi", 0x0a16 , 0 },
	{ "khaharabic", 0x062e , 0 },
	{ "khahfinalarabic", 0xfea6 , 0 },
	{ "khahinitialarabic", 0xfea7 , 0 },
	{ "khahmedialarabic", 0xfea8 , 0 },
	{ "kheicoptic", 0x03e7 , 0 },
	{ "khhadeva", 0x0959 , 0 },
	{ "khhagurmukhi", 0x0a59 , 0 },
	{ "khieukhacirclekorean", 0x3278 , 0 },
	{ "khieukhaparenkorean", 0x3218 , 0 },
	{ "khieukhcirclekorean", 0x326a , 0 },
	{ "khieukhkorean", 0x314b , 0 },
	{ "khieukhparenkorean", 0x320a , 0 },
	{ "khokhaithai", 0x0e02 , 0 },
	{ "khokhonthai", 0x0e05 , 0 },
	{ "khokhuatthai", 0x0e03 , 0 },
	{ "khokhwaithai", 0x0e04 , 0 },
	{ "khomutthai", 0x0e5b , 0 },
	{ "khook", 0x0199 , 0 },
	{ "khorakhangthai", 0x0e06 , 0 },
	{ "khzsquare", 0x3391 , 0 },
	{ "kihiragana", 0x304d , 0 },
	{ "kikatakana", 0x30ad , 0 },
	{ "kikatakanahalfwidth", 0xff77 , 0 },
	{ "kiroguramusquare", 0x3315 , 0 },
	{ "kiromeetorusquare", 0x3316 , 0 },
	{ "kirosquare", 0x3314 , 0 },
	{ "kiyeokacirclekorean", 0x326e , 0 },
	{ "kiyeokaparenkorean", 0x320e , 0 },
	{ "kiyeokcirclekorean", 0x3260 , 0 },
	{ "kiyeokkorean", 0x3131 , 0 },
	{ "kiyeokparenkorean", 0x3200 , 0 },
	{ "kiyeoksioskorean", 0x3133 , 0 },
	{ "kjecyrillic", 0x045c , 0 },
	{ "klinebelow", 0x1e35 , 0 },
	{ "klsquare", 0x3398 , 0 },
	{ "kmcubedsquare", 0x33a6 , 0 },
	{ "kmonospace", 0xff4b , 0 },
	{ "kmsquaredsquare", 0x33a2 , 0 },
	{ "kohiragana", 0x3053 , 0 },
	{ "kohmsquare", 0x33c0 , 0 },
	{ "kokaithai", 0x0e01 , 0 },
	{ "kokatakana", 0x30b3 , 0 },
	{ "kokatakanahalfwidth", 0xff7a , 0 },
	{ "kooposquare", 0x331e , 0 },
	{ "koppacyrillic", 0x0481 , 0 },
	{ "koreanstandardsymbol", 0x327f , 0 },
	{ "koroniscmb", 0x0343 , 0 },
	{ "kparen", 0x24a6 , 0 },
	{ "kpasquare", 0x33aa , 0 },
	{ "ksicyrillic", 0x046f , 0 },
	{ "ktsquare", 0x33cf , 0 },
	{ "kturned", 0x029e , 0 },
	{ "kuhiragana", 0x304f , 0 },
	{ "kukatakana", 0x30af , 0 },
	{ "kukatakanahalfwidth", 0xff78 , 0 },
	{ "kvsquare", 0x33b8 , 0 },
	{ "kwsquare", 0x33be , 0 },
	{ "labengali", 0x09b2 , 0 },
	{ "ladeva", 0x0932 , 0 },
	{ "lagujarati", 0x0ab2 , 0 },
	{ "lagurmukhi", 0x0a32 , 0 },
	{ "lakkhangyaothai", 0x0e45 , 0 },
	{ "lamaleffinalarabic", 0xfefc , 0 },
	{ "lamalefhamzaabovefinalarabic", 0xfef8 , 0 },
	{ "lamalefhamzaaboveisolatedarabic", 0xfef7 , 0 },
	{ "lamalefhamzabelowfinalarabic", 0xfefa , 0 },
	{ "lamalefhamzabelowisolatedarabic", 0xfef9 , 0 },
	{ "lamalefisolatedarabic", 0xfefb , 0 },
	{ "lamalefmaddaabovefinalarabic", 0xfef6 , 0 },
	{ "lamalefmaddaaboveisolatedarabic", 0xfef5 , 0 },
	{ "lamarabic", 0x0644 , 0 },
	{ "lambdastroke", 0x019b , 0 },
	{ "lamed", 0x05dc , 0 },
	{ "lameddagesh", 0xfb3c , 0 },
	{ "lameddageshhebrew", 0xfb3c , 0 },
	{ "lamedhebrew", 0x05dc , 0 },
	{ "lamedholam", 0x05dc , 0 },
	{ "lamedholamdagesh", 0x05dc , 0 },
	{ "lamedholamdageshhebrew", 0x05dc , 0 },
	{ "lamedholamhebrew", 0x05dc , 0 },
	{ "lamfinalarabic", 0xfede , 0 },
	{ "lamhahinitialarabic", 0xfcca , 0 },
	{ "laminitialarabic", 0xfedf , 0 },
	{ "lamjeeminitialarabic", 0xfcc9 , 0 },
	{ "lamkhahinitialarabic", 0xfccb , 0 },
	{ "lamlamhehisolatedarabic", 0xfdf2 , 0 },
	{ "lammedialarabic", 0xfee0 , 0 },
	{ "lammeemhahinitialarabic", 0xfd88 , 0 },
	{ "lammeeminitialarabic", 0xfccc , 0 },
	{ "lammeemjeeminitialarabic", 0xfedf , 0 },
	{ "lammeemkhahinitialarabic", 0xfedf , 0 },
	{ "largecircle", 0x25ef , 0 },
	{ "lbar", 0x019a , 0 },
	{ "lbelt", 0x026c , 0 },
	{ "lbopomofo", 0x310c , 0 },
	{ "lcedilla", 0x013c , 0 },
	{ "lcircle", 0x24db , 0 },
	{ "lcircumflexbelow", 0x1e3d , 0 },
	{ "ldotaccent", 0x0140 , 0 },
	{ "ldotbelow", 0x1e37 , 0 },
	{ "ldotbelowmacron", 0x1e39 , 0 },
	{ "leftangleabovecmb", 0x031a , 0 },
	{ "lefttackbelowcmb", 0x0318 , 0 },
	{ "lessequalorgreater", 0x22da , 0 },
	{ "lessmonospace", 0xff1c , 0 },
	{ "lessorequivalent", 0x2272 , 0 },
	{ "lessorgreater", 0x2276 , 0 },
	{ "lessoverequal", 0x2266 , 0 },
	{ "lesssmall", 0xfe64 , 0 },
	{ "lezh", 0x026e , 0 },
	{ "lhookretroflex", 0x026d , 0 },
	{ "liwnarmenian", 0x056c , 0 },
	{ "lj", 0x01c9 , 0 },
	{ "ljecyrillic", 0x0459 , 0 },
	{ "ll", 0xf6c0 , 0 },
	{ "lladeva", 0x0933 , 0 },
	{ "llagujarati", 0x0ab3 , 0 },
	{ "llinebelow", 0x1e3b , 0 },
	{ "llladeva", 0x0934 , 0 },
	{ "llvocalicbengali", 0x09e1 , 0 },
	{ "llvocalicdeva", 0x0961 , 0 },
	{ "llvocalicvowelsignbengali", 0x09e3 , 0 },
	{ "llvocalicvowelsigndeva", 0x0963 , 0 },
	{ "lmiddletilde", 0x026b , 0 },
	{ "lmonospace", 0xff4c , 0 },
	{ "lmsquare", 0x33d0 , 0 },
	{ "lochulathai", 0x0e2c , 0 },
	{ "logicalnotreversed", 0x2310 , 0 },
	{ "lolingthai", 0x0e25 , 0 },
	{ "lowlinecenterline", 0xfe4e , 0 },
	{ "lowlinecmb", 0x0332 , 0 },
	{ "lowlinedashed", 0xfe4d , 0 },
	{ "lparen", 0x24a7 , 0 },
	{ "lsquare", 0x2113 , 0 },
	{ "lsuperior", 0xf6ee , 0 },
	{ "luthai", 0x0e26 , 0 },
	{ "lvocalicbengali", 0x098c , 0 },
	{ "lvocalicdeva", 0x090c , 0 },
	{ "lvocalicvowelsignbengali", 0x09e2 , 0 },
	{ "lvocalicvowelsigndeva", 0x0962 , 0 },
	{ "lxsquare", 0x33d3 , 0 },
	{ "mabengali", 0x09ae , 0 },
	{ "macronbelowcmb", 0x0331 , 0 },
	{ "macroncmb", 0x0304 , 0 },
	{ "macronlowmod", 0x02cd , 0 },
	{ "macronmonospace", 0xffe3 , 0 },
	{ "macute", 0x1e3f , 0 },
	{ "madeva", 0x092e , 0 },
	{ "magujarati", 0x0aae , 0 },
	{ "magurmukhi", 0x0a2e , 0 },
	{ "mahapakhhebrew", 0x05a4 , 0 },
	{ "mahapakhlefthebrew", 0x05a4 , 0 },
	{ "mahiragana", 0x307e , 0 },
	{ "maichattawalowleftthai", 0xf895 , 0 },
	{ "maichattawalowrightthai", 0xf894 , 0 },
	{ "maichattawathai", 0x0e4b , 0 },
	{ "maichattawaupperleftthai", 0xf893 , 0 },
	{ "maieklowleftthai", 0xf88c , 0 },
	{ "maieklowrightthai", 0xf88b , 0 },
	{ "maiekthai", 0x0e48 , 0 },
	{ "maiekupperleftthai", 0xf88a , 0 },
	{ "maihanakatleftthai", 0xf884 , 0 },
	{ "maihanakatthai", 0x0e31 , 0 },
	{ "maitaikhuleftthai", 0xf889 , 0 },
	{ "maitaikhuthai", 0x0e47 , 0 },
	{ "maitholowleftthai", 0xf88f , 0 },
	{ "maitholowrightthai", 0xf88e , 0 },
	{ "maithothai", 0x0e49 , 0 },
	{ "maithoupperleftthai", 0xf88d , 0 },
	{ "maitrilowleftthai", 0xf892 , 0 },
	{ "maitrilowrightthai", 0xf891 , 0 },
	{ "maitrithai", 0x0e4a , 0 },
	{ "maitriupperleftthai", 0xf890 , 0 },
	{ "maiyamokthai", 0x0e46 , 0 },
	{ "makatakana", 0x30de , 0 },
	{ "makatakanahalfwidth", 0xff8f , 0 },
	{ "mansyonsquare", 0x3347 , 0 },
	{ "maqafhebrew", 0x05be , 0 },
	{ "mars", 0x2642 , 0 },
	{ "masoracirclehebrew", 0x05af , 0 },
	{ "masquare", 0x3383 , 0 },
	{ "mbopomofo", 0x3107 , 0 },
	{ "mbsquare", 0x33d4 , 0 },
	{ "mcircle", 0x24dc , 0 },
	{ "mcubedsquare", 0x33a5 , 0 },
	{ "mdotaccent", 0x1e41 , 0 },
	{ "mdotbelow", 0x1e43 , 0 },
	{ "meemarabic", 0x0645 , 0 },
	{ "meemfinalarabic", 0xfee2 , 0 },
	{ "meeminitialarabic", 0xfee3 , 0 },
	{ "meemmedialarabic", 0xfee4 , 0 },
	{ "meemmeeminitialarabic", 0xfcd1 , 0 },
	{ "meemmeemisolatedarabic", 0xfc48 , 0 },
	{ "meetorusquare", 0x334d , 0 },
	{ "mehiragana", 0x3081 , 0 },
	{ "meizierasquare", 0x337e , 0 },
	{ "mekatakana", 0x30e1 , 0 },
	{ "mekatakanahalfwidth", 0xff92 , 0 },
	{ "mem", 0x05de , 0 },
	{ "memdagesh", 0xfb3e , 0 },
	{ "memdageshhebrew", 0xfb3e , 0 },
	{ "memhebrew", 0x05de , 0 },
	{ "menarmenian", 0x0574 , 0 },
	{ "merkhahebrew", 0x05a5 , 0 },
	{ "merkhakefulahebrew", 0x05a6 , 0 },
	{ "merkhakefulalefthebrew", 0x05a6 , 0 },
	{ "merkhalefthebrew", 0x05a5 , 0 },
	{ "mhook", 0x0271 , 0 },
	{ "mhzsquare", 0x3392 , 0 },
	{ "middledotkatakanahalfwidth", 0xff65 , 0 },
	{ "middot", 0x00b7 , 0 },
	{ "mieumacirclekorean", 0x3272 , 0 },
	{ "mieumaparenkorean", 0x3212 , 0 },
	{ "mieumcirclekorean", 0x3264 , 0 },
	{ "mieumkorean", 0x3141 , 0 },
	{ "mieumpansioskorean", 0x3170 , 0 },
	{ "mieumparenkorean", 0x3204 , 0 },
	{ "mieumpieupkorean", 0x316e , 0 },
	{ "mieumsioskorean", 0x316f , 0 },
	{ "mihiragana", 0x307f , 0 },
	{ "mikatakana", 0x30df , 0 },
	{ "mikatakanahalfwidth", 0xff90 , 0 },
	{ "minusbelowcmb", 0x0320 , 0 },
	{ "minuscircle", 0x2296 , 0 },
	{ "minusmod", 0x02d7 , 0 },
	{ "minusplus", 0x2213 , 0 },
	{ "miribaarusquare", 0x334a , 0 },
	{ "mirisquare", 0x3349 , 0 },
	{ "mlonglegturned", 0x0270 , 0 },
	{ "mlsquare", 0x3396 , 0 },
	{ "mmcubedsquare", 0x33a3 , 0 },
	{ "mmonospace", 0xff4d , 0 },
	{ "mmsquaredsquare", 0x339f , 0 },
	{ "mohiragana", 0x3082 , 0 },
	{ "mohmsquare", 0x33c1 , 0 },
	{ "mokatakana", 0x30e2 , 0 },
	{ "mokatakanahalfwidth", 0xff93 , 0 },
	{ "molsquare", 0x33d6 , 0 },
	{ "momathai", 0x0e21 , 0 },
	{ "moverssquare", 0x33a7 , 0 },
	{ "moverssquaredsquare", 0x33a8 , 0 },
	{ "mparen", 0x24a8 , 0 },
	{ "mpasquare", 0x33ab , 0 },
	{ "mssquare", 0x33b3 , 0 },
	{ "msuperior", 0xf6ef , 0 },
	{ "mturned", 0x026f , 0 },
	{ "mu1", 0x00b5 , 0 },
	{ "muasquare", 0x3382 , 0 },
	{ "muchgreater", 0x226b , 0 },
	{ "muchless", 0x226a , 0 },
	{ "mufsquare", 0x338c , 0 },
	{ "mugreek", 0x03bc , 0 },
	{ "mugsquare", 0x338d , 0 },
	{ "muhiragana", 0x3080 , 0 },
	{ "mukatakana", 0x30e0 , 0 },
	{ "mukatakanahalfwidth", 0xff91 , 0 },
	{ "mulsquare", 0x3395 , 0 },
	{ "mumsquare", 0x339b , 0 },
	{ "munahhebrew", 0x05a3 , 0 },
	{ "munahlefthebrew", 0x05a3 , 0 },
	{ "musicflatsign", 0x266d , 0 },
	{ "musicsharpsign", 0x266f , 0 },
	{ "mussquare", 0x33b2 , 0 },
	{ "muvsquare", 0x33b6 , 0 },
	{ "muwsquare", 0x33bc , 0 },
	{ "mvmegasquare", 0x33b9 , 0 },
	{ "mvsquare", 0x33b7 , 0 },
	{ "mwmegasquare", 0x33bf , 0 },
	{ "mwsquare", 0x33bd , 0 },
	{ "nabengali", 0x09a8 , 0 },
	{ "nabla", 0x2207 , 0 },
	{ "nadeva", 0x0928 , 0 },
	{ "nagujarati", 0x0aa8 , 0 },
	{ "nagurmukhi", 0x0a28 , 0 },
	{ "nahiragana", 0x306a , 0 },
	{ "nakatakana", 0x30ca , 0 },
	{ "nakatakanahalfwidth", 0xff85 , 0 },
	{ "nasquare", 0x3381 , 0 },
	{ "nbopomofo", 0x310b , 0 },
	{ "nbspace", 0x00a0 , 0 },
	{ "ncedilla", 0x0146 , 0 },
	{ "ncircle", 0x24dd , 0 },
	{ "ncircumflexbelow", 0x1e4b , 0 },
	{ "ndotaccent", 0x1e45 , 0 },
	{ "ndotbelow", 0x1e47 , 0 },
	{ "nehiragana", 0x306d , 0 },
	{ "nekatakana", 0x30cd , 0 },
	{ "nekatakanahalfwidth", 0xff88 , 0 },
	{ "newsheqelsign", 0x20aa , 0 },
	{ "nfsquare", 0x338b , 0 },
	{ "ngabengali", 0x0999 , 0 },
	{ "ngadeva", 0x0919 , 0 },
	{ "ngagujarati", 0x0a99 , 0 },
	{ "ngagurmukhi", 0x0a19 , 0 },
	{ "ngonguthai", 0x0e07 , 0 },
	{ "nhiragana", 0x3093 , 0 },
	{ "nhookleft", 0x0272 , 0 },
	{ "nhookretroflex", 0x0273 , 0 },
	{ "nieunacirclekorean", 0x326f , 0 },
	{ "nieunaparenkorean", 0x320f , 0 },
	{ "nieuncieuckorean", 0x3135 , 0 },
	{ "nieuncirclekorean", 0x3261 , 0 },
	{ "nieunhieuhkorean", 0x3136 , 0 },
	{ "nieunkorean", 0x3134 , 0 },
	{ "nieunpansioskorean", 0x3168 , 0 },
	{ "nieunparenkorean", 0x3201 , 0 },
	{ "nieunsioskorean", 0x3167 , 0 },
	{ "nieuntikeutkorean", 0x3166 , 0 },
	{ "nihiragana", 0x306b , 0 },
	{ "nikatakana", 0x30cb , 0 },
	{ "nikatakanahalfwidth", 0xff86 , 0 },
	{ "nikhahitleftthai", 0xf899 , 0 },
	{ "nikhahitthai", 0x0e4d , 0 },
	{ "ninearabic", 0x0669 , 0 },
	{ "ninebengali", 0x09ef , 0 },
	{ "ninecircle", 0x2468 , 0 },
	{ "ninecircleinversesansserif", 0x2792 , 0 },
	{ "ninedeva", 0x096f , 0 },
	{ "ninegujarati", 0x0aef , 0 },
	{ "ninegurmukhi", 0x0a6f , 0 },
	{ "ninehackarabic", 0x0669 , 0 },
	{ "ninehangzhou", 0x3029 , 0 },
	{ "nineideographicparen", 0x3228 , 0 },
	{ "nineinferior", 0x2089 , 0 },
	{ "ninemonospace", 0xff19 , 0 },
	{ "nineoldstyle", 0xf739 , 0 },
	{ "nineparen", 0x247c , 0 },
	{ "nineperiod", 0x2490 , 0 },
	{ "ninepersian", 0x06f9 , 0 },
	{ "nineroman", 0x2178 , 0 },
	{ "ninesuperior", 0x2079 , 0 },
	{ "nineteencircle", 0x2472 , 0 },
	{ "nineteenparen", 0x2486 , 0 },
	{ "nineteenperiod", 0x249a , 0 },
	{ "ninethai", 0x0e59 , 0 },
	{ "nj", 0x01cc , 0 },
	{ "njecyrillic", 0x045a , 0 },
	{ "nkatakana", 0x30f3 , 0 },
	{ "nkatakanahalfwidth", 0xff9d , 0 },
	{ "nlegrightlong", 0x019e , 0 },
	{ "nlinebelow", 0x1e49 , 0 },
	{ "nmonospace", 0xff4e , 0 },
	{ "nmsquare", 0x339a , 0 },
	{ "nnabengali", 0x09a3 , 0 },
	{ "nnadeva", 0x0923 , 0 },
	{ "nnagujarati", 0x0aa3 , 0 },
	{ "nnagurmukhi", 0x0a23 , 0 },
	{ "nnnadeva", 0x0929 , 0 },
	{ "nohiragana", 0x306e , 0 },
	{ "nokatakana", 0x30ce , 0 },
	{ "nokatakanahalfwidth", 0xff89 , 0 },
	{ "nonbreakingspace", 0x00a0 , 0 },
	{ "nonenthai", 0x0e13 , 0 },
	{ "nonuthai", 0x0e19 , 0 },
	{ "noonarabic", 0x0646 , 0 },
	{ "noonfinalarabic", 0xfee6 , 0 },
	{ "noonghunnaarabic", 0x06ba , 0 },
	{ "noonghunnafinalarabic", 0xfb9f , 0 },
	{ "noonhehinitialarabic", 0xfee7 , 0 },
	{ "nooninitialarabic", 0xfee7 , 0 },
	{ "noonjeeminitialarabic", 0xfcd2 , 0 },
	{ "noonjeemisolatedarabic", 0xfc4b , 0 },
	{ "noonmedialarabic", 0xfee8 , 0 },
	{ "noonmeeminitialarabic", 0xfcd5 , 0 },
	{ "noonmeemisolatedarabic", 0xfc4e , 0 },
	{ "noonnoonfinalarabic", 0xfc8d , 0 },
	{ "notcontains", 0x220c , 0 },
	{ "notelementof", 0x2209 , 0 },
	{ "notgreater", 0x226f , 0 },
	{ "notgreaternorequal", 0x2271 , 0 },
	{ "notgreaternorless", 0x2279 , 0 },
	{ "notidentical", 0x2262 , 0 },
	{ "notless", 0x226e , 0 },
	{ "notlessnorequal", 0x2270 , 0 },
	{ "notparallel", 0x2226 , 0 },
	{ "notprecedes", 0x2280 , 0 },
	{ "notsucceeds", 0x2281 , 0 },
	{ "notsuperset", 0x2285 , 0 },
	{ "nowarmenian", 0x0576 , 0 },
	{ "nparen", 0x24a9 , 0 },
	{ "nssquare", 0x33b1 , 0 },
	{ "nsuperior", 0x207f , 0 },
	{ "nuhiragana", 0x306c , 0 },
	{ "nukatakana", 0x30cc , 0 },
	{ "nukatakanahalfwidth", 0xff87 , 0 },
	{ "nuktabengali", 0x09bc , 0 },
	{ "nuktadeva", 0x093c , 0 },
	{ "nuktagujarati", 0x0abc , 0 },
	{ "nuktagurmukhi", 0x0a3c , 0 },
	{ "numbersignmonospace", 0xff03 , 0 },
	{ "numbersignsmall", 0xfe5f , 0 },
	{ "numeralsigngreek", 0x0374 , 0 },
	{ "numeralsignlowergreek", 0x0375 , 0 },
	{ "numero", 0x2116 , 0 },
	{ "nun", 0x05e0 , 0 },
	{ "nundagesh", 0xfb40 , 0 },
	{ "nundageshhebrew", 0xfb40 , 0 },
	{ "nunhebrew", 0x05e0 , 0 },
	{ "nvsquare", 0x33b5 , 0 },
	{ "nwsquare", 0x33bb , 0 },
	{ "nyabengali", 0x099e , 0 },
	{ "nyadeva", 0x091e , 0 },
	{ "nyagujarati", 0x0a9e , 0 },
	{ "nyagurmukhi", 0x0a1e , 0 },
	{ "oangthai", 0x0e2d , 0 },
	{ "obarred", 0x0275 , 0 },
	{ "obarredcyrillic", 0x04e9 , 0 },
	{ "obarreddieresiscyrillic", 0x04eb , 0 },
	{ "obengali", 0x0993 , 0 },
	{ "obopomofo", 0x311b , 0 },
	{ "ocandradeva", 0x0911 , 0 },
	{ "ocandragujarati", 0x0a91 , 0 },
	{ "ocandravowelsigndeva", 0x0949 , 0 },
	{ "ocandravowelsigngujarati", 0x0ac9 , 0 },
	{ "ocaron", 0x01d2 , 0 },
	{ "ocircle", 0x24de , 0 },
	{ "ocircumflexacute", 0x1ed1 , 0 },
	{ "ocircumflexdotbelow", 0x1ed9 , 0 },
	{ "ocircumflexgrave", 0x1ed3 , 0 },
	{ "ocircumflexhookabove", 0x1ed5 , 0 },
	{ "ocircumflextilde", 0x1ed7 , 0 },
	{ "ocyrillic", 0x043e , 0 },
	{ "odblacute", 0x0151 , 0 },
	{ "odblgrave", 0x020d , 0 },
	{ "odeva", 0x0913 , 0 },
	{ "odieresiscyrillic", 0x04e7 , 0 },
	{ "odotbelow", 0x1ecd , 0 },
	{ "oekorean", 0x315a , 0 },
	{ "ogonekcmb", 0x0328 , 0 },
	{ "ogujarati", 0x0a93 , 0 },
	{ "oharmenian", 0x0585 , 0 },
	{ "ohiragana", 0x304a , 0 },
	{ "ohookabove", 0x1ecf , 0 },
	{ "ohornacute", 0x1edb , 0 },
	{ "ohorndotbelow", 0x1ee3 , 0 },
	{ "ohorngrave", 0x1edd , 0 },
	{ "ohornhookabove", 0x1edf , 0 },
	{ "ohorntilde", 0x1ee1 , 0 },
	{ "oi", 0x01a3 , 0 },
	{ "oinvertedbreve", 0x020f , 0 },
	{ "okatakana", 0x30aa , 0 },
	{ "okatakanahalfwidth", 0xff75 , 0 },
	{ "okorean", 0x3157 , 0 },
	{ "olehebrew", 0x05ab , 0 },
	{ "omacronacute", 0x1e53 , 0 },
	{ "omacrongrave", 0x1e51 , 0 },
	{ "omdeva", 0x0950 , 0 },
	{ "omegacyrillic", 0x0461 , 0 },
	{ "omegalatinclosed", 0x0277 , 0 },
	{ "omegaroundcyrillic", 0x047b , 0 },
	{ "omegatitlocyrillic", 0x047d , 0 },
	{ "omgujarati", 0x0ad0 , 0 },
	{ "omonospace", 0xff4f , 0 },
	{ "onearabic", 0x0661 , 0 },
	{ "onebengali", 0x09e7 , 0 },
	{ "onecircle", 0x2460 , 0 },
	{ "onecircleinversesansserif", 0x278a , 0 },
	{ "onedeva", 0x0967 , 0 },
	{ "onefitted", 0xf6dc , 0 },
	{ "onegujarati", 0x0ae7 , 0 },
	{ "onegurmukhi", 0x0a67 , 0 },
	{ "onehackarabic", 0x0661 , 0 },
	{ "onehangzhou", 0x3021 , 0 },
	{ "oneideographicparen", 0x3220 , 0 },
	{ "oneinferior", 0x2081 , 0 },
	{ "onemonospace", 0xff11 , 0 },
	{ "onenumeratorbengali", 0x09f4 , 0 },
	{ "oneoldstyle", 0xf731 , 0 },
	{ "oneparen", 0x2474 , 0 },
	{ "oneperiod", 0x2488 , 0 },
	{ "onepersian", 0x06f1 , 0 },
	{ "oneroman", 0x2170 , 0 },
	{ "onesuperior", 0x00b9 , 0 },
	{ "onethai", 0x0e51 , 0 },
	{ "oogonek", 0x01eb , 0 },
	{ "oogonekmacron", 0x01ed , 0 },
	{ "oogurmukhi", 0x0a13 , 0 },
	{ "oomatragurmukhi", 0x0a4b , 0 },
	{ "oopen", 0x0254 , 0 },
	{ "oparen", 0x24aa , 0 },
	{ "option", 0x2325 , 0 },
	{ "oshortdeva", 0x0912 , 0 },
	{ "oshortvowelsigndeva", 0x094a , 0 },
	{ "osmallhiragana", 0x3049 , 0 },
	{ "osmallkatakana", 0x30a9 , 0 },
	{ "osmallkatakanahalfwidth", 0xff6b , 0 },
	{ "ostrokeacute", 0x01ff , 0 },
	{ "osuperior", 0xf6f0 , 0 },
	{ "otcyrillic", 0x047f , 0 },
	{ "otildeacute", 0x1e4d , 0 },
	{ "otildedieresis", 0x1e4f , 0 },
	{ "oubopomofo", 0x3121 , 0 },
	{ "overline", 0x203e , 0 },
	{ "overlinecenterline", 0xfe4a , 0 },
	{ "overlinecmb", 0x0305 , 0 },
	{ "overlinedashed", 0xfe49 , 0 },
	{ "overlinedblwavy", 0xfe4c , 0 },
	{ "overlinewavy", 0xfe4b , 0 },
	{ "overscore", 0x00af , 0 },
	{ "ovowelsignbengali", 0x09cb , 0 },
	{ "ovowelsigndeva", 0x094b , 0 },
	{ "ovowelsigngujarati", 0x0acb , 0 },
	{ "paampssquare", 0x3380 , 0 },
	{ "paasentosquare", 0x332b , 0 },
	{ "pabengali", 0x09aa , 0 },
	{ "pacute", 0x1e55 , 0 },
	{ "padeva", 0x092a , 0 },
	{ "pagedown", 0x21df , 0 },
	{ "pageup", 0x21de , 0 },
	{ "pagujarati", 0x0aaa , 0 },
	{ "pagurmukhi", 0x0a2a , 0 },
	{ "pahiragana", 0x3071 , 0 },
	{ "paiyannoithai", 0x0e2f , 0 },
	{ "pakatakana", 0x30d1 , 0 },
	{ "palatalizationcyrilliccmb", 0x0484 , 0 },
	{ "palochkacyrillic", 0x04c0 , 0 },
	{ "pansioskorean", 0x317f , 0 },
	{ "parallel", 0x2225 , 0 },
	{ "parenleftaltonearabic", 0xfd3e , 0 },
	{ "parenleftbt", 0xf8ed , 0 },
	{ "parenleftex", 0xf8ec , 0 },
	{ "parenleftinferior", 0x208d , 0 },
	{ "parenleftmonospace", 0xff08 , 0 },
	{ "parenleftsmall", 0xfe59 , 0 },
	{ "parenleftsuperior", 0x207d , 0 },
	{ "parenlefttp", 0xf8eb , 0 },
	{ "parenleftvertical", 0xfe35 , 0 },
	{ "parenrightaltonearabic", 0xfd3f , 0 },
	{ "parenrightbt", 0xf8f8 , 0 },
	{ "parenrightex", 0xf8f7 , 0 },
	{ "parenrightinferior", 0x208e , 0 },
	{ "parenrightmonospace", 0xff09 , 0 },
	{ "parenrightsmall", 0xfe5a , 0 },
	{ "parenrightsuperior", 0x207e , 0 },
	{ "parenrighttp", 0xf8f6 , 0 },
	{ "parenrightvertical", 0xfe36 , 0 },
	{ "paseqhebrew", 0x05c0 , 0 },
	{ "pashtahebrew", 0x0599 , 0 },
	{ "pasquare", 0x33a9 , 0 },
	{ "patah", 0x05b7 , 0 },
	{ "patah11", 0x05b7 , 0 },
	{ "patah1d", 0x05b7 , 0 },
	{ "patah2a", 0x05b7 , 0 },
	{ "patahhebrew", 0x05b7 , 0 },
	{ "patahnarrowhebrew", 0x05b7 , 0 },
	{ "patahquarterhebrew", 0x05b7 , 0 },
	{ "patahwidehebrew", 0x05b7 , 0 },
	{ "pazerhebrew", 0x05a1 , 0 },
	{ "pbopomofo", 0x3106 , 0 },
	{ "pcircle", 0x24df , 0 },
	{ "pdotaccent", 0x1e57 , 0 },
	{ "pe", 0x05e4 , 0 },
	{ "pecyrillic", 0x043f , 0 },
	{ "pedagesh", 0xfb44 , 0 },
	{ "pedageshhebrew", 0xfb44 , 0 },
	{ "peezisquare", 0x333b , 0 },
	{ "pefinaldageshhebrew", 0xfb43 , 0 },
	{ "peharabic", 0x067e , 0 },
	{ "peharmenian", 0x057a , 0 },
	{ "pehebrew", 0x05e4 , 0 },
	{ "pehfinalarabic", 0xfb57 , 0 },
	{ "pehinitialarabic", 0xfb58 , 0 },
	{ "pehiragana", 0x307a , 0 },
	{ "pehmedialarabic", 0xfb59 , 0 },
	{ "pekatakana", 0x30da , 0 },
	{ "pemiddlehookcyrillic", 0x04a7 , 0 },
	{ "perafehebrew", 0xfb4e , 0 },
	{ "percentarabic", 0x066a , 0 },
	{ "percentmonospace", 0xff05 , 0 },
	{ "percentsmall", 0xfe6a , 0 },
	{ "periodarmenian", 0x0589 , 0 },
	{ "periodhalfwidth", 0xff61 , 0 },
	{ "periodinferior", 0xf6e7 , 0 },
	{ "periodmonospace", 0xff0e , 0 },
	{ "periodsmall", 0xfe52 , 0 },
	{ "periodsuperior", 0xf6e8 , 0 },
	{ "perispomenigreekcmb", 0x0342 , 0 },
	{ "pfsquare", 0x338a , 0 },
	{ "phabengali", 0x09ab , 0 },
	{ "phadeva", 0x092b , 0 },
	{ "phagujarati", 0x0aab , 0 },
	{ "phagurmukhi", 0x0a2b , 0 },
	{ "phieuphacirclekorean", 0x327a , 0 },
	{ "phieuphaparenkorean", 0x321a , 0 },
	{ "phieuphcirclekorean", 0x326c , 0 },
	{ "phieuphkorean", 0x314d , 0 },
	{ "phieuphparenkorean", 0x320c , 0 },
	{ "philatin", 0x0278 , 0 },
	{ "phinthuthai", 0x0e3a , 0 },
	{ "phisymbolgreek", 0x03d5 , 0 },
	{ "phook", 0x01a5 , 0 },
	{ "phophanthai", 0x0e1e , 0 },
	{ "phophungthai", 0x0e1c , 0 },
	{ "phosamphaothai", 0x0e20 , 0 },
	{ "pieupacirclekorean", 0x3273 , 0 },
	{ "pieupaparenkorean", 0x3213 , 0 },
	{ "pieupcieuckorean", 0x3176 , 0 },
	{ "pieupcirclekorean", 0x3265 , 0 },
	{ "pieupkiyeokkorean", 0x3172 , 0 },
	{ "pieupkorean", 0x3142 , 0 },
	{ "pieupparenkorean", 0x3205 , 0 },
	{ "pieupsioskiyeokkorean", 0x3174 , 0 },
	{ "pieupsioskorean", 0x3144 , 0 },
	{ "pieupsiostikeutkorean", 0x3175 , 0 },
	{ "pieupthieuthkorean", 0x3177 , 0 },
	{ "pieuptikeutkorean", 0x3173 , 0 },
	{ "pihiragana", 0x3074 , 0 },
	{ "pikatakana", 0x30d4 , 0 },
	{ "pisymbolgreek", 0x03d6 , 0 },
	{ "piwrarmenian", 0x0583 , 0 },
	{ "plusbelowcmb", 0x031f , 0 },
	{ "pluscircle", 0x2295 , 0 },
	{ "plusmod", 0x02d6 , 0 },
	{ "plusmonospace", 0xff0b , 0 },
	{ "plussmall", 0xfe62 , 0 },
	{ "plussuperior", 0x207a , 0 },
	{ "pmonospace", 0xff50 , 0 },
	{ "pmsquare", 0x33d8 , 0 },
	{ "pohiragana", 0x307d , 0 },
	{ "pointingindexdownwhite", 0x261f , 0 },
	{ "pointingindexleftwhite", 0x261c , 0 },
	{ "pointingindexrightwhite", 0x261e , 0 },
	{ "pointingindexupwhite", 0x261d , 0 },
	{ "pokatakana", 0x30dd , 0 },
	{ "poplathai", 0x0e1b , 0 },
	{ "postalmark", 0x3012 , 0 },
	{ "postalmarkface", 0x3020 , 0 },
	{ "pparen", 0x24ab , 0 },
	{ "precedes", 0x227a , 0 },
	{ "primemod", 0x02b9 , 0 },
	{ "primereversed", 0x2035 , 0 },
	{ "projective", 0x2305 , 0 },
	{ "prolongedkana", 0x30fc , 0 },
	{ "propellor", 0x2318 , 0 },
	{ "proportion", 0x2237 , 0 },
	{ "psicyrillic", 0x0471 , 0 },
	{ "psilipneumatacyrilliccmb", 0x0486 , 0 },
	{ "pssquare", 0x33b0 , 0 },
	{ "puhiragana", 0x3077 , 0 },
	{ "pukatakana", 0x30d7 , 0 },
	{ "pvsquare", 0x33b4 , 0 },
	{ "pwsquare", 0x33ba , 0 },
	{ "qadeva", 0x0958 , 0 },
	{ "qadmahebrew", 0x05a8 , 0 },
	{ "qafarabic", 0x0642 , 0 },
	{ "qaffinalarabic", 0xfed6 , 0 },
	{ "qafinitialarabic", 0xfed7 , 0 },
	{ "qafmedialarabic", 0xfed8 , 0 },
	{ "qamats", 0x05b8 , 0 },
	{ "qamats10", 0x05b8 , 0 },
	{ "qamats1a", 0x05b8 , 0 },
	{ "qamats1c", 0x05b8 , 0 },
	{ "qamats27", 0x05b8 , 0 },
	{ "qamats29", 0x05b8 , 0 },
	{ "qamats33", 0x05b8 , 0 },
	{ "qamatsde", 0x05b8 , 0 },
	{ "qamatshebrew", 0x05b8 , 0 },
	{ "qamatsnarrowhebrew", 0x05b8 , 0 },
	{ "qamatsqatanhebrew", 0x05b8 , 0 },
	{ "qamatsqatannarrowhebrew", 0x05b8 , 0 },
	{ "qamatsqatanquarterhebrew", 0x05b8 , 0 },
	{ "qamatsqatanwidehebrew", 0x05b8 , 0 },
	{ "qamatsquarterhebrew", 0x05b8 , 0 },
	{ "qamatswidehebrew", 0x05b8 , 0 },
	{ "qarneyparahebrew", 0x059f , 0 },
	{ "qbopomofo", 0x3111 , 0 },
	{ "qcircle", 0x24e0 , 0 },
	{ "qhook", 0x02a0 , 0 },
	{ "qmonospace", 0xff51 , 0 },
	{ "qof", 0x05e7 , 0 },
	{ "qofdagesh", 0xfb47 , 0 },
	{ "qofdageshhebrew", 0xfb47 , 0 },
	{ "qofhatafpatah", 0x05e7 , 0 },
	{ "qofhatafpatahhebrew", 0x05e7 , 0 },
	{ "qofhatafsegol", 0x05e7 , 0 },
	{ "qofhatafsegolhebrew", 0x05e7 , 0 },
	{ "qofhebrew", 0x05e7 , 0 },
	{ "qofhiriq", 0x05e7 , 0 },
	{ "qofhiriqhebrew", 0x05e7 , 0 },
	{ "qofholam", 0x05e7 , 0 },
	{ "qofholamhebrew", 0x05e7 , 0 },
	{ "qofpatah", 0x05e7 , 0 },
	{ "qofpatahhebrew", 0x05e7 , 0 },
	{ "qofqamats", 0x05e7 , 0 },
	{ "qofqamatshebrew", 0x05e7 , 0 },
	{ "qofqubuts", 0x05e7 , 0 },
	{ "qofqubutshebrew", 0x05e7 , 0 },
	{ "qofsegol", 0x05e7 , 0 },
	{ "qofsegolhebrew", 0x05e7 , 0 },
	{ "qofsheva", 0x05e7 , 0 },
	{ "qofshevahebrew", 0x05e7 , 0 },
	{ "qoftsere", 0x05e7 , 0 },
	{ "qoftserehebrew", 0x05e7 , 0 },
	{ "qparen", 0x24ac , 0 },
	{ "quarternote", 0x2669 , 0 },
	{ "qubuts", 0x05bb , 0 },
	{ "qubuts18", 0x05bb , 0 },
	{ "qubuts25", 0x05bb , 0 },
	{ "qubuts31", 0x05bb , 0 },
	{ "qubutshebrew", 0x05bb , 0 },
	{ "qubutsnarrowhebrew", 0x05bb , 0 },
	{ "qubutsquarterhebrew", 0x05bb , 0 },
	{ "qubutswidehebrew", 0x05bb , 0 },
	{ "questionarabic", 0x061f , 0 },
	{ "questionarmenian", 0x055e , 0 },
	{ "questiondownsmall", 0xf7bf , 0 },
	{ "questiongreek", 0x037e , 0 },
	{ "questionmonospace", 0xff1f , 0 },
	{ "questionsmall", 0xf73f , 0 },
	{ "quotedblmonospace", 0xff02 , 0 },
	{ "quotedblprime", 0x301e , 0 },
	{ "quotedblprimereversed", 0x301d , 0 },
	{ "quoteleftreversed", 0x201b , 0 },
	{ "quoterightn", 0x0149 , 0 },
	{ "quotesinglemonospace", 0xff07 , 0 },
	{ "raarmenian", 0x057c , 0 },
	{ "rabengali", 0x09b0 , 0 },
	{ "radeva", 0x0930 , 0 },
	{ "radicalex", 0xf8e5 , 0 },
	{ "radoverssquare", 0x33ae , 0 },
	{ "radoverssquaredsquare", 0x33af , 0 },
	{ "radsquare", 0x33ad , 0 },
	{ "rafe", 0x05bf , 0 },
	{ "rafehebrew", 0x05bf , 0 },
	{ "ragujarati", 0x0ab0 , 0 },
	{ "ragurmukhi", 0x0a30 , 0 },
	{ "rahiragana", 0x3089 , 0 },
	{ "rakatakana", 0x30e9 , 0 },
	{ "rakatakanahalfwidth", 0xff97 , 0 },
	{ "ralowerdiagonalbengali", 0x09f1 , 0 },
	{ "ramiddlediagonalbengali", 0x09f0 , 0 },
	{ "ramshorn", 0x0264 , 0 },
	{ "ratio", 0x2236 , 0 },
	{ "rbopomofo", 0x3116 , 0 },
	{ "rcedilla", 0x0157 , 0 },
	{ "rcircle", 0x24e1 , 0 },
	{ "rdblgrave", 0x0211 , 0 },
	{ "rdotaccent", 0x1e59 , 0 },
	{ "rdotbelow", 0x1e5b , 0 },
	{ "rdotbelowmacron", 0x1e5d , 0 },
	{ "referencemark", 0x203b , 0 },
	{ "registersans", 0xf8e8 , 0 },
	{ "registerserif", 0xf6da , 0 },
	{ "reharabic", 0x0631 , 0 },
	{ "reharmenian", 0x0580 , 0 },
	{ "rehfinalarabic", 0xfeae , 0 },
	{ "rehiragana", 0x308c , 0 },
	{ "rehyehaleflamarabic", 0x0631 , 0 },
	{ "rekatakana", 0x30ec , 0 },
	{ "rekatakanahalfwidth", 0xff9a , 0 },
	{ "resh", 0x05e8 , 0 },
	{ "reshdageshhebrew", 0xfb48 , 0 },
	{ "reshhatafpatah", 0x05e8 , 0 },
	{ "reshhatafpatahhebrew", 0x05e8 , 0 },
	{ "reshhatafsegol", 0x05e8 , 0 },
	{ "reshhatafsegolhebrew", 0x05e8 , 0 },
	{ "reshhebrew", 0x05e8 , 0 },
	{ "reshhiriq", 0x05e8 , 0 },
	{ "reshhiriqhebrew", 0x05e8 , 0 },
	{ "reshholam", 0x05e8 , 0 },
	{ "reshholamhebrew", 0x05e8 , 0 },
	{ "reshpatah", 0x05e8 , 0 },
	{ "reshpatahhebrew", 0x05e8 , 0 },
	{ "reshqamats", 0x05e8 , 0 },
	{ "reshqamatshebrew", 0x05e8 , 0 },
	{ "reshqubuts", 0x05e8 , 0 },
	{ "reshqubutshebrew", 0x05e8 , 0 },
	{ "reshsegol", 0x05e8 , 0 },
	{ "reshsegolhebrew", 0x05e8 , 0 },
	{ "reshsheva", 0x05e8 , 0 },
	{ "reshshevahebrew", 0x05e8 , 0 },
	{ "reshtsere", 0x05e8 , 0 },
	{ "reshtserehebrew", 0x05e8 , 0 },
	{ "reversedtilde", 0x223d , 0 },
	{ "reviahebrew", 0x0597 , 0 },
	{ "reviamugrashhebrew", 0x0597 , 0 },
	{ "rfishhook", 0x027e , 0 },
	{ "rfishhookreversed", 0x027f , 0 },
	{ "rhabengali", 0x09dd , 0 },
	{ "rhadeva", 0x095d , 0 },
	{ "rhook", 0x027d , 0 },
	{ "rhookturned", 0x027b , 0 },
	{ "rhookturnedsuperior", 0x02b5 , 0 },
	{ "rhosymbolgreek", 0x03f1 , 0 },
	{ "rhotichookmod", 0x02de , 0 },
	{ "rieulacirclekorean", 0x3271 , 0 },
	{ "rieulaparenkorean", 0x3211 , 0 },
	{ "rieulcirclekorean", 0x3263 , 0 },
	{ "rieulhieuhkorean", 0x3140 , 0 },
	{ "rieulkiyeokkorean", 0x313a , 0 },
	{ "rieulkiyeoksioskorean", 0x3169 , 0 },
	{ "rieulkorean", 0x3139 , 0 },
	{ "rieulmieumkorean", 0x313b , 0 },
	{ "rieulpansioskorean", 0x316c , 0 },
	{ "rieulparenkorean", 0x3203 , 0 },
	{ "rieulphieuphkorean", 0x313f , 0 },
	{ "rieulpieupkorean", 0x313c , 0 },
	{ "rieulpieupsioskorean", 0x316b , 0 },
	{ "rieulsioskorean", 0x313d , 0 },
	{ "rieulthieuthkorean", 0x313e , 0 },
	{ "rieultikeutkorean", 0x316a , 0 },
	{ "rieulyeorinhieuhkorean", 0x316d , 0 },
	{ "rightangle", 0x221f , 0 },
	{ "righttackbelowcmb", 0x0319 , 0 },
	{ "righttriangle", 0x22bf , 0 },
	{ "rihiragana", 0x308a , 0 },
	{ "rikatakana", 0x30ea , 0 },
	{ "rikatakanahalfwidth", 0xff98 , 0 },
	{ "ringbelowcmb", 0x0325 , 0 },
	{ "ringcmb", 0x030a , 0 },
	{ "ringhalfleft", 0x02bf , 0 },
	{ "ringhalfleftarmenian", 0x0559 , 0 },
	{ "ringhalfleftbelowcmb", 0x031c , 0 },
	{ "ringhalfleftcentered", 0x02d3 , 0 },
	{ "ringhalfright", 0x02be , 0 },
	{ "ringhalfrightbelowcmb", 0x0339 , 0 },
	{ "ringhalfrightcentered", 0x02d2 , 0 },
	{ "rinvertedbreve", 0x0213 , 0 },
	{ "rittorusquare", 0x3351 , 0 },
	{ "rlinebelow", 0x1e5f , 0 },
	{ "rlongleg", 0x027c , 0 },
	{ "rlonglegturned", 0x027a , 0 },
	{ "rmonospace", 0xff52 , 0 },
	{ "rohiragana", 0x308d , 0 },
	{ "rokatakana", 0x30ed , 0 },
	{ "rokatakanahalfwidth", 0xff9b , 0 },
	{ "roruathai", 0x0e23 , 0 },
	{ "rparen", 0x24ad , 0 },
	{ "rrabengali", 0x09dc , 0 },
	{ "rradeva", 0x0931 , 0 },
	{ "rragurmukhi", 0x0a5c , 0 },
	{ "rreharabic", 0x0691 , 0 },
	{ "rrehfinalarabic", 0xfb8d , 0 },
	{ "rrvocalicbengali", 0x09e0 , 0 },
	{ "rrvocalicdeva", 0x0960 , 0 },
	{ "rrvocalicgujarati", 0x0ae0 , 0 },
	{ "rrvocalicvowelsignbengali", 0x09c4 , 0 },
	{ "rrvocalicvowelsigndeva", 0x0944 , 0 },
	{ "rrvocalicvowelsigngujarati", 0x0ac4 , 0 },
	{ "rsuperior", 0xf6f1 , 0 },
	{ "rturned", 0x0279 , 0 },
	{ "rturnedsuperior", 0x02b4 , 0 },
	{ "ruhiragana", 0x308b , 0 },
	{ "rukatakana", 0x30eb , 0 },
	{ "rukatakanahalfwidth", 0xff99 , 0 },
	{ "rupeemarkbengali", 0x09f2 , 0 },
	{ "rupeesignbengali", 0x09f3 , 0 },
	{ "rupiah", 0xf6dd , 0 },
	{ "ruthai", 0x0e24 , 0 },
	{ "rvocalicbengali", 0x098b , 0 },
	{ "rvocalicdeva", 0x090b , 0 },
	{ "rvocalicgujarati", 0x0a8b , 0 },
	{ "rvocalicvowelsignbengali", 0x09c3 , 0 },
	{ "rvocalicvowelsigndeva", 0x0943 , 0 },
	{ "rvocalicvowelsigngujarati", 0x0ac3 , 0 },
	{ "sabengali", 0x09b8 , 0 },
	{ "sacutedotaccent", 0x1e65 , 0 },
	{ "sadarabic", 0x0635 , 0 },
	{ "sadeva", 0x0938 , 0 },
	{ "sadfinalarabic", 0xfeba , 0 },
	{ "sadinitialarabic", 0xfebb , 0 },
	{ "sadmedialarabic", 0xfebc , 0 },
	{ "sagujarati", 0x0ab8 , 0 },
	{ "sagurmukhi", 0x0a38 , 0 },
	{ "sahiragana", 0x3055 , 0 },
	{ "sakatakana", 0x30b5 , 0 },
	{ "sakatakanahalfwidth", 0xff7b , 0 },
	{ "sallallahoualayhewasallamarabic", 0xfdfa , 0 },
	{ "samekh", 0x05e1 , 0 },
	{ "samekhdagesh", 0xfb41 , 0 },
	{ "samekhdageshhebrew", 0xfb41 , 0 },
	{ "samekhhebrew", 0x05e1 , 0 },
	{ "saraaathai", 0x0e32 , 0 },
	{ "saraaethai", 0x0e41 , 0 },
	{ "saraaimaimalaithai", 0x0e44 , 0 },
	{ "saraaimaimuanthai", 0x0e43 , 0 },
	{ "saraamthai", 0x0e33 , 0 },
	{ "saraathai", 0x0e30 , 0 },
	{ "saraethai", 0x0e40 , 0 },
	{ "saraiileftthai", 0xf886 , 0 },
	{ "saraiithai", 0x0e35 , 0 },
	{ "saraileftthai", 0xf885 , 0 },
	{ "saraithai", 0x0e34 , 0 },
	{ "saraothai", 0x0e42 , 0 },
	{ "saraueeleftthai", 0xf888 , 0 },
	{ "saraueethai", 0x0e37 , 0 },
	{ "saraueleftthai", 0xf887 , 0 },
	{ "sarauethai", 0x0e36 , 0 },
	{ "sarauthai", 0x0e38 , 0 },
	{ "sarauuthai", 0x0e39 , 0 },
	{ "sbopomofo", 0x3119 , 0 },
	{ "scarondotaccent", 0x1e67 , 0 },
	{ "schwa", 0x0259 , 0 },
	{ "schwacyrillic", 0x04d9 , 0 },
	{ "schwadieresiscyrillic", 0x04db , 0 },
	{ "schwahook", 0x025a , 0 },
	{ "scircle", 0x24e2 , 0 },
	{ "sdotaccent", 0x1e61 , 0 },
	{ "sdotbelow", 0x1e63 , 0 },
	{ "sdotbelowdotaccent", 0x1e69 , 0 },
	{ "seagullbelowcmb", 0x033c , 0 },
	{ "secondtonechinese", 0x02ca , 0 },
	{ "seenarabic", 0x0633 , 0 },
	{ "seenfinalarabic", 0xfeb2 , 0 },
	{ "seeninitialarabic", 0xfeb3 , 0 },
	{ "seenmedialarabic", 0xfeb4 , 0 },
	{ "segol", 0x05b6 , 0 },
	{ "segol13", 0x05b6 , 0 },
	{ "segol1f", 0x05b6 , 0 },
	{ "segol2c", 0x05b6 , 0 },
	{ "segolhebrew", 0x05b6 , 0 },
	{ "segolnarrowhebrew", 0x05b6 , 0 },
	{ "segolquarterhebrew", 0x05b6 , 0 },
	{ "segoltahebrew", 0x0592 , 0 },
	{ "segolwidehebrew", 0x05b6 , 0 },
	{ "seharmenian", 0x057d , 0 },
	{ "sehiragana", 0x305b , 0 },
	{ "sekatakana", 0x30bb , 0 },
	{ "sekatakanahalfwidth", 0xff7e , 0 },
	{ "semicolonarabic", 0x061b , 0 },
	{ "semicolonmonospace", 0xff1b , 0 },
	{ "semicolonsmall", 0xfe54 , 0 },
	{ "semivoicedmarkkana", 0x309c , 0 },
	{ "semivoicedmarkkanahalfwidth", 0xff9f , 0 },
	{ "sentisquare", 0x3322 , 0 },
	{ "sentosquare", 0x3323 , 0 },
	{ "sevenarabic", 0x0667 , 0 },
	{ "sevenbengali", 0x09ed , 0 },
	{ "sevencircle", 0x2466 , 0 },
	{ "sevencircleinversesansserif", 0x2790 , 0 },
	{ "sevendeva", 0x096d , 0 },
	{ "sevengujarati", 0x0aed , 0 },
	{ "sevengurmukhi", 0x0a6d , 0 },
	{ "sevenhackarabic", 0x0667 , 0 },
	{ "sevenhangzhou", 0x3027 , 0 },
	{ "sevenideographicparen", 0x3226 , 0 },
	{ "seveninferior", 0x2087 , 0 },
	{ "sevenmonospace", 0xff17 , 0 },
	{ "sevenoldstyle", 0xf737 , 0 },
	{ "sevenparen", 0x247a , 0 },
	{ "sevenperiod", 0x248e , 0 },
	{ "sevenpersian", 0x06f7 , 0 },
	{ "sevenroman", 0x2176 , 0 },
	{ "sevensuperior", 0x2077 , 0 },
	{ "seventeencircle", 0x2470 , 0 },
	{ "seventeenparen", 0x2484 , 0 },
	{ "seventeenperiod", 0x2498 , 0 },
	{ "seventhai", 0x0e57 , 0 },
	{ "sfthyphen", 0x00ad , 0 },
	{ "shaarmenian", 0x0577 , 0 },
	{ "shabengali", 0x09b6 , 0 },
	{ "shacyrillic", 0x0448 , 0 },
	{ "shaddaarabic", 0x0651 , 0 },
	{ "shaddadammaarabic", 0xfc61 , 0 },
	{ "shaddadammatanarabic", 0xfc5e , 0 },
	{ "shaddafathaarabic", 0xfc60 , 0 },
	{ "shaddafathatanarabic", 0x0651 , 0 },
	{ "shaddakasraarabic", 0xfc62 , 0 },
	{ "shaddakasratanarabic", 0xfc5f , 0 },
	{ "shadedark", 0x2593 , 0 },
	{ "shadelight", 0x2591 , 0 },
	{ "shademedium", 0x2592 , 0 },
	{ "shadeva", 0x0936 , 0 },
	{ "shagujarati", 0x0ab6 , 0 },
	{ "shagurmukhi", 0x0a36 , 0 },
	{ "shalshelethebrew", 0x0593 , 0 },
	{ "shbopomofo", 0x3115 , 0 },
	{ "shchacyrillic", 0x0449 , 0 },
	{ "sheenarabic", 0x0634 , 0 },
	{ "sheenfinalarabic", 0xfeb6 , 0 },
	{ "sheeninitialarabic", 0xfeb7 , 0 },
	{ "sheenmedialarabic", 0xfeb8 , 0 },
	{ "sheicoptic", 0x03e3 , 0 },
	{ "sheqel", 0x20aa , 0 },
	{ "sheqelhebrew", 0x20aa , 0 },
	{ "sheva", 0x05b0 , 0 },
	{ "sheva115", 0x05b0 , 0 },
	{ "sheva15", 0x05b0 , 0 },
	{ "sheva22", 0x05b0 , 0 },
	{ "sheva2e", 0x05b0 , 0 },
	{ "shevahebrew", 0x05b0 , 0 },
	{ "shevanarrowhebrew", 0x05b0 , 0 },
	{ "shevaquarterhebrew", 0x05b0 , 0 },
	{ "shevawidehebrew", 0x05b0 , 0 },
	{ "shhacyrillic", 0x04bb , 0 },
	{ "shimacoptic", 0x03ed , 0 },
	{ "shin", 0x05e9 , 0 },
	{ "shindagesh", 0xfb49 , 0 },
	{ "shindageshhebrew", 0xfb49 , 0 },
	{ "shindageshshindot", 0xfb2c , 0 },
	{ "shindageshshindothebrew", 0xfb2c , 0 },
	{ "shindageshsindot", 0xfb2d , 0 },
	{ "shindageshsindothebrew", 0xfb2d , 0 },
	{ "shindothebrew", 0x05c1 , 0 },
	{ "shinhebrew", 0x05e9 , 0 },
	{ "shinshindot", 0xfb2a , 0 },
	{ "shinshindothebrew", 0xfb2a , 0 },
	{ "shinsindot", 0xfb2b , 0 },
	{ "shinsindothebrew", 0xfb2b , 0 },
	{ "shook", 0x0282 , 0 },
	{ "sigmafinal", 0x03c2 , 0 },
	{ "sigmalunatesymbolgreek", 0x03f2 , 0 },
	{ "sihiragana", 0x3057 , 0 },
	{ "sikatakana", 0x30b7 , 0 },
	{ "sikatakanahalfwidth", 0xff7c , 0 },
	{ "siluqhebrew", 0x05bd , 0 },
	{ "siluqlefthebrew", 0x05bd , 0 },
	{ "sindothebrew", 0x05c2 , 0 },
	{ "siosacirclekorean", 0x3274 , 0 },
	{ "siosaparenkorean", 0x3214 , 0 },
	{ "sioscieuckorean", 0x317e , 0 },
	{ "sioscirclekorean", 0x3266 , 0 },
	{ "sioskiyeokkorean", 0x317a , 0 },
	{ "sioskorean", 0x3145 , 0 },
	{ "siosnieunkorean", 0x317b , 0 },
	{ "siosparenkorean", 0x3206 , 0 },
	{ "siospieupkorean", 0x317d , 0 },
	{ "siostikeutkorean", 0x317c , 0 },
	{ "sixarabic", 0x0666 , 0 },
	{ "sixbengali", 0x09ec , 0 },
	{ "sixcircle", 0x2465 , 0 },
	{ "sixcircleinversesansserif", 0x278f , 0 },
	{ "sixdeva", 0x096c , 0 },
	{ "sixgujarati", 0x0aec , 0 },
	{ "sixgurmukhi", 0x0a6c , 0 },
	{ "sixhackarabic", 0x0666 , 0 },
	{ "sixhangzhou", 0x3026 , 0 },
	{ "sixideographicparen", 0x3225 , 0 },
	{ "sixinferior", 0x2086 , 0 },
	{ "sixmonospace", 0xff16 , 0 },
	{ "sixoldstyle", 0xf736 , 0 },
	{ "sixparen", 0x2479 , 0 },
	{ "sixperiod", 0x248d , 0 },
	{ "sixpersian", 0x06f6 , 0 },
	{ "sixroman", 0x2175 , 0 },
	{ "sixsuperior", 0x2076 , 0 },
	{ "sixteencircle", 0x246f , 0 },
	{ "sixteencurrencydenominatorbengali", 0x09f9 , 0 },
	{ "sixteenparen", 0x2483 , 0 },
	{ "sixteenperiod", 0x2497 , 0 },
	{ "sixthai", 0x0e56 , 0 },
	{ "slashmonospace", 0xff0f , 0 },
	{ "slong", 0x017f , 0 },
	{ "slongdotaccent", 0x1e9b , 0 },
	{ "smonospace", 0xff53 , 0 },
	{ "sofpasuqhebrew", 0x05c3 , 0 },
	{ "softhyphen", 0x00ad , 0 },
	{ "softsigncyrillic", 0x044c , 0 },
	{ "sohiragana", 0x305d , 0 },
	{ "sokatakana", 0x30bd , 0 },
	{ "sokatakanahalfwidth", 0xff7f , 0 },
	{ "soliduslongoverlaycmb", 0x0338 , 0 },
	{ "solidusshortoverlaycmb", 0x0337 , 0 },
	{ "sorusithai", 0x0e29 , 0 },
	{ "sosalathai", 0x0e28 , 0 },
	{ "sosothai", 0x0e0b , 0 },
	{ "sosuathai", 0x0e2a , 0 },
	{ "spacehackarabic", 0x0020 , 0 },
	{ "spadesuitblack", 0x2660 , 0 },
	{ "spadesuitwhite", 0x2664 , 0 },
	{ "sparen", 0x24ae , 0 },
	{ "squarebelowcmb", 0x033b , 0 },
	{ "squarecc", 0x33c4 , 0 },
	{ "squarecm", 0x339d , 0 },
	{ "squarediagonalcrosshatchfill", 0x25a9 , 0 },
	{ "squarehorizontalfill", 0x25a4 , 0 },
	{ "squarekg", 0x338f , 0 },
	{ "squarekm", 0x339e , 0 },
	{ "squarekmcapital", 0x33ce , 0 },
	{ "squareln", 0x33d1 , 0 },
	{ "squarelog", 0x33d2 , 0 },
	{ "squaremg", 0x338e , 0 },
	{ "squaremil", 0x33d5 , 0 },
	{ "squaremm", 0x339c , 0 },
	{ "squaremsquared", 0x33a1 , 0 },
	{ "squareorthogonalcrosshatchfill", 0x25a6 , 0 },
	{ "squareupperlefttolowerrightfill", 0x25a7 , 0 },
	{ "squareupperrighttolowerleftfill", 0x25a8 , 0 },
	{ "squareverticalfill", 0x25a5 , 0 },
	{ "squarewhitewithsmallblack", 0x25a3 , 0 },
	{ "srsquare", 0x33db , 0 },
	{ "ssabengali", 0x09b7 , 0 },
	{ "ssadeva", 0x0937 , 0 },
	{ "ssagujarati", 0x0ab7 , 0 },
	{ "ssangcieuckorean", 0x3149 , 0 },
	{ "ssanghieuhkorean", 0x3185 , 0 },
	{ "ssangieungkorean", 0x3180 , 0 },
	{ "ssangkiyeokkorean", 0x3132 , 0 },
	{ "ssangnieunkorean", 0x3165 , 0 },
	{ "ssangpieupkorean", 0x3143 , 0 },
	{ "ssangsioskorean", 0x3146 , 0 },
	{ "ssangtikeutkorean", 0x3138 , 0 },
	{ "ssuperior", 0xf6f2 , 0 },
	{ "sterlingmonospace", 0xffe1 , 0 },
	{ "strokelongoverlaycmb", 0x0336 , 0 },
	{ "strokeshortoverlaycmb", 0x0335 , 0 },
	{ "subset", 0x2282 , 0 },
	{ "subsetnotequal", 0x228a , 0 },
	{ "subsetorequal", 0x2286 , 0 },
	{ "succeeds", 0x227b , 0 },
	{ "suhiragana", 0x3059 , 0 },
	{ "sukatakana", 0x30b9 , 0 },
	{ "sukatakanahalfwidth", 0xff7d , 0 },
	{ "sukunarabic", 0x0652 , 0 },
	{ "superset", 0x2283 , 0 },
	{ "supersetnotequal", 0x228b , 0 },
	{ "supersetorequal", 0x2287 , 0 },
	{ "svsquare", 0x33dc , 0 },
	{ "syouwaerasquare", 0x337c , 0 },
	{ "tabengali", 0x09a4 , 0 },
	{ "tackdown", 0x22a4 , 0 },
	{ "tackleft", 0x22a3 , 0 },
	{ "tadeva", 0x0924 , 0 },
	{ "tagujarati", 0x0aa4 , 0 },
	{ "tagurmukhi", 0x0a24 , 0 },
	{ "taharabic", 0x0637 , 0 },
	{ "tahfinalarabic", 0xfec2 , 0 },
	{ "tahinitialarabic", 0xfec3 , 0 },
	{ "tahiragana", 0x305f , 0 },
	{ "tahmedialarabic", 0xfec4 , 0 },
	{ "taisyouerasquare", 0x337d , 0 },
	{ "takatakana", 0x30bf , 0 },
	{ "takatakanahalfwidth", 0xff80 , 0 },
	{ "tatweelarabic", 0x0640 , 0 },
	{ "tav", 0x05ea , 0 },
	{ "tavdages", 0xfb4a , 0 },
	{ "tavdagesh", 0xfb4a , 0 },
	{ "tavdageshhebrew", 0xfb4a , 0 },
	{ "tavhebrew", 0x05ea , 0 },
	{ "tbopomofo", 0x310a , 0 },
	{ "tccurl", 0x02a8 , 0 },
	{ "tcedilla", 0x0163 , 0 },
	{ "tcheharabic", 0x0686 , 0 },
	{ "tchehfinalarabic", 0xfb7b , 0 },
	{ "tchehinitialarabic", 0xfb7c , 0 },
	{ "tchehmedialarabic", 0xfb7d , 0 },
	{ "tchehmeeminitialarabic", 0xfb7c , 0 },
	{ "tcircle", 0x24e3 , 0 },
	{ "tcircumflexbelow", 0x1e71 , 0 },
	{ "tdieresis", 0x1e97 , 0 },
	{ "tdotaccent", 0x1e6b , 0 },
	{ "tdotbelow", 0x1e6d , 0 },
	{ "tecyrillic", 0x0442 , 0 },
	{ "tedescendercyrillic", 0x04ad , 0 },
	{ "teharabic", 0x062a , 0 },
	{ "tehfinalarabic", 0xfe96 , 0 },
	{ "tehhahinitialarabic", 0xfca2 , 0 },
	{ "tehhahisolatedarabic", 0xfc0c , 0 },
	{ "tehinitialarabic", 0xfe97 , 0 },
	{ "tehiragana", 0x3066 , 0 },
	{ "tehjeeminitialarabic", 0xfca1 , 0 },
	{ "tehjeemisolatedarabic", 0xfc0b , 0 },
	{ "tehmarbutaarabic", 0x0629 , 0 },
	{ "tehmarbutafinalarabic", 0xfe94 , 0 },
	{ "tehmedialarabic", 0xfe98 , 0 },
	{ "tehmeeminitialarabic", 0xfca4 , 0 },
	{ "tehmeemisolatedarabic", 0xfc0e , 0 },
	{ "tehnoonfinalarabic", 0xfc73 , 0 },
	{ "tekatakana", 0x30c6 , 0 },
	{ "tekatakanahalfwidth", 0xff83 , 0 },
	{ "telephone", 0x2121 , 0 },
	{ "telephoneblack", 0x260e , 0 },
	{ "telishagedolahebrew", 0x05a0 , 0 },
	{ "telishaqetanahebrew", 0x05a9 , 0 },
	{ "tencircle", 0x2469 , 0 },
	{ "tenideographicparen", 0x3229 , 0 },
	{ "tenparen", 0x247d , 0 },
	{ "tenperiod", 0x2491 , 0 },
	{ "tenroman", 0x2179 , 0 },
	{ "tesh", 0x02a7 , 0 },
	{ "tet", 0x05d8 , 0 },
	{ "tetdagesh", 0xfb38 , 0 },
	{ "tetdageshhebrew", 0xfb38 , 0 },
	{ "tethebrew", 0x05d8 , 0 },
	{ "tetsecyrillic", 0x04b5 , 0 },
	{ "tevirhebrew", 0x059b , 0 },
	{ "tevirlefthebrew", 0x059b , 0 },
	{ "thabengali", 0x09a5 , 0 },
	{ "thadeva", 0x0925 , 0 },
	{ "thagujarati", 0x0aa5 , 0 },
	{ "thagurmukhi", 0x0a25 , 0 },
	{ "thalarabic", 0x0630 , 0 },
	{ "thalfinalarabic", 0xfeac , 0 },
	{ "thanthakhatlowleftthai", 0xf898 , 0 },
	{ "thanthakhatlowrightthai", 0xf897 , 0 },
	{ "thanthakhatthai", 0x0e4c , 0 },
	{ "thanthakhatupperleftthai", 0xf896 , 0 },
	{ "theharabic", 0x062b , 0 },
	{ "thehfinalarabic", 0xfe9a , 0 },
	{ "thehinitialarabic", 0xfe9b , 0 },
	{ "thehmedialarabic", 0xfe9c , 0 },
	{ "thereexists", 0x2203 , 0 },
	{ "thetasymbolgreek", 0x03d1 , 0 },
	{ "thieuthacirclekorean", 0x3279 , 0 },
	{ "thieuthaparenkorean", 0x3219 , 0 },
	{ "thieuthcirclekorean", 0x326b , 0 },
	{ "thieuthkorean", 0x314c , 0 },
	{ "thieuthparenkorean", 0x320b , 0 },
	{ "thirteencircle", 0x246c , 0 },
	{ "thirteenparen", 0x2480 , 0 },
	{ "thirteenperiod", 0x2494 , 0 },
	{ "thonangmonthothai", 0x0e11 , 0 },
	{ "thook", 0x01ad , 0 },
	{ "thophuthaothai", 0x0e12 , 0 },
	{ "thothahanthai", 0x0e17 , 0 },
	{ "thothanthai", 0x0e10 , 0 },
	{ "thothongthai", 0x0e18 , 0 },
	{ "thothungthai", 0x0e16 , 0 },
	{ "thousandcyrillic", 0x0482 , 0 },
	{ "thousandsseparatorarabic", 0x066c , 0 },
	{ "thousandsseparatorpersian", 0x066c , 0 },
	{ "threearabic", 0x0663 , 0 },
	{ "threebengali", 0x09e9 , 0 },
	{ "threecircle", 0x2462 , 0 },
	{ "threecircleinversesansserif", 0x278c , 0 },
	{ "threedeva", 0x0969 , 0 },
	{ "threegujarati", 0x0ae9 , 0 },
	{ "threegurmukhi", 0x0a69 , 0 },
	{ "threehackarabic", 0x0663 , 0 },
	{ "threehangzhou", 0x3023 , 0 },
	{ "threeideographicparen", 0x3222 , 0 },
	{ "threeinferior", 0x2083 , 0 },
	{ "threemonospace", 0xff13 , 0 },
	{ "threenumeratorbengali", 0x09f6 , 0 },
	{ "threeoldstyle", 0xf733 , 0 },
	{ "threeparen", 0x2476 , 0 },
	{ "threeperiod", 0x248a , 0 },
	{ "threepersian", 0x06f3 , 0 },
	{ "threequartersemdash", 0xf6de , 0 },
	{ "threeroman", 0x2172 , 0 },
	{ "threesuperior", 0x00b3 , 0 },
	{ "threethai", 0x0e53 , 0 },
	{ "thzsquare", 0x3394 , 0 },
	{ "tihiragana", 0x3061 , 0 },
	{ "tikatakana", 0x30c1 , 0 },
	{ "tikatakanahalfwidth", 0xff81 , 0 },
	{ "tikeutacirclekorean", 0x3270 , 0 },
	{ "tikeutaparenkorean", 0x3210 , 0 },
	{ "tikeutcirclekorean", 0x3262 , 0 },
	{ "tikeutkorean", 0x3137 , 0 },
	{ "tikeutparenkorean", 0x3202 , 0 },
	{ "tildebelowcmb", 0x0330 , 0 },
	{ "tildecmb", 0x0303 , 0 },
	{ "tildedoublecmb", 0x0360 , 0 },
	{ "tildeoperator", 0x223c , 0 },
	{ "tildeoverlaycmb", 0x0334 , 0 },
	{ "tildeverticalcmb", 0x033e , 0 },
	{ "timescircle", 0x2297 , 0 },
	{ "tipehahebrew", 0x0596 , 0 },
	{ "tipehalefthebrew", 0x0596 , 0 },
	{ "tippigurmukhi", 0x0a70 , 0 },
	{ "titlocyrilliccmb", 0x0483 , 0 },
	{ "tiwnarmenian", 0x057f , 0 },
	{ "tlinebelow", 0x1e6f , 0 },
	{ "tmonospace", 0xff54 , 0 },
	{ "toarmenian", 0x0569 , 0 },
	{ "tohiragana", 0x3068 , 0 },
	{ "tokatakana", 0x30c8 , 0 },
	{ "tokatakanahalfwidth", 0xff84 , 0 },
	{ "tonebarextrahighmod", 0x02e5 , 0 },
	{ "tonebarextralowmod", 0x02e9 , 0 },
	{ "tonebarhighmod", 0x02e6 , 0 },
	{ "tonebarlowmod", 0x02e8 , 0 },
	{ "tonebarmidmod", 0x02e7 , 0 },
	{ "tonefive", 0x01bd , 0 },
	{ "tonesix", 0x0185 , 0 },
	{ "tonetwo", 0x01a8 , 0 },
	{ "tonsquare", 0x3327 , 0 },
	{ "topatakthai", 0x0e0f , 0 },
	{ "tortoiseshellbracketleft", 0x3014 , 0 },
	{ "tortoiseshellbracketleftsmall", 0xfe5d , 0 },
	{ "tortoiseshellbracketleftvertical", 0xfe39 , 0 },
	{ "tortoiseshellbracketright", 0x3015 , 0 },
	{ "tortoiseshellbracketrightsmall", 0xfe5e , 0 },
	{ "tortoiseshellbracketrightvertical", 0xfe3a , 0 },
	{ "totaothai", 0x0e15 , 0 },
	{ "tpalatalhook", 0x01ab , 0 },
	{ "tparen", 0x24af , 0 },
	{ "trademarksans", 0xf8ea , 0 },
	{ "trademarkserif", 0xf6db , 0 },
	{ "tretroflexhook", 0x0288 , 0 },
	{ "ts", 0x02a6 , 0 },
	{ "tsadi", 0x05e6 , 0 },
	{ "tsadidagesh", 0xfb46 , 0 },
	{ "tsadidageshhebrew", 0xfb46 , 0 },
	{ "tsadihebrew", 0x05e6 , 0 },
	{ "tsecyrillic", 0x0446 , 0 },
	{ "tsere", 0x05b5 , 0 },
	{ "tsere12", 0x05b5 , 0 },
	{ "tsere1e", 0x05b5 , 0 },
	{ "tsere2b", 0x05b5 , 0 },
	{ "tserehebrew", 0x05b5 , 0 },
	{ "tserenarrowhebrew", 0x05b5 , 0 },
	{ "tserequarterhebrew", 0x05b5 , 0 },
	{ "tserewidehebrew", 0x05b5 , 0 },
	{ "tshecyrillic", 0x045b , 0 },
	{ "tsuperior", 0xf6f3 , 0 },
	{ "ttabengali", 0x099f , 0 },
	{ "ttadeva", 0x091f , 0 },
	{ "ttagujarati", 0x0a9f , 0 },
	{ "ttagurmukhi", 0x0a1f , 0 },
	{ "tteharabic", 0x0679 , 0 },
	{ "ttehfinalarabic", 0xfb67 , 0 },
	{ "ttehinitialarabic", 0xfb68 , 0 },
	{ "ttehmedialarabic", 0xfb69 , 0 },
	{ "tthabengali", 0x09a0 , 0 },
	{ "tthadeva", 0x0920 , 0 },
	{ "tthagujarati", 0x0aa0 , 0 },
	{ "tthagurmukhi", 0x0a20 , 0 },
	{ "tturned", 0x0287 , 0 },
	{ "tuhiragana", 0x3064 , 0 },
	{ "tukatakana", 0x30c4 , 0 },
	{ "tukatakanahalfwidth", 0xff82 , 0 },
	{ "tusmallhiragana", 0x3063 , 0 },
	{ "tusmallkatakana", 0x30c3 , 0 },
	{ "tusmallkatakanahalfwidth", 0xff6f , 0 },
	{ "twelvecircle", 0x246b , 0 },
	{ "twelveparen", 0x247f , 0 },
	{ "twelveperiod", 0x2493 , 0 },
	{ "twelveroman", 0x217b , 0 },
	{ "twentycircle", 0x2473 , 0 },
	{ "twentyhangzhou", 0x5344 , 0 },
	{ "twentyparen", 0x2487 , 0 },
	{ "twentyperiod", 0x249b , 0 },
	{ "twoarabic", 0x0662 , 0 },
	{ "twobengali", 0x09e8 , 0 },
	{ "twocircle", 0x2461 , 0 },
	{ "twocircleinversesansserif", 0x278b , 0 },
	{ "twodeva", 0x0968 , 0 },
	{ "twodotleader", 0x2025 , 0 },
	{ "twodotleadervertical", 0xfe30 , 0 },
	{ "twogujarati", 0x0ae8 , 0 },
	{ "twogurmukhi", 0x0a68 , 0 },
	{ "twohackarabic", 0x0662 , 0 },
	{ "twohangzhou", 0x3022 , 0 },
	{ "twoideographicparen", 0x3221 , 0 },
	{ "twoinferior", 0x2082 , 0 },
	{ "twomonospace", 0xff12 , 0 },
	{ "twonumeratorbengali", 0x09f5 , 0 },
	{ "twooldstyle", 0xf732 , 0 },
	{ "twoparen", 0x2475 , 0 },
	{ "twoperiod", 0x2489 , 0 },
	{ "twopersian", 0x06f2 , 0 },
	{ "tworoman", 0x2171 , 0 },
	{ "twostroke", 0x01bb , 0 },
	{ "twosuperior", 0x00b2 , 0 },
	{ "twothai", 0x0e52 , 0 },
	{ "ubar", 0x0289 , 0 },
	{ "ubengali", 0x0989 , 0 },
	{ "ubopomofo", 0x3128 , 0 },
	{ "ucaron", 0x01d4 , 0 },
	{ "ucircle", 0x24e4 , 0 },
	{ "ucircumflexbelow", 0x1e77 , 0 },
	{ "ucyrillic", 0x0443 , 0 },
	{ "udattadeva", 0x0951 , 0 },
	{ "udblacute", 0x0171 , 0 },
	{ "udblgrave", 0x0215 , 0 },
	{ "udeva", 0x0909 , 0 },
	{ "udieresisacute", 0x01d8 , 0 },
	{ "udieresisbelow", 0x1e73 , 0 },
	{ "udieresiscaron", 0x01da , 0 },
	{ "udieresiscyrillic", 0x04f1 , 0 },
	{ "udieresisgrave", 0x01dc , 0 },
	{ "udieresismacron", 0x01d6 , 0 },
	{ "udotbelow", 0x1ee5 , 0 },
	{ "ugujarati", 0x0a89 , 0 },
	{ "ugurmukhi", 0x0a09 , 0 },
	{ "uhiragana", 0x3046 , 0 },
	{ "uhookabove", 0x1ee7 , 0 },
	{ "uhornacute", 0x1ee9 , 0 },
	{ "uhorndotbelow", 0x1ef1 , 0 },
	{ "uhorngrave", 0x1eeb , 0 },
	{ "uhornhookabove", 0x1eed , 0 },
	{ "uhorntilde", 0x1eef , 0 },
	{ "uhungarumlautcyrillic", 0x04f3 , 0 },
	{ "uinvertedbreve", 0x0217 , 0 },
	{ "ukatakana", 0x30a6 , 0 },
	{ "ukatakanahalfwidth", 0xff73 , 0 },
	{ "ukcyrillic", 0x0479 , 0 },
	{ "ukorean", 0x315c , 0 },
	{ "umacroncyrillic", 0x04ef , 0 },
	{ "umacrondieresis", 0x1e7b , 0 },
	{ "umatragurmukhi", 0x0a41 , 0 },
	{ "umonospace", 0xff55 , 0 },
	{ "underscoremonospace", 0xff3f , 0 },
	{ "underscorevertical", 0xfe33 , 0 },
	{ "underscorewavy", 0xfe4f , 0 },
	{ "uparen", 0x24b0 , 0 },
	{ "upperdothebrew", 0x05c4 , 0 },
	{ "upsilonlatin", 0x028a , 0 },
	{ "uptackbelowcmb", 0x031d , 0 },
	{ "uptackmod", 0x02d4 , 0 },
	{ "uragurmukhi", 0x0a73 , 0 },
	{ "ushortcyrillic", 0x045e , 0 },
	{ "usmallhiragana", 0x3045 , 0 },
	{ "usmallkatakana", 0x30a5 , 0 },
	{ "usmallkatakanahalfwidth", 0xff69 , 0 },
	{ "ustraightcyrillic", 0x04af , 0 },
	{ "ustraightstrokecyrillic", 0x04b1 , 0 },
	{ "utildeacute", 0x1e79 , 0 },
	{ "utildebelow", 0x1e75 , 0 },
	{ "uubengali", 0x098a , 0 },
	{ "uudeva", 0x090a , 0 },
	{ "uugujarati", 0x0a8a , 0 },
	{ "uugurmukhi", 0x0a0a , 0 },
	{ "uumatragurmukhi", 0x0a42 , 0 },
	{ "uuvowelsignbengali", 0x09c2 , 0 },
	{ "uuvowelsigndeva", 0x0942 , 0 },
	{ "uuvowelsigngujarati", 0x0ac2 , 0 },
	{ "uvowelsignbengali", 0x09c1 , 0 },
	{ "uvowelsigndeva", 0x0941 , 0 },
	{ "uvowelsigngujarati", 0x0ac1 , 0 },
	{ "vadeva", 0x0935 , 0 },
	{ "vagujarati", 0x0ab5 , 0 },
	{ "vagurmukhi", 0x0a35 , 0 },
	{ "vakatakana", 0x30f7 , 0 },
	{ "vav", 0x05d5 , 0 },
	{ "vavdagesh", 0xfb35 , 0 },
	{ "vavdagesh65", 0xfb35 , 0 },
	{ "vavdageshhebrew", 0xfb35 , 0 },
	{ "vavhebrew", 0x05d5 , 0 },
	{ "vavholam", 0xfb4b , 0 },
	{ "vavholamhebrew", 0xfb4b , 0 },
	{ "vavvavhebrew", 0x05f0 , 0 },
	{ "vavyodhebrew", 0x05f1 , 0 },
	{ "vcircle", 0x24e5 , 0 },
	{ "vdotbelow", 0x1e7f , 0 },
	{ "vecyrillic", 0x0432 , 0 },
	{ "veharabic", 0x06a4 , 0 },
	{ "vehfinalarabic", 0xfb6b , 0 },
	{ "vehinitialarabic", 0xfb6c , 0 },
	{ "vehmedialarabic", 0xfb6d , 0 },
	{ "vekatakana", 0x30f9 , 0 },
	{ "venus", 0x2640 , 0 },
	{ "verticalbar", 0x007c , 0 },
	{ "verticallineabovecmb", 0x030d , 0 },
	{ "verticallinebelowcmb", 0x0329 , 0 },
	{ "verticallinelowmod", 0x02cc , 0 },
	{ "verticallinemod", 0x02c8 , 0 },
	{ "vewarmenian", 0x057e , 0 },
	{ "vhook", 0x028b , 0 },
	{ "vikatakana", 0x30f8 , 0 },
	{ "viramabengali", 0x09cd , 0 },
	{ "viramadeva", 0x094d , 0 },
	{ "viramagujarati", 0x0acd , 0 },
	{ "visargabengali", 0x0983 , 0 },
	{ "visargadeva", 0x0903 , 0 },
	{ "visargagujarati", 0x0a83 , 0 },
	{ "vmonospace", 0xff56 , 0 },
	{ "voarmenian", 0x0578 , 0 },
	{ "voicediterationhiragana", 0x309e , 0 },
	{ "voicediterationkatakana", 0x30fe , 0 },
	{ "voicedmarkkana", 0x309b , 0 },
	{ "voicedmarkkanahalfwidth", 0xff9e , 0 },
	{ "vokatakana", 0x30fa , 0 },
	{ "vparen", 0x24b1 , 0 },
	{ "vtilde", 0x1e7d , 0 },
	{ "vturned", 0x028c , 0 },
	{ "vuhiragana", 0x3094 , 0 },
	{ "vukatakana", 0x30f4 , 0 },
	{ "waekorean", 0x3159 , 0 },
	{ "wahiragana", 0x308f , 0 },
	{ "wakatakana", 0x30ef , 0 },
	{ "wakatakanahalfwidth", 0xff9c , 0 },
	{ "wakorean", 0x3158 , 0 },
	{ "wasmallhiragana", 0x308e , 0 },
	{ "wasmallkatakana", 0x30ee , 0 },
	{ "wattosquare", 0x3357 , 0 },
	{ "wavedash", 0x301c , 0 },
	{ "wavyunderscorevertical", 0xfe34 , 0 },
	{ "wawarabic", 0x0648 , 0 },
	{ "wawfinalarabic", 0xfeee , 0 },
	{ "wawhamzaabovearabic", 0x0624 , 0 },
	{ "wawhamzaabovefinalarabic", 0xfe86 , 0 },
	{ "wbsquare", 0x33dd , 0 },
	{ "wcircle", 0x24e6 , 0 },
	{ "wdotaccent", 0x1e87 , 0 },
	{ "wdotbelow", 0x1e89 , 0 },
	{ "wehiragana", 0x3091 , 0 },
	{ "wekatakana", 0x30f1 , 0 },
	{ "wekorean", 0x315e , 0 },
	{ "weokorean", 0x315d , 0 },
	{ "whitebullet", 0x25e6 , 0 },
	{ "whitecircle", 0x25cb , 0 },
	{ "whitecircleinverse", 0x25d9 , 0 },
	{ "whitecornerbracketleft", 0x300e , 0 },
	{ "whitecornerbracketleftvertical", 0xfe43 , 0 },
	{ "whitecornerbracketright", 0x300f , 0 },
	{ "whitecornerbracketrightvertical", 0xfe44 , 0 },
	{ "whitediamond", 0x25c7 , 0 },
	{ "whitediamondcontainingblacksmalldiamond", 0x25c8 , 0 },
	{ "whitedownpointingsmalltriangle", 0x25bf , 0 },
	{ "whitedownpointingtriangle", 0x25bd , 0 },
	{ "whiteleftpointingsmalltriangle", 0x25c3 , 0 },
	{ "whiteleftpointingtriangle", 0x25c1 , 0 },
	{ "whitelenticularbracketleft", 0x3016 , 0 },
	{ "whitelenticularbracketright", 0x3017 , 0 },
	{ "whiterightpointingsmalltriangle", 0x25b9 , 0 },
	{ "whiterightpointingtriangle", 0x25b7 , 0 },
	{ "whitesmallsquare", 0x25ab , 0 },
	{ "whitesmilingface", 0x263a , 0 },
	{ "whitesquare", 0x25a1 , 0 },
	{ "whitestar", 0x2606 , 0 },
	{ "whitetelephone", 0x260f , 0 },
	{ "whitetortoiseshellbracketleft", 0x3018 , 0 },
	{ "whitetortoiseshellbracketright", 0x3019 , 0 },
	{ "whiteuppointingsmalltriangle", 0x25b5 , 0 },
	{ "whiteuppointingtriangle", 0x25b3 , 0 },
	{ "wihiragana", 0x3090 , 0 },
	{ "wikatakana", 0x30f0 , 0 },
	{ "wikorean", 0x315f , 0 },
	{ "wmonospace", 0xff57 , 0 },
	{ "wohiragana", 0x3092 , 0 },
	{ "wokatakana", 0x30f2 , 0 },
	{ "wokatakanahalfwidth", 0xff66 , 0 },
	{ "won", 0x20a9 , 0 },
	{ "wonmonospace", 0xffe6 , 0 },
	{ "wowaenthai", 0x0e27 , 0 },
	{ "wparen", 0x24b2 , 0 },
	{ "wring", 0x1e98 , 0 },
	{ "wsuperior", 0x02b7 , 0 },
	{ "wturned", 0x028d , 0 },
	{ "wynn", 0x01bf , 0 },
	{ "xabovecmb", 0x033d , 0 },
	{ "xbopomofo", 0x3112 , 0 },
	{ "xcircle", 0x24e7 , 0 },
	{ "xdieresis", 0x1e8d , 0 },
	{ "xdotaccent", 0x1e8b , 0 },
	{ "xeharmenian", 0x056d , 0 },
	{ "xmonospace", 0xff58 , 0 },
	{ "xparen", 0x24b3 , 0 },
	{ "xsuperior", 0x02e3 , 0 },
	{ "yaadosquare", 0x334e , 0 },
	{ "yabengali", 0x09af , 0 },
	{ "yadeva", 0x092f , 0 },
	{ "yaekorean", 0x3152 , 0 },
	{ "yagujarati", 0x0aaf , 0 },
	{ "yagurmukhi", 0x0a2f , 0 },
	{ "yahiragana", 0x3084 , 0 },
	{ "yakatakana", 0x30e4 , 0 },
	{ "yakatakanahalfwidth", 0xff94 , 0 },
	{ "yakorean", 0x3151 , 0 },
	{ "yamakkanthai", 0x0e4e , 0 },
	{ "yasmallhiragana", 0x3083 , 0 },
	{ "yasmallkatakana", 0x30e3 , 0 },
	{ "yasmallkatakanahalfwidth", 0xff6c , 0 },
	{ "yatcyrillic", 0x0463 , 0 },
	{ "ycircle", 0x24e8 , 0 },
	{ "ydotaccent", 0x1e8f , 0 },
	{ "ydotbelow", 0x1ef5 , 0 },
	{ "yeharabic", 0x064a , 0 },
	{ "yehbarreearabic", 0x06d2 , 0 },
	{ "yehbarreefinalarabic", 0xfbaf , 0 },
	{ "yehfinalarabic", 0xfef2 , 0 },
	{ "yehhamzaabovearabic", 0x0626 , 0 },
	{ "yehhamzaabovefinalarabic", 0xfe8a , 0 },
	{ "yehhamzaaboveinitialarabic", 0xfe8b , 0 },
	{ "yehhamzaabovemedialarabic", 0xfe8c , 0 },
	{ "yehinitialarabic", 0xfef3 , 0 },
	{ "yehmedialarabic", 0xfef4 , 0 },
	{ "yehmeeminitialarabic", 0xfcdd , 0 },
	{ "yehmeemisolatedarabic", 0xfc58 , 0 },
	{ "yehnoonfinalarabic", 0xfc94 , 0 },
	{ "yehthreedotsbelowarabic", 0x06d1 , 0 },
	{ "yekorean", 0x3156 , 0 },
	{ "yenmonospace", 0xffe5 , 0 },
	{ "yeokorean", 0x3155 , 0 },
	{ "yeorinhieuhkorean", 0x3186 , 0 },
	{ "yerahbenyomohebrew", 0x05aa , 0 },
	{ "yerahbenyomolefthebrew", 0x05aa , 0 },
	{ "yericyrillic", 0x044b , 0 },
	{ "yerudieresiscyrillic", 0x04f9 , 0 },
	{ "yesieungkorean", 0x3181 , 0 },
	{ "yesieungpansioskorean", 0x3183 , 0 },
	{ "yesieungsioskorean", 0x3182 , 0 },
	{ "yetivhebrew", 0x059a , 0 },
	{ "yhook", 0x01b4 , 0 },
	{ "yhookabove", 0x1ef7 , 0 },
	{ "yiarmenian", 0x0575 , 0 },
	{ "yicyrillic", 0x0457 , 0 },
	{ "yikorean", 0x3162 , 0 },
	{ "yinyang", 0x262f , 0 },
	{ "yiwnarmenian", 0x0582 , 0 },
	{ "ymonospace", 0xff59 , 0 },
	{ "yod", 0x05d9 , 0 },
	{ "yoddagesh", 0xfb39 , 0 },
	{ "yoddageshhebrew", 0xfb39 , 0 },
	{ "yodhebrew", 0x05d9 , 0 },
	{ "yodyodhebrew", 0x05f2 , 0 },
	{ "yodyodpatahhebrew", 0xfb1f , 0 },
	{ "yohiragana", 0x3088 , 0 },
	{ "yoikorean", 0x3189 , 0 },
	{ "yokatakana", 0x30e8 , 0 },
	{ "yokatakanahalfwidth", 0xff96 , 0 },
	{ "yokorean", 0x315b , 0 },
	{ "yosmallhiragana", 0x3087 , 0 },
	{ "yosmallkatakana", 0x30e7 , 0 },
	{ "yosmallkatakanahalfwidth", 0xff6e , 0 },
	{ "yotgreek", 0x03f3 , 0 },
	{ "yoyaekorean", 0x3188 , 0 },
	{ "yoyakorean", 0x3187 , 0 },
	{ "yoyakthai", 0x0e22 , 0 },
	{ "yoyingthai", 0x0e0d , 0 },
	{ "yparen", 0x24b4 , 0 },
	{ "ypogegrammeni", 0x037a , 0 },
	{ "ypogegrammenigreekcmb", 0x0345 , 0 },
	{ "yr", 0x01a6 , 0 },
	{ "yring", 0x1e99 , 0 },
	{ "ysuperior", 0x02b8 , 0 },
	{ "ytilde", 0x1ef9 , 0 },
	{ "yturned", 0x028e , 0 },
	{ "yuhiragana", 0x3086 , 0 },
	{ "yuikorean", 0x318c , 0 },
	{ "yukatakana", 0x30e6 , 0 },
	{ "yukatakanahalfwidth", 0xff95 , 0 },
	{ "yukorean", 0x3160 , 0 },
	{ "yusbigcyrillic", 0x046b , 0 },
	{ "yusbigiotifiedcyrillic", 0x046d , 0 },
	{ "yuslittlecyrillic", 0x0467 , 0 },
	{ "yuslittleiotifiedcyrillic", 0x0469 , 0 },
	{ "yusmallhiragana", 0x3085 , 0 },
	{ "yusmallkatakana", 0x30e5 , 0 },
	{ "yusmallkatakanahalfwidth", 0xff6d , 0 },
	{ "yuyekorean", 0x318b , 0 },
	{ "yuyeokorean", 0x318a , 0 },
	{ "yyabengali", 0x09df , 0 },
	{ "yyadeva", 0x095f , 0 },
	{ "zaarmenian", 0x0566 , 0 },
	{ "zadeva", 0x095b , 0 },
	{ "zagurmukhi", 0x0a5b , 0 },
	{ "zaharabic", 0x0638 , 0 },
	{ "zahfinalarabic", 0xfec6 , 0 },
	{ "zahinitialarabic", 0xfec7 , 0 },
	{ "zahiragana", 0x3056 , 0 },
	{ "zahmedialarabic", 0xfec8 , 0 },
	{ "zainarabic", 0x0632 , 0 },
	{ "zainfinalarabic", 0xfeb0 , 0 },
	{ "zakatakana", 0x30b6 , 0 },
	{ "zaqefgadolhebrew", 0x0595 , 0 },
	{ "zaqefqatanhebrew", 0x0594 , 0 },
	{ "zarqahebrew", 0x0598 , 0 },
	{ "zayin", 0x05d6 , 0 },
	{ "zayindagesh", 0xfb36 , 0 },
	{ "zayindageshhebrew", 0xfb36 , 0 },
	{ "zayinhebrew", 0x05d6 , 0 },
	{ "zbopomofo", 0x3117 , 0 },
	{ "zcircle", 0x24e9 , 0 },
	{ "zcircumflex", 0x1e91 , 0 },
	{ "zcurl", 0x0291 , 0 },
	{ "zdot", 0x017c , 0 },
	{ "zdotbelow", 0x1e93 , 0 },
	{ "zecyrillic", 0x0437 , 0 },
	{ "zedescendercyrillic", 0x0499 , 0 },
	{ "zedieresiscyrillic", 0x04df , 0 },
	{ "zehiragana", 0x305c , 0 },
	{ "zekatakana", 0x30bc , 0 },
	{ "zeroarabic", 0x0660 , 0 },
	{ "zerobengali", 0x09e6 , 0 },
	{ "zerodeva", 0x0966 , 0 },
	{ "zerogujarati", 0x0ae6 , 0 },
	{ "zerogurmukhi", 0x0a66 , 0 },
	{ "zerohackarabic", 0x0660 , 0 },
	{ "zeroinferior", 0x2080 , 0 },
	{ "zeromonospace", 0xff10 , 0 },
	{ "zerooldstyle", 0xf730 , 0 },
	{ "zeropersian", 0x06f0 , 0 },
	{ "zerosuperior", 0x2070 , 0 },
	{ "zerothai", 0x0e50 , 0 },
	{ "zerowidthjoiner", 0xfeff , 0 },
	{ "zerowidthnonjoiner", 0x200c , 0 },
	{ "zerowidthspace", 0x200b , 0 },
	{ "zhbopomofo", 0x3113 , 0 },
	{ "zhearmenian", 0x056a , 0 },
	{ "zhebrevecyrillic", 0x04c2 , 0 },
	{ "zhecyrillic", 0x0436 , 0 },
	{ "zhedescendercyrillic", 0x0497 , 0 },
	{ "zhedieresiscyrillic", 0x04dd , 0 },
	{ "zihiragana", 0x3058 , 0 },
	{ "zikatakana", 0x30b8 , 0 },
	{ "zinorhebrew", 0x05ae , 0 },
	{ "zlinebelow", 0x1e95 , 0 },
	{ "zmonospace", 0xff5a , 0 },
	{ "zohiragana", 0x305e , 0 },
	{ "zokatakana", 0x30be , 0 },
	{ "zparen", 0x24b5 , 0 },
	{ "zretroflexhook", 0x0290 , 0 },
	{ "zstroke", 0x01b6 , 0 },
	{ "zuhiragana", 0x305a , 0 },
	{ "zukatakana", 0x30ba , 0 },
	{ "dotaccent.cap", 0xefed , 0 },
	{ "breve.cap", 0xefee , 0 },
	{ "ogonek.cap", 0xeff1 , 0 },
	{ "cedilla.cap", 0xeff2 , 0 },
	{ "ring.cap", 0xeff3 , 0 },
	{ "tilde.cap", 0xeff5 , 0 },
	{ "circumflex.cap", 0xeff7 , 0 },
	{ "a1", 0x2701 , 0 },
	{ "a2", 0x2702 , 0 },
	{ "a202", 0x2703 , 0 },
	{ "a3", 0x2704 , 0 },
	{ "a4", 0x260e , 0 },
	{ "a5", 0x2706 , 0 },
	{ "a119", 0x2707 , 0 },
	{ "a118", 0x2708 , 0 },
	{ "a117", 0x2709 , 0 },
	{ "a11", 0x261b , 0 },
	{ "a12", 0x261e , 0 },
	{ "a13", 0x270c , 0 },
	{ "a14", 0x270d , 0 },
	{ "a15", 0x270e , 0 },
	{ "a16", 0x270f , 0 },
	{ "a105", 0x2710 , 0 },
	{ "a17", 0x2711 , 0 },
	{ "a18", 0x2712 , 0 },
	{ "a19", 0x2713 , 0 },
	{ "a20", 0x2714 , 0 },
	{ "a21", 0x2715 , 0 },
	{ "a22", 0x2716 , 0 },
	{ "a23", 0x2717 , 0 },
	{ "a24", 0x2718 , 0 },
	{ "a25", 0x2719 , 0 },
	{ "a26", 0x271a , 0 },
	{ "a27", 0x271b , 0 },
	{ "a28", 0x271c , 0 },
	{ "a6", 0x271d , 0 },
	{ "a7", 0x271e , 0 },
	{ "a8", 0x271f , 0 },
	{ "a9", 0x2720 , 0 },
	{ "a10", 0x2721 , 0 },
	{ "a29", 0x2722 , 0 },
	{ "a30", 0x2723 , 0 },
	{ "a31", 0x2724 , 0 },
	{ "a32", 0x2725 , 0 },
	{ "a33", 0x2726 , 0 },
	{ "a34", 0x2727 , 0 },
	{ "a35", 0x2605 , 0 },
	{ "a36", 0x2729 , 0 },
	{ "a37", 0x272a , 0 },
	{ "a38", 0x272b , 0 },
	{ "a39", 0x272c , 0 },
	{ "a40", 0x272d , 0 },
	{ "a41", 0x272e , 0 },
	{ "a42", 0x272f , 0 },
	{ "a43", 0x2730 , 0 },
	{ "a44", 0x2731 , 0 },
	{ "a45", 0x2732 , 0 },
	{ "a46", 0x2733 , 0 },
	{ "a47", 0x2734 , 0 },
	{ "a48", 0x2735 , 0 },
	{ "a49", 0x2736 , 0 },
	{ "a50", 0x2737 , 0 },
	{ "a51", 0x2738 , 0 },
	{ "a52", 0x2739 , 0 },
	{ "a53", 0x273a , 0 },
	{ "a54", 0x273b , 0 },
	{ "a55", 0x273c , 0 },
	{ "a56", 0x273d , 0 },
	{ "a57", 0x273e , 0 },
	{ "a58", 0x273f , 0 },
	{ "a59", 0x2740 , 0 },
	{ "a60", 0x2741 , 0 },
	{ "a61", 0x2742 , 0 },
	{ "a62", 0x2743 , 0 },
	{ "a63", 0x2744 , 0 },
	{ "a64", 0x2745 , 0 },
	{ "a65", 0x2746 , 0 },
	{ "a66", 0x2747 , 0 },
	{ "a67", 0x2748 , 0 },
	{ "a68", 0x2749 , 0 },
	{ "a69", 0x274a , 0 },
	{ "a70", 0x274b , 0 },
	{ "a71", 0x25cf , 0 },
	{ "a72", 0x274d , 0 },
	{ "a73", 0x25a0 , 0 },
	{ "a74", 0x274f , 0 },
	{ "a203", 0x2750 , 0 },
	{ "a75", 0x2751 , 0 },
	{ "a204", 0x2752 , 0 },
	{ "a76", 0x25b2 , 0 },
	{ "a77", 0x25bc , 0 },
	{ "a78", 0x25c6 , 0 },
	{ "a79", 0x2756 , 0 },
	{ "a81", 0x25d7 , 0 },
	{ "a82", 0x2758 , 0 },
	{ "a83", 0x2759 , 0 },
	{ "a84", 0x275a , 0 },
	{ "a97", 0x275b , 0 },
	{ "a98", 0x275c , 0 },
	{ "a99", 0x275d , 0 },
	{ "a100", 0x275e , 0 },
	{ "a89", 0x2768 , 0 },
	{ "a90", 0x2769 , 0 },
	{ "a93", 0x276a , 0 },
	{ "a94", 0x276b , 0 },
	{ "a91", 0x276c , 0 },
	{ "a92", 0x276d , 0 },
	{ "a205", 0x276e , 0 },
	{ "a85", 0x276f , 0 },
	{ "a206", 0x2770 , 0 },
	{ "a86", 0x2771 , 0 },
	{ "a87", 0x2772 , 0 },
	{ "a88", 0x2773 , 0 },
	{ "a95", 0x2774 , 0 },
	{ "a96", 0x2775 , 0 },
	{ "a101", 0x2761 , 0 },
	{ "a102", 0x2762 , 0 },
	{ "a103", 0x2763 , 0 },
	{ "a104", 0x2764 , 0 },
	{ "a106", 0x2765 , 0 },
	{ "a107", 0x2766 , 0 },
	{ "a108", 0x2767 , 0 },
	{ "a112", 0x2663 , 0 },
	{ "a111", 0x2666 , 0 },
	{ "a110", 0x2665 , 0 },
	{ "a109", 0x2660 , 0 },
	{ "a120", 0x2460 , 0 },
	{ "a121", 0x2461 , 0 },
	{ "a122", 0x2462 , 0 },
	{ "a123", 0x2463 , 0 },
	{ "a124", 0x2464 , 0 },
	{ "a125", 0x2465 , 0 },
	{ "a126", 0x2466 , 0 },
	{ "a127", 0x2467 , 0 },
	{ "a128", 0x2468 , 0 },
	{ "a129", 0x2469 , 0 },
	{ "a130", 0x2776 , 0 },
	{ "a131", 0x2777 , 0 },
	{ "a132", 0x2778 , 0 },
	{ "a133", 0x2779 , 0 },
	{ "a134", 0x277a , 0 },
	{ "a135", 0x277b , 0 },
	{ "a136", 0x277c , 0 },
	{ "a137", 0x277d , 0 },
	{ "a138", 0x277e , 0 },
	{ "a139", 0x277f , 0 },
	{ "a140", 0x2780 , 0 },
	{ "a141", 0x2781 , 0 },
	{ "a142", 0x2782 , 0 },
	{ "a143", 0x2783 , 0 },
	{ "a144", 0x2784 , 0 },
	{ "a145", 0x2785 , 0 },
	{ "a146", 0x2786 , 0 },
	{ "a147", 0x2787 , 0 },
	{ "a148", 0x2788 , 0 },
	{ "a149", 0x2789 , 0 },
	{ "a150", 0x278a , 0 },
	{ "a151", 0x278b , 0 },
	{ "a152", 0x278c , 0 },
	{ "a153", 0x278d , 0 },
	{ "a154", 0x278e , 0 },
	{ "a155", 0x278f , 0 },
	{ "a156", 0x2790 , 0 },
	{ "a157", 0x2791 , 0 },
	{ "a158", 0x2792 , 0 },
	{ "a159", 0x2793 , 0 },
	{ "a160", 0x2794 , 0 },
	{ "a161", 0x2192 , 0 },
	{ "a163", 0x2194 , 0 },
	{ "a164", 0x2195 , 0 },
	{ "a196", 0x2798 , 0 },
	{ "a165", 0x2799 , 0 },
	{ "a192", 0x279a , 0 },
	{ "a166", 0x279b , 0 },
	{ "a167", 0x279c , 0 },
	{ "a168", 0x279d , 0 },
	{ "a169", 0x279e , 0 },
	{ "a170", 0x279f , 0 },
	{ "a171", 0x27a0 , 0 },
	{ "a172", 0x27a1 , 0 },
	{ "a173", 0x27a2 , 0 },
	{ "a162", 0x27a3 , 0 },
	{ "a174", 0x27a4 , 0 },
	{ "a175", 0x27a5 , 0 },
	{ "a176", 0x27a6 , 0 },
	{ "a177", 0x27a7 , 0 },
	{ "a178", 0x27a8 , 0 },
	{ "a179", 0x27a9 , 0 },
	{ "a193", 0x27aa , 0 },
	{ "a180", 0x27ab , 0 },
	{ "a199", 0x27ac , 0 },
	{ "a181", 0x27ad , 0 },
	{ "a200", 0x27ae , 0 },
	{ "a182", 0x27af , 0 },
	{ "a201", 0x27b1 , 0 },
	{ "a183", 0x27b2 , 0 },
	{ "a184", 0x27b3 , 0 },
	{ "a197", 0x27b4 , 0 },
	{ "a185", 0x27b5 , 0 },
	{ "a194", 0x27b6 , 0 },
	{ "a198", 0x27b7 , 0 },
	{ "a186", 0x27b8 , 0 },
	{ "a195", 0x27b9 , 0 },
	{ "a187", 0x27ba , 0 },
	{ "a188", 0x27bb , 0 },
	{ "a189", 0x27bc , 0 },
	{ "a190", 0x27bd , 0 },
	{ "a191", 0x27be , 0 },
	{ "register.sans", 0xf8e8 , 0 },
	{ "register.serif", 0xf6da , 0 },
	{ "one.superior", 0x00b9 , 0 },
	{ "two.superior", 0x00b2 , 0 },
	{ "three.superior", 0x00b3 , 0 },
	{ "parenleft.superior", 0x207d , 0 },
	{ "parenright.superior", 0x207e , 0 },
	{ "n.superior", 0x207f , 0 },
	{ "parenleft.inferior", 0x208d , 0 },
	{ "parenright.inferior", 0x208e , 0 },
	{ "zero.superior", 0x2070 , 0 },
	{ "four.superior", 0x2074 , 0 },
	{ "five.superior", 0x2075 , 0 },
	{ "six.superior", 0x2076 , 0 },
	{ "seven.superior", 0x2077 , 0 },
	{ "eight.superior", 0x2078 , 0 },
	{ "nine.superior", 0x2079 , 0 },
	{ "zero.inferior", 0x2080 , 0 },
	{ "one.inferior", 0x2081 , 0 },
	{ "two.inferior", 0x2082 , 0 },
	{ "three.inferior", 0x2083 , 0 },
	{ "four.inferior", 0x2084 , 0 },
	{ "five.inferior", 0x2085 , 0 },
	{ "six.inferior", 0x2086 , 0 },
	{ "seven.inferior", 0x2087 , 0 },
	{ "eight.inferior", 0x2088 , 0 },
	{ "nine.inferior", 0x2089 , 0 },
	{ "mu.greek", 0x03bc , 0 },
	{ "Delta.greek", 0x0394 , 0 },
	{ "Omega.greek", 0x03a9 , 0 },
	{ "sigma.end", 0x03c2 , 0 },
	{ "beta.alt", 0x03d0 , 0 },
	{ "kappa.alt", 0x03f0 , 0 },
	{ "theta.alt", 0x03d1 , 0 },
	{ "Upsilon.alt", 0x03d2 , 0 },
	{ "phi.alt", 0x03d5 , 0 },
	{ "pi.alt", 0x03d6 , 0 },
	{ "A.cyr", 0x0410 , 0 },
	{ "Ghe.up", 0x0490 , 0 },
	{ "I.cyr", 0x0418 , 0 },
	{ "I.short", 0x0419 , 0 },
	{ "O.cyr", 0x041e , 0 },
	{ "U.cyr", 0x0423 , 0 },
	{ "E.cyr", 0x042d , 0 },
	{ "Ie.ukran", 0x0404 , 0 },
	{ "I.ukran", 0x0406 , 0 },
	{ "U.short", 0x040e , 0 },
	{ "a.cyr", 0x0430 , 0 },
	{ "ghe.up", 0x0491 , 0 },
	{ "i.cyr", 0x0438 , 0 },
	{ "i.short", 0x0439 , 0 },
	{ "o.cyr", 0x043e , 0 },
	{ "u.cyr", 0x0443 , 0 },
	{ "e.cyr", 0x044d , 0 },
	{ "ie.ukran", 0x0454 , 0 },
	{ "i.ukran", 0x0456 , 0 },
	{ "u.short", 0x045e , 0 },
	{ "ghe.ital", 0xf6c4 , 0 },
	{ "afii10068.ital", 0xf6c4 , 0 },
	{ "afii10066.ital", 0xf6c5 , 0 },
	{ "be.alt", 0xf6c5 , 0 },
	{ "afii10069.ital", 0xf6c6 , 0 },
	{ "de.ital", 0xf6c6 , 0 },
	{ "afii10081.ital", 0xf6c7 , 0 },
	{ "pe.ital", 0xf6c7 , 0 },
	{ "afii10084.ital", 0xf6c8 , 0 },
	{ "te.ital", 0xf6c8 , 0 },
	{ "Omega.math", 0x2126 , 0 },
	{ "Delta.math", 0x2206 , 0 },
	{ "afii57689_afii57804", 0xfb2a , 0 },
	{ "afii57689_afii57803", 0xfb2b , 0 },
	{ "afii57669_afii57806", 0xfb4b , 0 },
	{ "afii57718_afii57798", 0xfb1f , 0 },
	{ "afii57669_afii57807", 0xfb35 , 0 },
	{ ".null", 0x0000 , 0 },
	{ "dialytika_perispomeni", 0x1fc1 , 0 },
	{ "psili_varia", 0x1fcd , 0 },
	{ "psili_oxia", 0x1fce , 0 },
	{ "psili_perispomeni", 0x1fcf , 0 },
	{ "dasia_varia", 0x1fdd , 0 },
	{ "dasia_oxia", 0x1fde , 0 },
	{ "dasia_perispomeni", 0x1fdf , 0 },
	{ "dialytika_varia", 0x1fed , 0 },
	{ "dialytika_oxia", 0x1fee , 0 },
	{ "Alphaacute", 0x1fbb , 0 },
	{ "Alphaasper", 0x1f09 , 0 },
	{ "Alphaasperacute", 0x1f0d , 0 },
	{ "Alphaaspergrave", 0x1f0b , 0 },
	{ "Alphaaspertilde", 0x1f0f , 0 },
	{ "Alphabreve", 0x1fb8 , 0 },
	{ "Alphagrave", 0x1fba , 0 },
	{ "Alphaiotasub", 0x1fbc , 0 },
	{ "Alphaiotasubasper", 0x1f89 , 0 },
	{ "Alphaiotasubasperacute", 0x1f8d , 0 },
	{ "Alphaiotasubaspergrave", 0x1f8b , 0 },
	{ "Alphaiotasubaspertilde", 0x1f8f , 0 },
	{ "Alphaiotasublenis", 0x1f88 , 0 },
	{ "Alphaiotasublenisacute", 0x1f8c , 0 },
	{ "Alphaiotasublenisgrave", 0x1f8a , 0 },
	{ "Alphaiotasublenistilde", 0x1f8e , 0 },
	{ "Alphalenis", 0x1f08 , 0 },
	{ "Alphalenisacute", 0x1f0c , 0 },
	{ "Alphalenisgrave", 0x1f0a , 0 },
	{ "Alphalenistilde", 0x1f0e , 0 },
	{ "Alphamacron", 0x1fb9 , 0 },
	{ "Epsilonacute", 0x1fc9 , 0 },
	{ "Epsilonasper", 0x1f19 , 0 },
	{ "Epsilonasperacute", 0x1f1d , 0 },
	{ "Epsilonaspergrave", 0x1f1b , 0 },
	{ "Epsilongrave", 0x1fc8 , 0 },
	{ "Epsilonlenis", 0x1f18 , 0 },
	{ "Epsilonlenisacute", 0x1f1c , 0 },
	{ "Epsilonlenisgrave", 0x1f1a , 0 },
	{ "Etaacute", 0x1fcb , 0 },
	{ "Etaasper", 0x1f29 , 0 },
	{ "Etaasperacute", 0x1f2d , 0 },
	{ "Etaaspergrave", 0x1f2b , 0 },
	{ "Etaaspertilde", 0x1f2f , 0 },
	{ "Etagrave", 0x1fca , 0 },
	{ "Etaiotasub", 0x1fcc , 0 },
	{ "Etaiotasubasper", 0x1f99 , 0 },
	{ "Etaiotasubasperacute", 0x1f9d , 0 },
	{ "Etaiotasubaspergrave", 0x1f9b , 0 },
	{ "Etaiotasubaspertilde", 0x1f9f , 0 },
	{ "Etaiotasublenis", 0x1f98 , 0 },
	{ "Etaiotasublenisacute", 0x1f9c , 0 },
	{ "Etaiotasublenisgrave", 0x1f9a , 0 },
	{ "Etaiotasublenistilde", 0x1f9e , 0 },
	{ "Etalenis", 0x1f28 , 0 },
	{ "Etalenisacute", 0x1f2c , 0 },
	{ "Etalenisgrave", 0x1f2a , 0 },
	{ "Etalenistilde", 0x1f2e , 0 },
	{ "Iotaacute", 0x1fdb , 0 },
	{ "Iotaasper", 0x1f39 , 0 },
	{ "Iotaasperacute", 0x1f3d , 0 },
	{ "Iotaaspergrave", 0x1f3b , 0 },
	{ "Iotaaspertilde", 0x1f3f , 0 },
	{ "Iotabreve", 0x1fd8 , 0 },
	{ "Iotagrave", 0x1fda , 0 },
	{ "Iotalenis", 0x1f38 , 0 },
	{ "Iotalenisacute", 0x1f3c , 0 },
	{ "Iotalenisgrave", 0x1f3a , 0 },
	{ "Iotalenistilde", 0x1f3e , 0 },
	{ "Iotamacron", 0x1fd9 , 0 },
	{ "Omegaacute", 0x1ffb , 0 },
	{ "Omegaasper", 0x1f69 , 0 },
	{ "Omegaasperacute", 0x1f6d , 0 },
	{ "Omegaaspergrave", 0x1f6b , 0 },
	{ "Omegaaspertilde", 0x1f6f , 0 },
	{ "Omegagrave", 0x1ffa , 0 },
	{ "Omegaiotasub", 0x1ffc , 0 },
	{ "Omegaiotasubasper", 0x1fa9 , 0 },
	{ "Omegaiotasubasperacute", 0x1fad , 0 },
	{ "Omegaiotasubaspergrave", 0x1fab , 0 },
	{ "Omegaiotasubaspertilde", 0x1faf , 0 },
	{ "Omegaiotasublenis", 0x1fa8 , 0 },
	{ "Omegaiotasublenisacute", 0x1fac , 0 },
	{ "Omegaiotasublenisgrave", 0x1faa , 0 },
	{ "Omegaiotasublenistilde", 0x1fae , 0 },
	{ "Omegalenis", 0x1f68 , 0 },
	{ "Omegalenisacute", 0x1f6c , 0 },
	{ "Omegalenisgrave", 0x1f6a , 0 },
	{ "Omegalenistilde", 0x1f6e , 0 },
	{ "Omicronacute", 0x1ff9 , 0 },
	{ "Omicronasper", 0x1f49 , 0 },
	{ "Omicronasperacute", 0x1f4d , 0 },
	{ "Omicronaspergrave", 0x1f4b , 0 },
	{ "Omicrongrave", 0x1ff8 , 0 },
	{ "Omicronlenis", 0x1f48 , 0 },
	{ "Omicronlenisacute", 0x1f4c , 0 },
	{ "Omicronlenisgrave", 0x1f4a , 0 },
	{ "Rhoasper", 0x1fec , 0 },
	{ "Upsilonacute", 0x1feb , 0 },
	{ "Upsilonasper", 0x1f59 , 0 },
	{ "Upsilonasperacute", 0x1f5d , 0 },
	{ "Upsilonaspergrave", 0x1f5b , 0 },
	{ "Upsilonaspertilde", 0x1f5f , 0 },
	{ "Upsilonbreve", 0x1fe8 , 0 },
	{ "Upsilongrave", 0x1fea , 0 },
	{ "Upsilonmacron", 0x1fe9 , 0 },
	{ "numeralsign", 0x0374 , 0 },
	{ "lownumeralsign", 0x0375 , 0 },
	{ "lowcomma", 0x037a , 0 },
	{ "stigma", 0x03da , 0 },
	{ "koppa", 0x03de , 0 },
	{ "sampi", 0x03e0 , 0 },
	{ "lenis", 0x1fbd , 0 },
	{ "iotasubscript", 0x1fbe , 0 },
	{ "dieresistilde", 0x1fc1 , 0 },
	{ "lenisgrave", 0x1fcd , 0 },
	{ "lenisacute", 0x1fce , 0 },
	{ "lenistilde", 0x1fcf , 0 },
	{ "aspergrave", 0x1fdd , 0 },
	{ "asperacute", 0x1fde , 0 },
	{ "aspertilde", 0x1fdf , 0 },
	{ "dialytikagrave", 0x1fed , 0 },
	{ "dialytikaacute", 0x1fee , 0 },
	{ "asper", 0x1ffe , 0 },
	{ "alphalenis", 0x1f00 , 0 },
	{ "alphaasper", 0x1f01 , 0 },
	{ "alphalenisgrave", 0x1f02 , 0 },
	{ "alphaaspergrave", 0x1f03 , 0 },
	{ "alphalenisacute", 0x1f04 , 0 },
	{ "alphaasperacute", 0x1f05 , 0 },
	{ "alphalenistilde", 0x1f06 , 0 },
	{ "alphaaspertilde", 0x1f07 , 0 },
	{ "epsilonlenis", 0x1f10 , 0 },
	{ "epsilonasper", 0x1f11 , 0 },
	{ "epsilonlenisgrave", 0x1f12 , 0 },
	{ "epsilonaspergrave", 0x1f13 , 0 },
	{ "epsilonlenisacute", 0x1f14 , 0 },
	{ "epsilonasperacute", 0x1f15 , 0 },
	{ "etalenis", 0x1f20 , 0 },
	{ "etaasper", 0x1f21 , 0 },
	{ "etalenisgrave", 0x1f22 , 0 },
	{ "etaaspergrave", 0x1f23 , 0 },
	{ "etalenisacute", 0x1f24 , 0 },
	{ "etaasperacute", 0x1f25 , 0 },
	{ "etalenistilde", 0x1f26 , 0 },
	{ "etaaspertilde", 0x1f27 , 0 },
	{ "iotalenis", 0x1f30 , 0 },
	{ "iotaasper", 0x1f31 , 0 },
	{ "iotalenisgrave", 0x1f32 , 0 },
	{ "iotaaspergrave", 0x1f33 , 0 },
	{ "iotalenisacute", 0x1f34 , 0 },
	{ "iotaasperacute", 0x1f35 , 0 },
	{ "iotalenistilde", 0x1f36 , 0 },
	{ "iotaaspertilde", 0x1f37 , 0 },
	{ "omicronlenis", 0x1f40 , 0 },
	{ "omicronasper", 0x1f41 , 0 },
	{ "omicronlenisgrave", 0x1f42 , 0 },
	{ "omicronaspergrave", 0x1f43 , 0 },
	{ "omicronlenisacute", 0x1f44 , 0 },
	{ "omicronasperacute", 0x1f45 , 0 },
	{ "upsilonlenis", 0x1f50 , 0 },
	{ "upsilonasper", 0x1f51 , 0 },
	{ "upsilonlenisgrave", 0x1f52 , 0 },
	{ "upsilonaspergrave", 0x1f53 , 0 },
	{ "upsilonlenisacute", 0x1f54 , 0 },
	{ "upsilonasperacute", 0x1f55 , 0 },
	{ "upsilonlenistilde", 0x1f56 , 0 },
	{ "upsilonaspertilde", 0x1f57 , 0 },
	{ "omegalenis", 0x1f60 , 0 },
	{ "omegaasper", 0x1f61 , 0 },
	{ "omegalenisgrave", 0x1f62 , 0 },
	{ "omegaaspergrave", 0x1f63 , 0 },
	{ "omegalenisacute", 0x1f64 , 0 },
	{ "omegaasperacute", 0x1f65 , 0 },
	{ "omegalenistilde", 0x1f66 , 0 },
	{ "omegaaspertilde", 0x1f67 , 0 },
	{ "alphagrave", 0x1f70 , 0 },
	{ "alphaacute", 0x1f71 , 0 },
	{ "epsilongrave", 0x1f72 , 0 },
	{ "epsilonacute", 0x1f73 , 0 },
	{ "etagrave", 0x1f74 , 0 },
	{ "etaacute", 0x1f75 , 0 },
	{ "iotagrave", 0x1f76 , 0 },
	{ "iotaacute", 0x1f77 , 0 },
	{ "omicrongrave", 0x1f78 , 0 },
	{ "omicronacute", 0x1f79 , 0 },
	{ "upsilongrave", 0x1f7a , 0 },
	{ "upsilonacute", 0x1f7b , 0 },
	{ "omegagrave", 0x1f7c , 0 },
	{ "omegaacute", 0x1f7d , 0 },
	{ "alphaiotasublenis", 0x1f80 , 0 },
	{ "alphaiotasubasper", 0x1f81 , 0 },
	{ "alphaiotasublenisgrave", 0x1f82 , 0 },
	{ "alphaiotasubaspergrave", 0x1f83 , 0 },
	{ "alphaiotasublenisacute", 0x1f84 , 0 },
	{ "alphaiotasubasperacute", 0x1f85 , 0 },
	{ "alphaiotasublenistilde", 0x1f86 , 0 },
	{ "alphaiotasubaspertilde", 0x1f87 , 0 },
	{ "etaiotasublenis", 0x1f90 , 0 },
	{ "etaiotasubasper", 0x1f91 , 0 },
	{ "etaiotasublenisgrave", 0x1f92 , 0 },
	{ "etaiotasubaspergrave", 0x1f93 , 0 },
	{ "etaiotasublenisacute", 0x1f94 , 0 },
	{ "etaiotasubasperacute", 0x1f95 , 0 },
	{ "etaiotasublenistilde", 0x1f96 , 0 },
	{ "etaiotasubaspertilde", 0x1f97 , 0 },
	{ "omegaiotasublenis", 0x1fa0 , 0 },
	{ "omegaiotasubasper", 0x1fa1 , 0 },
	{ "omegaiotasublenisgrave", 0x1fa2 , 0 },
	{ "omegaiotasubaspergrave", 0x1fa3 , 0 },
	{ "omegaiotasublenisacute", 0x1fa4 , 0 },
	{ "omegaiotasubasperacute", 0x1fa5 , 0 },
	{ "omegaiotasublenistilde", 0x1fa6 , 0 },
	{ "omegaiotasubaspertilde", 0x1fa7 , 0 },
	{ "alphabreve", 0x1fb0 , 0 },
	{ "alphamacron", 0x1fb1 , 0 },
	{ "alphaiotasubgrave", 0x1fb2 , 0 },
	{ "alphaiotasub", 0x1fb3 , 0 },
	{ "alphatilde", 0x1fb6 , 0 },
	{ "alphaiotasubtilde", 0x1fb7 , 0 },
	{ "etaiotasubgrave", 0x1fc2 , 0 },
	{ "etaiotasub", 0x1fc3 , 0 },
	{ "etaiotasubacute", 0x1fc4 , 0 },
	{ "etatilde", 0x1fc6 , 0 },
	{ "etaiotasubtilde", 0x1fc7 , 0 },
	{ "iotabreve", 0x1fd0 , 0 },
	{ "iotamacron", 0x1fd1 , 0 },
	{ "iotadieresisgrave", 0x1fd2 , 0 },
	{ "iotadieresisacute", 0x1fd3 , 0 },
	{ "iotatilde", 0x1fd6 , 0 },
	{ "iotadieresistilde", 0x1fd7 , 0 },
	{ "upsilonbreve", 0x1fe0 , 0 },
	{ "upsilonmacron", 0x1fe1 , 0 },
	{ "upsilondieresisgrave", 0x1fe2 , 0 },
	{ "upsilondieresisacute", 0x1fe3 , 0 },
	{ "rholenis", 0x1fe4 , 0 },
	{ "rhoasper", 0x1fe5 , 0 },
	{ "upsilontilde", 0x1fe6 , 0 },
	{ "omegaiotasubgrave", 0x1ff2 , 0 },
	{ "omegaiotasub", 0x1ff3 , 0 },
	{ "omegaiotasubacute", 0x1ff4 , 0 },
	{ "omegatilde", 0x1ff6 , 0 },
	{ "omegaiotasubtilde", 0x1ff7 , 0 },
	{ "alphaiotasubacute", 0x1fb4 , 0 },
	{ "upsilondieresistilde", 0x1fe7 , 0 },
	{ "digamma", 0xefd5 , 0 },
	{ "Be", 0x0411 , 0 },
	{ "Ve", 0x0412 , 0 },
	{ "Ghe", 0x0413 , 0 },
	{ "De", 0x0414 , 0 },
	{ "Ie", 0x0415 , 0 },
	{ "Zhe", 0x0416 , 0 },
	{ "Ze", 0x0417 , 0 },
	{ "Ka", 0x041a , 0 },
	{ "El", 0x041b , 0 },
	{ "Em", 0x041c , 0 },
	{ "En", 0x041d , 0 },
	{ "Pe", 0x041f , 0 },
	{ "Er", 0x0420 , 0 },
	{ "Es", 0x0421 , 0 },
	{ "Te", 0x0422 , 0 },
	{ "Ef", 0x0424 , 0 },
	{ "Ha", 0x0425 , 0 },
	{ "Tse", 0x0426 , 0 },
	{ "Che", 0x0427 , 0 },
	{ "Sha", 0x0428 , 0 },
	{ "Shcha", 0x0429 , 0 },
	{ "Hard", 0x042a , 0 },
	{ "Yeru", 0x042b , 0 },
	{ "Soft", 0x042c , 0 },
	{ "Yu", 0x042e , 0 },
	{ "Ya", 0x042f , 0 },
	{ "Io", 0x0401 , 0 },
	{ "Dje", 0x0402 , 0 },
	{ "Gje", 0x0403 , 0 },
	{ "Dze", 0x0405 , 0 },
	{ "Yi", 0x0407 , 0 },
	{ "Je", 0x0408 , 0 },
	{ "Lje", 0x0409 , 0 },
	{ "Nje", 0x040a , 0 },
	{ "Tshe", 0x040b , 0 },
	{ "Kje", 0x040c , 0 },
	{ "Dzhe", 0x040f , 0 },
	{ "Yat", 0x0462 , 0 },
	{ "Fita", 0x0472 , 0 },
	{ "Izhitsa", 0x0474 , 0 },
	{ "be", 0x0431 , 0 },
	{ "ve", 0x0432 , 0 },
	{ "ghe", 0x0433 , 0 },
	{ "de", 0x0434 , 0 },
	{ "ie", 0x0435 , 0 },
	{ "zhe", 0x0436 , 0 },
	{ "ze", 0x0437 , 0 },
	{ "ka", 0x043a , 0 },
	{ "el", 0x043b , 0 },
	{ "em", 0x043c , 0 },
	{ "en", 0x043d , 0 },
	{ "er", 0x0440 , 0 },
	{ "es", 0x0441 , 0 },
	{ "te", 0x0442 , 0 },
	{ "ef", 0x0444 , 0 },
	{ "ha", 0x0445 , 0 },
	{ "tse", 0x0446 , 0 },
	{ "che", 0x0447 , 0 },
	{ "sha", 0x0448 , 0 },
	{ "shcha", 0x0449 , 0 },
	{ "hard", 0x044a , 0 },
	{ "yeru", 0x044b , 0 },
	{ "soft", 0x044c , 0 },
	{ "yu", 0x044e , 0 },
	{ "ya", 0x044f , 0 },
	{ "io", 0x0451 , 0 },
	{ "dje", 0x0452 , 0 },
	{ "gje", 0x0453 , 0 },
	{ "dze", 0x0455 , 0 },
	{ "yi", 0x0457 , 0 },
	{ "je", 0x0458 , 0 },
	{ "lje", 0x0459 , 0 },
	{ "nje", 0x045a , 0 },
	{ "tshe", 0x045b , 0 },
	{ "kje", 0x045c , 0 },
	{ "dzhe", 0x045f , 0 },
	{ "yat", 0x0463 , 0 },
	{ "fita", 0x0473 , 0 },
	{ "izhitsa", 0x0475 , 0 },
	{ "litre", 0x2113 , 0 },
	{ "Dotaccent", 0xefed , 0 },
	{ "Breve", 0xefee , 0 },
	{ "Ogonek", 0xeff1 , 0 },
	{ "Cedilla", 0xeff2 , 0 },
	{ "Ring", 0xeff3 , 0 },
	{ "Tilde", 0xeff5 , 0 },
	{ "Circumflex", 0xeff7 , 0 },
	{ "dbar", 0x0111 , 0 },
	{ "st", 0xfb06 , 0 },
	{ "TeXtext32", 0x0337 , 0 },
	{ "DD070", 0x2014 , 0 },
	{ "monospacedash", 0x2014 , 0 },
	{ "Dash1cent", 0x2015 , 0 },
	{ "apostrophereverse", 0x201b , 0 },
	{ "Pts", 0x20a7 , 0 },
	{ "SM760000", 0x2195 , 0 },
	{ "SM770000", 0x21a8 , 0 },
	{ "SP320000", 0x2219 , 0 },
	{ "SD630000", 0x22c5 , 0 },
	{ "SM790000", 0x2302 , 0 },
	{ "caretinverted", 0x2304 , 0 },
	{ "SM680000", 0x2310 , 0 },
	{ "SA420000", 0x2319 , 0 },
	{ "BorderULCorner1", 0x231c , 0 },
	{ "DD010", 0x231c , 0 },
	{ "DD101", 0x231c , 0 },
	{ "BorderURCorner1", 0x231d , 0 },
	{ "DD012", 0x231d , 0 },
	{ "DD104", 0x231d , 0 },
	{ "BorderLLCorner1", 0x231e , 0 },
	{ "DD014", 0x231e , 0 },
	{ "DD109", 0x231e , 0 },
	{ "BorderLRCorner1", 0x231f , 0 },
	{ "DD015", 0x231f , 0 },
	{ "DD112", 0x231f , 0 },
	{ "SS260000", 0x2320 , 0 },
	{ "SS270000", 0x2321 , 0 },
	{ "hook", 0x2440 , 0 },
	{ "chair", 0x2441 , 0 },
	{ "fork", 0x2442 , 0 },
	{ "SF600000", 0x2580 , 0 },
	{ "BorderLower1", 0x2581 , 0 },
	{ "DD011", 0x2581 , 0 },
	{ "DD021", 0x2581 , 0 },
	{ "DD110", 0x2581 , 0 },
	{ "Flower1l", 0x2581 , 0 },
	{ "SF570000", 0x2584 , 0 },
	{ "SF610000", 0x2588 , 0 },
	{ "Tile", 0x2588 , 0 },
	{ "HalfTile", 0x258c , 0 },
	{ "SF580000", 0x258c , 0 },
	{ "BorderLeftVertical1", 0x258f , 0 },
	{ "DD013", 0x258f , 0 },
	{ "DD034", 0x258f , 0 },
	{ "DD105", 0x258f , 0 },
	{ "Flower1d", 0x258f , 0 },
	{ "SF590000", 0x2590 , 0 },
	{ "SF140000", 0x2591 , 0 },
	{ "SF150000", 0x2592 , 0 },
	{ "SF160000", 0x2593 , 0 },
	{ "BorderUpper1", 0x2594 , 0 },
	{ "DD024", 0x2594 , 0 },
	{ "DD102", 0x2594 , 0 },
	{ "BorderRightVertical1", 0x2595 , 0 },
	{ "DD031", 0x2595 , 0 },
	{ "DD106", 0x2595 , 0 },
	{ "Flower2dr", 0x2595 , 0 },
	{ "SM600000", 0x25b2 , 0 },
	{ "SM590000", 0x25b6 , 0 },
	{ "SV040000", 0x25bc , 0 },
	{ "SM630000", 0x25c0 , 0 },
	{ "SM750000", 0x25cb , 0 },
	{ "SM570001", 0x25d8 , 0 },
	{ "SM750002", 0x25d9 , 0 },
	{ "ShootingStar", 0x2604 , 0 },
	{ "DD130", 0x2605 , 0 },
	{ "StarDingbat1", 0x2605 , 0 },
	{ "TheSun", 0x2609 , 0 },
	{ "FleuronLeft", 0x2619 , 0 },
	{ "cross", 0x2629 , 0 },
	{ "SS000000", 0x263a , 0 },
	{ "SS010000", 0x263b , 0 },
	{ "SM690000", 0x263c , 0 },
	{ "TheMoon", 0x263d , 0 },
	{ "Moon", 0x263e , 0 },
	{ "Mercury", 0x263f , 0 },
	{ "Venus", 0x2640 , 0 },
	{ "Earth", 0x2641 , 0 },
	{ "Mars", 0x2642 , 0 },
	{ "Jupiter", 0x2643 , 0 },
	{ "Saturn", 0x2644 , 0 },
	{ "Uranus", 0x2645 , 0 },
	{ "Neptune", 0x2646 , 0 },
	{ "Pluto", 0x2647 , 0 },
	{ "Aries", 0x2648 , 0 },
	{ "Taurus", 0x2649 , 0 },
	{ "Gemini", 0x264a , 0 },
	{ "Cancer", 0x264b , 0 },
	{ "Leo", 0x264c , 0 },
	{ "Virgo", 0x264d , 0 },
	{ "Libra", 0x264e , 0 },
	{ "Scorpio", 0x264f , 0 },
	{ "Sagittarius", 0x2650 , 0 },
	{ "Capricorn", 0x2651 , 0 },
	{ "Aquarius", 0x2652 , 0 },
	{ "Pisces", 0x2653 , 0 },
	{ "diamondopen", 0x2662 , 0 },
	{ "FleuronCenter", 0x2766 , 0 },
	{ "FleuronRight", 0x2767 , 0 },
	{ "st1", 0xfb06 , 0 },
	{ "zeroslash", 0xf638 , 0 },
	{ "pi1", 0x03d6 , 0 },
	{ "hyphen-minus", 0x002d , 0 },
	{ "hyphenminus", 0x002d , 0 },
	{ "nonmarkingreturn", 0x000d , 0 },
	{ "micro", 0x00b5 , 0 },
	{ "Dmacron", 0x0110 , 0 },
	{ "kra", 0x0138 , 0 },
	{ "bbar", 0x0180 , 0 },
	{ "Dbar", 0x0189 , 0 },
	{ "deltaturn", 0x018d , 0 },
	{ "Eturn", 0x018e , 0 },
	{ "Epsilonlatin", 0x0190 , 0 },
	{ "Gammalatin", 0x0194 , 0 },
	{ "Iotalatin", 0x0196 , 0 },
	{ "Ibar", 0x0197 , 0 },
	{ "lambdabar", 0x019b , 0 },
	{ "mcapturn", 0x019c , 0 },
	{ "Nhook", 0x019d , 0 },
	{ "nleg", 0x019e , 0 },
	{ "Obar", 0x019f , 0 },
	{ "OI", 0x01a2 , 0 },
	{ "YR", 0x01a6 , 0 },
	{ "eshlooprev", 0x01aa , 0 },
	{ "Trthook", 0x01ae , 0 },
	{ "Upsilonlatin", 0x01b1 , 0 },
	{ "Vcursive", 0x01b2 , 0 },
	{ "Zbar", 0x01b5 , 0 },
	{ "zbar", 0x01b6 , 0 },
	{ "Yogh", 0x01b7 , 0 },
	{ "Yoghrev", 0x01b8 , 0 },
	{ "yoghrev", 0x01b9 , 0 },
	{ "yoghtail", 0x01ba , 0 },
	{ "twobar", 0x01bb , 0 },
	{ "glottalstopbarinv", 0x01be , 0 },
	{ "pipe", 0x01c0 , 0 },
	{ "pipedbl", 0x01c1 , 0 },
	{ "pipedblbar", 0x01c2 , 0 },
	{ "exclamlatin", 0x01c3 , 0 },
	{ "DZhacek", 0x01c4 , 0 },
	{ "Dzhacek", 0x01c5 , 0 },
	{ "dzhacek", 0x01c6 , 0 },
	{ "Ahacek", 0x01cd , 0 },
	{ "ahacek", 0x01ce , 0 },
	{ "Ihacek", 0x01cf , 0 },
	{ "ihacek", 0x01d0 , 0 },
	{ "Ohacek", 0x01d1 , 0 },
	{ "ohacek", 0x01d2 , 0 },
	{ "Uhacek", 0x01d3 , 0 },
	{ "uhacek", 0x01d4 , 0 },
	{ "Udieresishacek", 0x01d9 , 0 },
	{ "udieresishacek", 0x01da , 0 },
	{ "eturn", 0x01dd , 0 },
	{ "Gbar", 0x01e4 , 0 },
	{ "gbar", 0x01e5 , 0 },
	{ "Ghacek", 0x01e6 , 0 },
	{ "ghacek", 0x01e7 , 0 },
	{ "Khacek", 0x01e8 , 0 },
	{ "khacek", 0x01e9 , 0 },
	{ "Yoghhacek", 0x01ee , 0 },
	{ "yoghhacek", 0x01ef , 0 },
	{ "jhacek", 0x01f0 , 0 },
	{ "aturn", 0x0250 , 0 },
	{ "ascriptturn", 0x0252 , 0 },
	{ "cturn", 0x0254 , 0 },
	{ "drthook", 0x0256 , 0 },
	{ "erev", 0x0258 , 0 },
	{ "epsilonlatin", 0x025b , 0 },
	{ "epsilonlatinrev", 0x025c , 0 },
	{ "epsilonlatinrevhook", 0x025d , 0 },
	{ "epsilonlatinrevclosed", 0x025e , 0 },
	{ "jdotlessbar", 0x025f , 0 },
	{ "gcursive", 0x0261 , 0 },
	{ "Gsmallcap", 0x0262 , 0 },
	{ "gammalatin", 0x0263 , 0 },
	{ "babygamma", 0x0264 , 0 },
	{ "hturn", 0x0265 , 0 },
	{ "ibar", 0x0268 , 0 },
	{ "Ismallcap", 0x026a , 0 },
	{ "lmidtilde", 0x026b , 0 },
	{ "lrthook", 0x026d , 0 },
	{ "lyogh", 0x026e , 0 },
	{ "mturn", 0x026f , 0 },
	{ "mturndescend", 0x0270 , 0 },
	{ "nlftlfthook", 0x0272 , 0 },
	{ "nrthook", 0x0273 , 0 },
	{ "Nsmallcap", 0x0274 , 0 },
	{ "obar", 0x0275 , 0 },
	{ "OEsmallcap", 0x0276 , 0 },
	{ "omegaclosed", 0x0277 , 0 },
	{ "rturn", 0x0279 , 0 },
	{ "rturnascend", 0x027a , 0 },
	{ "rturnrthook", 0x027b , 0 },
	{ "rdescend", 0x027c , 0 },
	{ "rrthook", 0x027d , 0 },
	{ "rfishhookrev", 0x027f , 0 },
	{ "Rsmallcap", 0x0280 , 0 },
	{ "Rsmallcapinv", 0x0281 , 0 },
	{ "srthook", 0x0282 , 0 },
	{ "jhookdblbar", 0x0284 , 0 },
	{ "eshshortrev", 0x0285 , 0 },
	{ "tturn", 0x0287 , 0 },
	{ "trthook", 0x0288 , 0 },
	{ "vscript", 0x028b , 0 },
	{ "vturn", 0x028c , 0 },
	{ "wturn", 0x028d , 0 },
	{ "yturn", 0x028e , 0 },
	{ "Ysmallcap", 0x028f , 0 },
	{ "zrthook", 0x0290 , 0 },
	{ "yogh", 0x0292 , 0 },
	{ "yoghcurl", 0x0293 , 0 },
	{ "glottalstoprevinv", 0x0295 , 0 },
	{ "glottalstopinv", 0x0296 , 0 },
	{ "cstretch", 0x0297 , 0 },
	{ "kiss", 0x0298 , 0 },
	{ "Bsmallcap", 0x0299 , 0 },
	{ "epsilonclosed", 0x029a , 0 },
	{ "Gsmallcaphook", 0x029b , 0 },
	{ "Hsmallcap", 0x029c , 0 },
	{ "jcrosstail", 0x029d , 0 },
	{ "kturn", 0x029e , 0 },
	{ "Lsmallcap", 0x029f , 0 },
	{ "glottalstopbar", 0x02a1 , 0 },
	{ "glottalstopbarrev", 0x02a2 , 0 },
	{ "dyogh", 0x02a4 , 0 },
	{ "hsuper", 0x02b0 , 0 },
	{ "hhooksuper", 0x02b1 , 0 },
	{ "jsuper", 0x02b2 , 0 },
	{ "rsuper", 0x02b3 , 0 },
	{ "rturnsuper", 0x02b4 , 0 },
	{ "rturnrthooksuper", 0x02b5 , 0 },
	{ "Rturnsuper", 0x02b6 , 0 },
	{ "wsuper", 0x02b7 , 0 },
	{ "ysuper", 0x02b8 , 0 },
	{ "primedblmod", 0x02ba , 0 },
	{ "quoteleftmod", 0x02bb , 0 },
	{ "apostrophe", 0x02bc , 0 },
	{ "apostropherev", 0x02bd , 0 },
	{ "ringrighthalfsuper", 0x02be , 0 },
	{ "ringlefthalfsuper", 0x02bf , 0 },
	{ "glottal", 0x02c0 , 0 },
	{ "glottalrev", 0x02c1 , 0 },
	{ "fronted", 0x02c2 , 0 },
	{ "backed", 0x02c3 , 0 },
	{ "raised", 0x02c4 , 0 },
	{ "lowered", 0x02c5 , 0 },
	{ "linevert", 0x02c8 , 0 },
	{ "macronmodifier", 0x02c9 , 0 },
	{ "acutemodifier", 0x02ca , 0 },
	{ "gravemodifier", 0x02cb , 0 },
	{ "linevertsub", 0x02cc , 0 },
	{ "macronsub", 0x02cd , 0 },
	{ "gravesub", 0x02ce , 0 },
	{ "acutesub", 0x02cf , 0 },
	{ "length", 0x02d0 , 0 },
	{ "halflength", 0x02d1 , 0 },
	{ "ringrighthalfcenter", 0x02d2 , 0 },
	{ "ringlefthalfsup", 0x02d3 , 0 },
	{ "tackupmid", 0x02d4 , 0 },
	{ "tackdownmid", 0x02d5 , 0 },
	{ "rhotichook", 0x02de , 0 },
	{ "gammasuper", 0x02e0 , 0 },
	{ "lsuper", 0x02e1 , 0 },
	{ "ssuper", 0x02e2 , 0 },
	{ "xsuper", 0x02e3 , 0 },
	{ "glottalrevsuper", 0x02e4 , 0 },
	{ "toneextrahigh", 0x02e5 , 0 },
	{ "tonehigh", 0x02e6 , 0 },
	{ "tonemid", 0x02e7 , 0 },
	{ "tonelow", 0x02e8 , 0 },
	{ "toneextralow", 0x02e9 , 0 },
	{ "gravenosp", 0x0300 , 0 },
	{ "acutenosp", 0x0301 , 0 },
	{ "circumflexnosp", 0x0302 , 0 },
	{ "tildenosp", 0x0303 , 0 },
	{ "macronnosp", 0x0304 , 0 },
	{ "overscorenosp", 0x0305 , 0 },
	{ "brevenosp", 0x0306 , 0 },
	{ "dotnosp", 0x0307 , 0 },
	{ "dieresisnosp", 0x0308 , 0 },
	{ "hooksupnosp", 0x0309 , 0 },
	{ "ringnosp", 0x030a , 0 },
	{ "acutedblnosp", 0x030b , 0 },
	{ "haceknosp", 0x030c , 0 },
	{ "linevertnosp", 0x030d , 0 },
	{ "linevertdblnosp", 0x030e , 0 },
	{ "gravedblnosp", 0x030f , 0 },
	{ "candrabindunosp", 0x0310 , 0 },
	{ "breveinvnosp", 0x0311 , 0 },
	{ "commaturnsupnosp", 0x0312 , 0 },
	{ "apostrophesupnosp", 0x0313 , 0 },
	{ "commasuprevnosp", 0x0314 , 0 },
	{ "commasuprightnosp", 0x0315 , 0 },
	{ "gravesubnosp", 0x0316 , 0 },
	{ "acutesubnosp", 0x0317 , 0 },
	{ "tackleftsubnosp", 0x0318 , 0 },
	{ "tackrightsubnosp", 0x0319 , 0 },
	{ "anglesupnosp", 0x031a , 0 },
	{ "hornnosp", 0x031b , 0 },
	{ "ringlefthalfsubnosp", 0x031c , 0 },
	{ "tackupsubnosp", 0x031d , 0 },
	{ "tackdownsubnosp", 0x031e , 0 },
	{ "plussubnosp", 0x031f , 0 },
	{ "minussubnosp", 0x0320 , 0 },
	{ "hooksubpalatnosp", 0x0321 , 0 },
	{ "hooksubretronosp", 0x0322 , 0 },
	{ "dotsubnosp", 0x0323 , 0 },
	{ "dotdblsubnosp", 0x0324 , 0 },
	{ "ringsubnosp", 0x0325 , 0 },
	{ "commasubnosp", 0x0326 , 0 },
	{ "cedillanosp", 0x0327 , 0 },
	{ "ogoneknosp", 0x0328 , 0 },
	{ "linevertsubnosp", 0x0329 , 0 },
	{ "bridgesubnosp", 0x032a , 0 },
	{ "archdblsubnosp", 0x032b , 0 },
	{ "haceksubnosp", 0x032c , 0 },
	{ "circumflexsubnosp", 0x032d , 0 },
	{ "brevesubnosp", 0x032e , 0 },
	{ "breveinvsubnosp", 0x032f , 0 },
	{ "tildesubnosp", 0x0330 , 0 },
	{ "macronsubnosp", 0x0331 , 0 },
	{ "underscorenosp", 0x0332 , 0 },
	{ "underscoredblnosp", 0x0333 , 0 },
	{ "tildemidnosp", 0x0334 , 0 },
	{ "barmidshortnosp", 0x0335 , 0 },
	{ "barmidlongnosp", 0x0336 , 0 },
	{ "slashshortnosp", 0x0337 , 0 },
	{ "slashlongnosp", 0x0338 , 0 },
	{ "ringrighthalfsubnosp", 0x0339 , 0 },
	{ "bridgeinvsubnosp", 0x033a , 0 },
	{ "squaresubnosp", 0x033b , 0 },
	{ "seagullsubnosp", 0x033c , 0 },
	{ "xsupnosp", 0x033d , 0 },
	{ "tildevertsupnosp", 0x033e , 0 },
	{ "overscoredblnosp", 0x033f , 0 },
	{ "graveleftnosp", 0x0340 , 0 },
	{ "acuterightnosp", 0x0341 , 0 },
	{ "wavyoverscorenosp", 0x0342 , 0 },
	{ "zigzagoverscorenosp", 0x0343 , 0 },
	{ "diaeresistonosnosp", 0x0344 , 0 },
	{ "iotasubnosp", 0x0345 , 0 },
	{ "iotasub", 0x037a , 0 },
	{ "diaeresistonos", 0x0385 , 0 },
	{ "iotadiaeresistonos", 0x0390 , 0 },
	{ "Iotadiaeresis", 0x03aa , 0 },
	{ "Upsilondiaeresis", 0x03ab , 0 },
	{ "upsilondiaeresistonos", 0x03b0 , 0 },
	{ "iotadiaeresis", 0x03ca , 0 },
	{ "upsilondiaeresis", 0x03cb , 0 },
	{ "betacurled", 0x03d0 , 0 },
	{ "thetascript", 0x03d1 , 0 },
	{ "Upsilonhook", 0x03d2 , 0 },
	{ "Upsilonhooktonos", 0x03d3 , 0 },
	{ "Upsilonhookdiaeresis", 0x03d4 , 0 },
	{ "phiscript", 0x03d5 , 0 },
	{ "omegapi", 0x03d6 , 0 },
	{ "Stigma", 0x03da , 0 },
	{ "stigma", 0x03db , 0 },
	{ "Digamma", 0x03dc , 0 },
	{ "digamma", 0x03dd , 0 },
	{ "Koppa", 0x03de , 0 },
	{ "koppa", 0x03df , 0 },
	{ "Sampi", 0x03e0 , 0 },
	{ "sampi", 0x03e1 , 0 },
	{ "Shei", 0x03e2 , 0 },
	{ "shei", 0x03e3 , 0 },
	{ "Fei", 0x03e4 , 0 },
	{ "fei", 0x03e5 , 0 },
	{ "Khei", 0x03e6 , 0 },
	{ "khei", 0x03e7 , 0 },
	{ "Hori", 0x03e8 , 0 },
	{ "hori", 0x03e9 , 0 },
	{ "Gangia", 0x03ea , 0 },
	{ "gangia", 0x03eb , 0 },
	{ "Shima", 0x03ec , 0 },
	{ "shima", 0x03ed , 0 },
	{ "Dei", 0x03ee , 0 },
	{ "dei", 0x03ef , 0 },
	{ "kappascript", 0x03f0 , 0 },
	{ "sigmalunate", 0x03f2 , 0 },
	{ "Io", 0x0401 , 0 },
	{ "Dje", 0x0402 , 0 },
	{ "Gje", 0x0403 , 0 },
	{ "Ecyril", 0x0404 , 0 },
	{ "Dze", 0x0405 , 0 },
	{ "Icyril", 0x0406 , 0 },
	{ "Yi", 0x0407 , 0 },
	{ "Je", 0x0408 , 0 },
	{ "Lje", 0x0409 , 0 },
	{ "Nje", 0x040a , 0 },
	{ "Tshe", 0x040b , 0 },
	{ "Kje", 0x040c , 0 },
	{ "Ucyrilbreve", 0x040e , 0 },
	{ "Dzhe", 0x040f , 0 },
	{ "Acyril", 0x0410 , 0 },
	{ "Be", 0x0411 , 0 },
	{ "Ve", 0x0412 , 0 },
	{ "Ge", 0x0413 , 0 },
	{ "De", 0x0414 , 0 },
	{ "Ie", 0x0415 , 0 },
	{ "Zhe", 0x0416 , 0 },
	{ "Ze", 0x0417 , 0 },
	{ "Ii", 0x0418 , 0 },
	{ "Iibreve", 0x0419 , 0 },
	{ "Ka", 0x041a , 0 },
	{ "El", 0x041b , 0 },
	{ "Em", 0x041c , 0 },
	{ "En", 0x041d , 0 },
	{ "Ocyril", 0x041e , 0 },
	{ "Pecyril", 0x041f , 0 },
	{ "Er", 0x0420 , 0 },
	{ "Es", 0x0421 , 0 },
	{ "Te", 0x0422 , 0 },
	{ "Ucyril", 0x0423 , 0 },
	{ "Ef", 0x0424 , 0 },
	{ "Kha", 0x0425 , 0 },
	{ "Tse", 0x0426 , 0 },
	{ "Che", 0x0427 , 0 },
	{ "Sha", 0x0428 , 0 },
	{ "Shcha", 0x0429 , 0 },
	{ "Hard", 0x042a , 0 },
	{ "Yeri", 0x042b , 0 },
	{ "Soft", 0x042c , 0 },
	{ "Ecyrilrev", 0x042d , 0 },
	{ "Iu", 0x042e , 0 },
	{ "Ia", 0x042f , 0 },
	{ "acyril", 0x0430 , 0 },
	{ "be", 0x0431 , 0 },
	{ "ve", 0x0432 , 0 },
	{ "ge", 0x0433 , 0 },
	{ "de", 0x0434 , 0 },
	{ "ie", 0x0435 , 0 },
	{ "zhe", 0x0436 , 0 },
	{ "ze", 0x0437 , 0 },
	{ "ii", 0x0438 , 0 },
	{ "iibreve", 0x0439 , 0 },
	{ "ka", 0x043a , 0 },
	{ "el", 0x043b , 0 },
	{ "em", 0x043c , 0 },
	{ "en", 0x043d , 0 },
	{ "ocyril", 0x043e , 0 },
	{ "pecyril", 0x043f , 0 },
	{ "er", 0x0440 , 0 },
	{ "es", 0x0441 , 0 },
	{ "te", 0x0442 , 0 },
	{ "ucyril", 0x0443 , 0 },
	{ "ef", 0x0444 , 0 },
	{ "kha", 0x0445 , 0 },
	{ "tse", 0x0446 , 0 },
	{ "che", 0x0447 , 0 },
	{ "sha", 0x0448 , 0 },
	{ "shcha", 0x0449 , 0 },
	{ "hard", 0x044a , 0 },
	{ "yeri", 0x044b , 0 },
	{ "soft", 0x044c , 0 },
	{ "ecyrilrev", 0x044d , 0 },
	{ "iu", 0x044e , 0 },
	{ "ia", 0x044f , 0 },
	{ "io", 0x0451 , 0 },
	{ "dje", 0x0452 , 0 },
	{ "gje", 0x0453 , 0 },
	{ "ecyril", 0x0454 , 0 },
	{ "dze", 0x0455 , 0 },
	{ "icyril", 0x0456 , 0 },
	{ "yi", 0x0457 , 0 },
	{ "je", 0x0458 , 0 },
	{ "lje", 0x0459 , 0 },
	{ "nje", 0x045a , 0 },
	{ "tshe", 0x045b , 0 },
	{ "kje", 0x045c , 0 },
	{ "ucyrilbreve", 0x045e , 0 },
	{ "dzhe", 0x045f , 0 },
	{ "Yat", 0x0462 , 0 },
	{ "yat", 0x0463 , 0 },
	{ "Yusbig", 0x046a , 0 },
	{ "yusbig", 0x046b , 0 },
	{ "Psicyril", 0x0470 , 0 },
	{ "psicyril", 0x0471 , 0 },
	{ "Fita", 0x0472 , 0 },
	{ "fita", 0x0473 , 0 },
	{ "Izhitsa", 0x0474 , 0 },
	{ "izhitsa", 0x0475 , 0 },
	{ "Izhitsagravedbl", 0x0476 , 0 },
	{ "izhitsagravedbl", 0x0477 , 0 },
	{ "Digraphuk", 0x0478 , 0 },
	{ "digraphuk", 0x0479 , 0 },
	{ "Omegaround", 0x047a , 0 },
	{ "omegaround", 0x047b , 0 },
	{ "Omegatitlo", 0x047c , 0 },
	{ "omegatitlo", 0x047d , 0 },
	{ "OT", 0x047e , 0 },
	{ "ot", 0x047f , 0 },
	{ "Geupturn", 0x0490 , 0 },
	{ "geupturn", 0x0491 , 0 },
	{ "Gebar", 0x0492 , 0 },
	{ "gebar", 0x0493 , 0 },
	{ "Gehook", 0x0494 , 0 },
	{ "gehook", 0x0495 , 0 },
	{ "Zhertdes", 0x0496 , 0 },
	{ "zhertdes", 0x0497 , 0 },
	{ "Zecedilla", 0x0498 , 0 },
	{ "zecedilla", 0x0499 , 0 },
	{ "Kartdes", 0x049a , 0 },
	{ "kartdes", 0x049b , 0 },
	{ "Kavertbar", 0x049c , 0 },
	{ "kavertbar", 0x049d , 0 },
	{ "Kabar", 0x049e , 0 },
	{ "kabar", 0x049f , 0 },
	{ "GeKarev", 0x04a0 , 0 },
	{ "gekarev", 0x04a1 , 0 },
	{ "Enrtdes", 0x04a2 , 0 },
	{ "enrtdes", 0x04a3 , 0 },
	{ "EnGe", 0x04a4 , 0 },
	{ "enge", 0x04a5 , 0 },
	{ "Pehook", 0x04a6 , 0 },
	{ "pehook", 0x04a7 , 0 },
	{ "Ohook", 0x04a8 , 0 },
	{ "ohook", 0x04a9 , 0 },
	{ "Escedilla", 0x04aa , 0 },
	{ "escedilla", 0x04ab , 0 },
	{ "Tertdes", 0x04ac , 0 },
	{ "tertdes", 0x04ad , 0 },
	{ "Ustrt", 0x04ae , 0 },
	{ "ustrt", 0x04af , 0 },
	{ "Ustrtbar", 0x04b0 , 0 },
	{ "ustrtbar", 0x04b1 , 0 },
	{ "Khartdes", 0x04b2 , 0 },
	{ "khartdes", 0x04b3 , 0 },
	{ "TeTse", 0x04b4 , 0 },
	{ "tetse", 0x04b5 , 0 },
	{ "Chertdes", 0x04b6 , 0 },
	{ "chertdes", 0x04b7 , 0 },
	{ "Chevertbar", 0x04b8 , 0 },
	{ "chevertbar", 0x04b9 , 0 },
	{ "Hcyril", 0x04ba , 0 },
	{ "hcyril", 0x04bb , 0 },
	{ "Iehook", 0x04bc , 0 },
	{ "iehook", 0x04bd , 0 },
	{ "Iehookogonek", 0x04be , 0 },
	{ "iehookogonek", 0x04bf , 0 },
	{ "Icyril1", 0x04c0 , 0 },
	{ "Zhebreve", 0x04c1 , 0 },
	{ "zhebreve", 0x04c2 , 0 },
	{ "Kahook", 0x04c3 , 0 },
	{ "kahook", 0x04c4 , 0 },
	{ "Enhook", 0x04c7 , 0 },
	{ "enhook", 0x04c8 , 0 },
	{ "Cheleftdes", 0x04cb , 0 },
	{ "cheleftdes", 0x04cc , 0 },
	{ "qibuts", 0x05bb , 0 },
	{ "meteg", 0x05bd , 0 },
	{ "maqaf", 0x05be , 0 },
	{ "paseq", 0x05c0 , 0 },
	{ "shindot", 0x05c1 , 0 },
	{ "sindot", 0x05c2 , 0 },
	{ "sofpasuq", 0x05c3 , 0 },
	{ "kaffinal", 0x05da , 0 },
	{ "memfinal", 0x05dd , 0 },
	{ "nunfinal", 0x05df , 0 },
	{ "pefinal", 0x05e3 , 0 },
	{ "tsadifinal", 0x05e5 , 0 },
	{ "vavdbl", 0x05f0 , 0 },
	{ "vavyod", 0x05f1 , 0 },
	{ "yoddbl", 0x05f2 , 0 },
	{ "geresh", 0x05f3 , 0 },
	{ "gershayim", 0x05f4 , 0 },
	{ "varika", 0x05f5 , 0 },
	{ "enquad", 0x2000 , 0 },
	{ "emquad", 0x2001 , 0 },
	{ "emspace", 0x2003 , 0 },
	{ "threeperemspace", 0x2004 , 0 },
	{ "fourperemspace", 0x2005 , 0 },
	{ "sixperemspace", 0x2006 , 0 },
	{ "figurespace", 0x2007 , 0 },
	{ "punctuationspace", 0x2008 , 0 },
	{ "thinspace", 0x2009 , 0 },
	{ "hairspace", 0x200a , 0 },
	{ "zerospace", 0x200b , 0 },
	{ "zeronojoin", 0x200c , 0 },
	{ "zerojoin", 0x200d , 0 },
	{ "hyphennobreak", 0x2011 , 0 },
	{ "quotedash", 0x2015 , 0 },
	{ "dashdbl", 0x2016 , 0 },
	{ "quotesinglrev", 0x201b , 0 },
	{ "quotedblrev", 0x201f , 0 },
	{ "trianglebullet", 0x2023 , 0 },
	{ "onedotlead", 0x2024 , 0 },
	{ "twodotlead", 0x2025 , 0 },
	{ "hyphendot", 0x2027 , 0 },
	{ "lineseparator", 0x2028 , 0 },
	{ "paragraphseparator", 0x2029 , 0 },
	{ "lre", 0x202a , 0 },
	{ "rle", 0x202b , 0 },
	{ "pdf", 0x202c , 0 },
	{ "lro", 0x202d , 0 },
	{ "rlo", 0x202e , 0 },
	{ "pertenthousand", 0x2031 , 0 },
	{ "prime", 0x2032 , 0 },
	{ "primedbl", 0x2033 , 0 },
	{ "primetriple", 0x2034 , 0 },
	{ "primerev", 0x2035 , 0 },
	{ "primedblrev", 0x2036 , 0 },
	{ "primetriplerev", 0x2037 , 0 },
	{ "caret", 0x2038 , 0 },
	{ "refmark", 0x203b , 0 },
	{ "interrobang", 0x203d , 0 },
	{ "tie", 0x2040 , 0 },
	{ "caretinsert", 0x2041 , 0 },
	{ "hyphenbullet", 0x2043 , 0 },
	{ "minussuperior", 0x207b , 0 },
	{ "plusinferior", 0x208a , 0 },
	{ "equalinferior", 0x208c , 0 },
	{ "eurocurrency", 0x20a0 , 0 },
	{ "coloncurrency", 0x20a1 , 0 },
	{ "mill", 0x20a5 , 0 },
	{ "naira", 0x20a6 , 0 },
	{ "pesetas", 0x20a7 , 0 },
	{ "rupee", 0x20a8 , 0 },
	{ "newsheqel", 0x20aa , 0 },
	{ "accountof", 0x2100 , 0 },
	{ "addresssubject", 0x2101 , 0 },
	{ "Cbb", 0x2102 , 0 },
	{ "degreecentigrade", 0x2103 , 0 },
	{ "CL", 0x2104 , 0 },
	{ "cadauna", 0x2106 , 0 },
	{ "Euler", 0x2107 , 0 },
	{ "scruple", 0x2108 , 0 },
	{ "degreefarenheit", 0x2109 , 0 },
	{ "Hscript", 0x210b , 0 },
	{ "Hblackletter", 0x210c , 0 },
	{ "Hbb", 0x210d , 0 },
	{ "planck", 0x210e , 0 },
	{ "planckover2pi", 0x210f , 0 },
	{ "Iscript", 0x2110 , 0 },
	{ "Lscript", 0x2112 , 0 },
	{ "lscript", 0x2113 , 0 },
	{ "lbbar", 0x2114 , 0 },
	{ "Nbb", 0x2115 , 0 },
	{ "recordright", 0x2117 , 0 },
	{ "Pbb", 0x2119 , 0 },
	{ "Qbb", 0x211a , 0 },
	{ "Rscript", 0x211b , 0 },
	{ "Rfractur", 0x211c , 0 },
	{ "Rbb", 0x211d , 0 },
	{ "Rx", 0x211e , 0 },
	{ "response", 0x211f , 0 },
	{ "servicemark", 0x2120 , 0 },
	{ "tel", 0x2121 , 0 },
	{ "versicle", 0x2123 , 0 },
	{ "Zbb", 0x2124 , 0 },
	{ "ounce", 0x2125 , 0 },
	{ "ohm", 0x2126 , 0 },
	{ "mho", 0x2127 , 0 },
	{ "Zblackletter", 0x2128 , 0 },
	{ "iotaturn", 0x2129 , 0 },
	{ "degreekelvin", 0x212a , 0 },
	{ "Bscript", 0x212c , 0 },
	{ "Cblackletter", 0x212d , 0 },
	{ "escript", 0x212f , 0 },
	{ "Escript", 0x2130 , 0 },
	{ "Fscript", 0x2131 , 0 },
	{ "Fturn", 0x2132 , 0 },
	{ "Mscript", 0x2133 , 0 },
	{ "u0scrip", 0x2134 , 0 },
	{ "alephmath", 0x2135 , 0 },
	{ "gimelmath", 0x2137 , 0 },
	{ "dalethmath", 0x2138 , 0 },
	{ "twothird", 0x2154 , 0 },
	{ "onefifth", 0x2155 , 0 },
	{ "twofifths", 0x2156 , 0 },
	{ "threefifths", 0x2157 , 0 },
	{ "fourfifths", 0x2158 , 0 },
	{ "onesixth", 0x2159 , 0 },
	{ "fivesixths", 0x215a , 0 },
	{ "onenumerator", 0x215f , 0 },
	{ "arrowlongboth", 0x2194 , 0 },
	{ "arrowlongbothv", 0x2195 , 0 },
	{ "arrownorthwest", 0x2196 , 0 },
	{ "arrownortheast", 0x2197 , 0 },
	{ "arrowsoutheast", 0x2198 , 0 },
	{ "arrowsouthwest", 0x2199 , 0 },
	{ "arrowleftnot", 0x219a , 0 },
	{ "arrowrightnot", 0x219b , 0 },
	{ "arrowwaveleft", 0x219c , 0 },
	{ "arrowwaveright", 0x219d , 0 },
	{ "dblarrowheadleft", 0x219e , 0 },
	{ "dblarrowheadup", 0x219f , 0 },
	{ "dblarrowheadright", 0x21a0 , 0 },
	{ "dblarrowheaddown", 0x21a1 , 0 },
	{ "arrowtailleft", 0x21a2 , 0 },
	{ "arrowtailright", 0x21a3 , 0 },
	{ "arrowbarleft", 0x21a4 , 0 },
	{ "arrowbarup", 0x21a5 , 0 },
	{ "arrowbarright", 0x21a6 , 0 },
	{ "arrowbardown", 0x21a7 , 0 },
	{ "arrowbothvbase", 0x21a8 , 0 },
	{ "arrowhookleft", 0x21a9 , 0 },
	{ "arrowhookright", 0x21aa , 0 },
	{ "arrowloopleft", 0x21ab , 0 },
	{ "arrowloopright", 0x21ac , 0 },
	{ "arrowwaveboth", 0x21ad , 0 },
	{ "arrowlongbothnot", 0x21ae , 0 },
	{ "arrowzigzag", 0x21af , 0 },
	{ "arrowrightdown", 0x21b4 , 0 },
	{ "carriagerreturn", 0x21b5 , 0 },
	{ "arrowsemanticlockw", 0x21b6 , 0 },
	{ "arrowsemclockw", 0x21b7 , 0 },
	{ "home", 0x21b8 , 0 },
	{ "tableftright", 0x21b9 , 0 },
	{ "arrowanticlockw", 0x21ba , 0 },
	{ "arrowclockw", 0x21bb , 0 },
	{ "arrowlefttophalf", 0x21bc , 0 },
	{ "arrowleftbothalf", 0x21bd , 0 },
	{ "harpoonupright", 0x21be , 0 },
	{ "harpoonupleft", 0x21bf , 0 },
	{ "arrowrighttophalf", 0x21c0 , 0 },
	{ "arrowrightbothalf", 0x21c1 , 0 },
	{ "harpoondownright", 0x21c2 , 0 },
	{ "harpoondownleft", 0x21c3 , 0 },
	{ "arrowparrrightleft", 0x21c4 , 0 },
	{ "dblarrowupdown", 0x21c5 , 0 },
	{ "arrowparrleftright", 0x21c6 , 0 },
	{ "dblarrowup", 0x21c8 , 0 },
	{ "dblarrowdown", 0x21ca , 0 },
	{ "harpoonleftright", 0x21cb , 0 },
	{ "harpoonrightleft", 0x21cc , 0 },
	{ "arrowdblleftnot", 0x21cd , 0 },
	{ "arrowdbllongbothnot", 0x21ce , 0 },
	{ "arrowdblrightnot", 0x21cf , 0 },
	{ "arrowdbllongboth", 0x21d4 , 0 },
	{ "arrowdbllongbothv", 0x21d5 , 0 },
	{ "arrowdblnw", 0x21d6 , 0 },
	{ "arrowdblne", 0x21d7 , 0 },
	{ "arrowdblse", 0x21d8 , 0 },
	{ "arrowdblsw", 0x21d9 , 0 },
	{ "arrowtripleleft", 0x21da , 0 },
	{ "arrowtripleright", 0x21db , 0 },
	{ "arrowsquiggleleft", 0x21dc , 0 },
	{ "arrowsquiggleright", 0x21dd , 0 },
	{ "arrowopenleft", 0x21e6 , 0 },
	{ "arrowopenup", 0x21e7 , 0 },
	{ "arrowopenright", 0x21e8 , 0 },
	{ "arrowopendown", 0x21e9 , 0 },
	{ "complement", 0x2201 , 0 },
	{ "notexistential", 0x2204 , 0 },
	{ "elementsmall", 0x220a , 0 },
	{ "owner", 0x220b , 0 },
	{ "notowner", 0x220c , 0 },
	{ "ownersmall", 0x220d , 0 },
	{ "eop", 0x220e , 0 },
	{ "coproduct", 0x2210 , 0 },
	{ "dotplus", 0x2214 , 0 },
	{ "slashmath", 0x2215 , 0 },
	{ "backslashmath", 0x2216 , 0 },
	{ "ringoperator", 0x2218 , 0 },
	{ "bulletmath", 0x2219 , 0 },
	{ "cuberoot", 0x221b , 0 },
	{ "fourthroot", 0x221c , 0 },
	{ "measuredangle", 0x2221 , 0 },
	{ "sphericalangle", 0x2222 , 0 },
	{ "notbar", 0x2224 , 0 },
	{ "parallelto", 0x2225 , 0 },
	{ "notbardbl", 0x2226 , 0 },
	{ "integraldbl", 0x222c , 0 },
	{ "integraltrpl", 0x222d , 0 },
	{ "contintegral", 0x222e , 0 },
	{ "surfintegral", 0x222f , 0 },
	{ "volintegral", 0x2230 , 0 },
	{ "clwintegral", 0x2231 , 0 },
	{ "clwcontintegral", 0x2232 , 0 },
	{ "cclwcontintegral", 0x2233 , 0 },
	{ "dotminus", 0x2238 , 0 },
	{ "excess", 0x2239 , 0 },
	{ "geomproportion", 0x223a , 0 },
	{ "homothetic", 0x223b , 0 },
	{ "revsimilar", 0x223d , 0 },
	{ "lazysinv", 0x223e , 0 },
	{ "sine", 0x223f , 0 },
	{ "wreathproduct", 0x2240 , 0 },
	{ "notsimilar", 0x2241 , 0 },
	{ "minustilde", 0x2242 , 0 },
	{ "asymptequal", 0x2243 , 0 },
	{ "notasymptequal", 0x2244 , 0 },
	{ "approxorequal", 0x2245 , 0 },
	{ "approxnotequal", 0x2246 , 0 },
	{ "notapproxequal", 0x2247 , 0 },
	{ "almostequal", 0x2248 , 0 },
	{ "notalmostequal", 0x2249 , 0 },
	{ "almostorequal", 0x224a , 0 },
	{ "tildetrpl", 0x224b , 0 },
	{ "equivasymptotic", 0x224d , 0 },
	{ "geomequivalent", 0x224e , 0 },
	{ "difference", 0x224f , 0 },
	{ "approachlimit", 0x2250 , 0 },
	{ "geomequal", 0x2251 , 0 },
	{ "imageorapproxequal", 0x2253 , 0 },
	{ "colonequal", 0x2254 , 0 },
	{ "equalcolon", 0x2255 , 0 },
	{ "ringinequal", 0x2256 , 0 },
	{ "ringequal", 0x2257 , 0 },
	{ "corresponds", 0x2258 , 0 },
	{ "estimates", 0x2259 , 0 },
	{ "equiangular", 0x225a , 0 },
	{ "starequal", 0x225b , 0 },
	{ "deltaequal", 0x225c , 0 },
	{ "definequal", 0x225d , 0 },
	{ "measurequal", 0x225e , 0 },
	{ "questionequal", 0x225f , 0 },
	{ "notequivalence", 0x2262 , 0 },
	{ "strictequivalence", 0x2263 , 0 },
	{ "lessdblequal", 0x2266 , 0 },
	{ "greaterdblequal", 0x2267 , 0 },
	{ "lessnotdblequal", 0x2268 , 0 },
	{ "greaternotdblequal", 0x2269 , 0 },
	{ "lessmuch", 0x226a , 0 },
	{ "greatermuch", 0x226b , 0 },
	{ "between", 0x226c , 0 },
	{ "notequivasymptotic", 0x226d , 0 },
	{ "notlessequal", 0x2270 , 0 },
	{ "notgreaterequal", 0x2271 , 0 },
	{ "lessequivlnt", 0x2272 , 0 },
	{ "greaterequivlnt", 0x2273 , 0 },
	{ "notlessequivlnt", 0x2274 , 0 },
	{ "notgreaterequivlnt", 0x2275 , 0 },
	{ "notlessgreater", 0x2278 , 0 },
	{ "notgreaterless", 0x2279 , 0 },
	{ "follows", 0x227b , 0 },
	{ "precedesequal", 0x227c , 0 },
	{ "followsequal", 0x227d , 0 },
	{ "precedequivlnt", 0x227e , 0 },
	{ "followsequivlnt", 0x227f , 0 },
	{ "notpreceeds", 0x2280 , 0 },
	{ "notfollows", 0x2281 , 0 },
	{ "notpropersubset", 0x2284 , 0 },
	{ "notpropersuperset", 0x2285 , 0 },
	{ "notreflexsubset", 0x2288 , 0 },
	{ "notreflexsuperset", 0x2289 , 0 },
	{ "multiset", 0x228c , 0 },
	{ "multiplymultiset", 0x228d , 0 },
	{ "unionmulti", 0x228e , 0 },
	{ "squareimage", 0x228f , 0 },
	{ "squareoriginal", 0x2290 , 0 },
	{ "subsetsqequal", 0x2291 , 0 },
	{ "supersetsqequal", 0x2292 , 0 },
	{ "intersectionsq", 0x2293 , 0 },
	{ "unionsq", 0x2294 , 0 },
	{ "circleminus", 0x2296 , 0 },
	{ "circledivide", 0x2298 , 0 },
	{ "circledot", 0x2299 , 0 },
	{ "circlering", 0x229a , 0 },
	{ "circleasterisk", 0x229b , 0 },
	{ "circleequal", 0x229c , 0 },
	{ "circlevertbar", 0x229d , 0 },
	{ "squareplus", 0x229e , 0 },
	{ "squareminus", 0x229f , 0 },
	{ "squaremultiply", 0x22a0 , 0 },
	{ "squaredot", 0x22a1 , 0 },
	{ "turnstileleft", 0x22a2 , 0 },
	{ "turnstileright", 0x22a3 , 0 },
	{ "latticetop", 0x22a4 , 0 },
	{ "assertion", 0x22a6 , 0 },
	{ "truestate", 0x22a7 , 0 },
	{ "satisfy", 0x22a8 , 0 },
	{ "force", 0x22a9 , 0 },
	{ "tacktrpl", 0x22aa , 0 },
	{ "forceextr", 0x22ab , 0 },
	{ "notturnstileleft", 0x22ac , 0 },
	{ "notsatisfy", 0x22ad , 0 },
	{ "notforce", 0x22ae , 0 },
	{ "notforceextr", 0x22af , 0 },
	{ "lowerrank", 0x22b0 , 0 },
	{ "higherrank", 0x22b1 , 0 },
	{ "triangleright", 0x22b2 , 0 },
	{ "triangleleft", 0x22b3 , 0 },
	{ "triangleftequal", 0x22b4 , 0 },
	{ "triangrightequal", 0x22b5 , 0 },
	{ "original", 0x22b6 , 0 },
	{ "image", 0x22b7 , 0 },
	{ "multimap", 0x22b8 , 0 },
	{ "hermitconjmatrix", 0x22b9 , 0 },
	{ "intercal", 0x22ba , 0 },
	{ "xor", 0x22bb , 0 },
	{ "nand", 0x22bc , 0 },
	{ "nor", 0x22bd , 0 },
	{ "rightanglearc", 0x22be , 0 },
	{ "narylogicaland", 0x22c0 , 0 },
	{ "narylogicalor", 0x22c1 , 0 },
	{ "naryintersection", 0x22c2 , 0 },
	{ "naryunion", 0x22c3 , 0 },
	{ "diamondmath", 0x22c4 , 0 },
	{ "divideonmultiply", 0x22c7 , 0 },
	{ "bowtie", 0x22c8 , 0 },
	{ "multicloseleft", 0x22c9 , 0 },
	{ "multicloseright", 0x22ca , 0 },
	{ "multiopenleft", 0x22cb , 0 },
	{ "multiopenright", 0x22cc , 0 },
	{ "revasymptequal", 0x22cd , 0 },
	{ "curlor", 0x22ce , 0 },
	{ "curland", 0x22cf , 0 },
	{ "subsetdbl", 0x22d0 , 0 },
	{ "supersetdbl", 0x22d1 , 0 },
	{ "uniondbl", 0x22d2 , 0 },
	{ "intersectiondbl", 0x22d3 , 0 },
	{ "fork", 0x22d4 , 0 },
	{ "equalparallel", 0x22d5 , 0 },
	{ "lessdot", 0x22d6 , 0 },
	{ "greaterdot", 0x22d7 , 0 },
	{ "verymuchless", 0x22d8 , 0 },
	{ "verymuchgreater", 0x22d9 , 0 },
	{ "lessequalgreater", 0x22da , 0 },
	{ "greaterequalless", 0x22db , 0 },
	{ "equalless", 0x22dc , 0 },
	{ "equalgreater", 0x22dd , 0 },
	{ "equalprecedes", 0x22de , 0 },
	{ "equalfollows", 0x22df , 0 },
	{ "preceedsnotequal", 0x22e0 , 0 },
	{ "followsnotequal", 0x22e1 , 0 },
	{ "notsubsetsqequal", 0x22e2 , 0 },
	{ "notsupersetsqequal", 0x22e3 , 0 },
	{ "sqimageornotequal", 0x22e4 , 0 },
	{ "sqoriginornotequal", 0x22e5 , 0 },
	{ "lessnotequivlnt", 0x22e6 , 0 },
	{ "greaternotequivlnt", 0x22e7 , 0 },
	{ "preceedsnotsimilar", 0x22e8 , 0 },
	{ "followsnotequivlnt", 0x22e9 , 0 },
	{ "nottriangleleft", 0x22ea , 0 },
	{ "nottriangleright", 0x22eb , 0 },
	{ "nottriangleleftequal", 0x22ec , 0 },
	{ "nottrianglerightequal", 0x22ed , 0 },
	{ "vertellipsis", 0x22ee , 0 },
	{ "midhorizellipsis", 0x22ef , 0 },
	{ "upslopeellipsis", 0x22f0 , 0 },
	{ "downslopeellipsis", 0x22f1 , 0 },
	{ "perspcorrespond", 0x2306 , 0 },
	{ "ceilingleft", 0x2308 , 0 },
	{ "ceilingright", 0x2309 , 0 },
	{ "floorleft", 0x230a , 0 },
	{ "floorright", 0x230b , 0 },
	{ "slurabove", 0x2322 , 0 },
	{ "slurbelow", 0x2323 , 0 },
	{ "null", 0x2400 , 0 },
	{ "startofhead", 0x2401 , 0 },
	{ "starttext", 0x2402 , 0 },
	{ "endtext", 0x2403 , 0 },
	{ "endtrans", 0x2404 , 0 },
	{ "enquiry", 0x2405 , 0 },
	{ "acknowledge", 0x2406 , 0 },
	{ "bell", 0x2407 , 0 },
	{ "backspace", 0x2408 , 0 },
	{ "horiztab", 0x2409 , 0 },
	{ "linefeed", 0x240a , 0 },
	{ "verttab", 0x240b , 0 },
	{ "formfeed", 0x240c , 0 },
	{ "shiftout", 0x240e , 0 },
	{ "shiftin", 0x240f , 0 },
	{ "datalinkescape", 0x2410 , 0 },
	{ "devcon1", 0x2411 , 0 },
	{ "devcon2", 0x2412 , 0 },
	{ "devcon3", 0x2413 , 0 },
	{ "devcon4", 0x2414 , 0 },
	{ "negacknowledge", 0x2415 , 0 },
	{ "synch", 0x2416 , 0 },
	{ "endtransblock", 0x2417 , 0 },
	{ "cancel", 0x2418 , 0 },
	{ "endmedium", 0x2419 , 0 },
	{ "substitute", 0x241a , 0 },
	{ "escape", 0x241b , 0 },
	{ "fileseparator", 0x241c , 0 },
	{ "groupseparator", 0x241d , 0 },
	{ "recordseparator", 0x241e , 0 },
	{ "unitseparator", 0x241f , 0 },
	{ "spaceliteral", 0x2420 , 0 },
	{ "delete", 0x2421 , 0 },
	{ "blankb", 0x2422 , 0 },
	{ "spaceopenbox", 0x2423 , 0 },
	{ "newline", 0x2424 , 0 },
	{ "lthorizform", 0x2500 , 0 },
	{ "hvhorizform", 0x2501 , 0 },
	{ "ltvertform", 0x2502 , 0 },
	{ "hvvertform", 0x2503 , 0 },
	{ "ltdashtriphorizform", 0x2504 , 0 },
	{ "hvdashtriphorizform", 0x2505 , 0 },
	{ "ltdashtripvertform", 0x2506 , 0 },
	{ "hvdashtripvertform", 0x2507 , 0 },
	{ "ltdashquadhorizform", 0x2508 , 0 },
	{ "hvdashquadhorizform", 0x2509 , 0 },
	{ "ltdashquadvertform", 0x250a , 0 },
	{ "hvdashquadvertform", 0x250b , 0 },
	{ "ltdnrtform", 0x250c , 0 },
	{ "dnltrthvform", 0x250d , 0 },
	{ "dnhvrtltform", 0x250e , 0 },
	{ "hvdnrtform", 0x250f , 0 },
	{ "ltdnleftform", 0x2510 , 0 },
	{ "dnltlefthvform", 0x2511 , 0 },
	{ "dnhvleftltform", 0x2512 , 0 },
	{ "hvdnleftform", 0x2513 , 0 },
	{ "ltuprtform", 0x2514 , 0 },
	{ "upltrthvform", 0x2515 , 0 },
	{ "uphvrtltform", 0x2516 , 0 },
	{ "hvuprtform", 0x2517 , 0 },
	{ "ltupleftform", 0x2518 , 0 },
	{ "upltlefthvform", 0x2519 , 0 },
	{ "uphvleftltform", 0x251a , 0 },
	{ "hvupleftform", 0x251b , 0 },
	{ "ltvertrightform", 0x251c , 0 },
	{ "vertltrthvform", 0x251d , 0 },
	{ "uphvrtdnltform", 0x251e , 0 },
	{ "dnhvrtupltform", 0x251f , 0 },
	{ "verthvrtltform", 0x2520 , 0 },
	{ "dnltrtuphvform", 0x2521 , 0 },
	{ "upltrtdnhvform", 0x2522 , 0 },
	{ "hvvertrtform", 0x2523 , 0 },
	{ "ltvertleftform", 0x2524 , 0 },
	{ "vtltlefthvform", 0x2525 , 0 },
	{ "uphvleftdnltform", 0x2526 , 0 },
	{ "dnhvleftupltform", 0x2527 , 0 },
	{ "verthvleftltform", 0x2528 , 0 },
	{ "dnltleftuphvform", 0x2529 , 0 },
	{ "upltleftdnhvform", 0x252a , 0 },
	{ "hvvertleftform", 0x252b , 0 },
	{ "ltdnhorizform", 0x252c , 0 },
	{ "lefthvrtdnltform", 0x252d , 0 },
	{ "rthvleftdnltform", 0x252e , 0 },
	{ "dnlthorizhvform", 0x252f , 0 },
	{ "dnhvhorizltform", 0x2530 , 0 },
	{ "rtltrtdnhvform", 0x2531 , 0 },
	{ "leftltrtdnhvform", 0x2532 , 0 },
	{ "hvdnhorizform", 0x2533 , 0 },
	{ "ltuphorizform", 0x2534 , 0 },
	{ "lefthvrtupltform", 0x2535 , 0 },
	{ "rthvleftupltform", 0x2536 , 0 },
	{ "uplthorizhvform", 0x2537 , 0 },
	{ "uphvhorizltform", 0x2538 , 0 },
	{ "rtltleftuphvform", 0x2539 , 0 },
	{ "leftltrtuphvform", 0x253a , 0 },
	{ "hvuphorizform", 0x253b , 0 },
	{ "ltverthorizform", 0x253c , 0 },
	{ "lefthvrtvertltform", 0x253d , 0 },
	{ "rthvleftvertltform", 0x253e , 0 },
	{ "vertlthorizhvform", 0x253f , 0 },
	{ "uphvdnhorizltform", 0x2540 , 0 },
	{ "dnhvuphorizltform", 0x2541 , 0 },
	{ "verthvhorizltform", 0x2542 , 0 },
	{ "leftuphvrtdnltform", 0x2543 , 0 },
	{ "rtuphvleftdnltform", 0x2544 , 0 },
	{ "leftdnhvrtupltform", 0x2545 , 0 },
	{ "rtdnhvleftupltform", 0x2546 , 0 },
	{ "dnltuphorizhvform", 0x2547 , 0 },
	{ "upltdnhorizhvform", 0x2548 , 0 },
	{ "rtltleftverthvform", 0x2549 , 0 },
	{ "leftltrtverthvform", 0x254a , 0 },
	{ "hvverthorizform", 0x254b , 0 },
	{ "ltdashdblhorizform", 0x254c , 0 },
	{ "hvdashdblhorizform", 0x254d , 0 },
	{ "ltdashdblvertform", 0x254e , 0 },
	{ "hvdashdblvertform", 0x254f , 0 },
	{ "horizdblbar", 0x2550 , 0 },
	{ "vertdblbar", 0x2551 , 0 },
	{ "dnrtdblform", 0x2552 , 0 },
	{ "dndblrtform", 0x2553 , 0 },
	{ "dbldnrtform", 0x2554 , 0 },
	{ "dnleftdblform", 0x2555 , 0 },
	{ "dndblleftform", 0x2556 , 0 },
	{ "dbldnleftform", 0x2557 , 0 },
	{ "uprtdblform", 0x2558 , 0 },
	{ "updblrtform", 0x2559 , 0 },
	{ "dbluprtform", 0x255a , 0 },
	{ "upleftdblform", 0x255b , 0 },
	{ "updblleftform", 0x255c , 0 },
	{ "dblupleftform", 0x255d , 0 },
	{ "vertrtdblform", 0x255e , 0 },
	{ "vertdblrtform", 0x255f , 0 },
	{ "dblvertrtform", 0x2560 , 0 },
	{ "vertleftdblform", 0x2561 , 0 },
	{ "vertdblleftform", 0x2562 , 0 },
	{ "dblvertleftform", 0x2563 , 0 },
	{ "dnhorizdblform", 0x2564 , 0 },
	{ "dndblhorizform", 0x2565 , 0 },
	{ "dbldnhorizform", 0x2566 , 0 },
	{ "uphorizdblform", 0x2567 , 0 },
	{ "updblhorizform", 0x2568 , 0 },
	{ "dbluphorizform", 0x2569 , 0 },
	{ "verthorizdblform", 0x256a , 0 },
	{ "vertdblhorizform", 0x256b , 0 },
	{ "dblverthorizform", 0x256c , 0 },
	{ "ltarcdnrtform", 0x256d , 0 },
	{ "ltarcdnleftform", 0x256e , 0 },
	{ "ltarcupleftform", 0x256f , 0 },
	{ "ltarcuprtform", 0x2570 , 0 },
	{ "forwarddiagonal", 0x2571 , 0 },
	{ "backwarddiagonal", 0x2572 , 0 },
	{ "ltdiagonalcross", 0x2573 , 0 },
	{ "dneighthblock", 0x2581 , 0 },
	{ "dnquarterblock", 0x2582 , 0 },
	{ "dnthreeeighthblock", 0x2583 , 0 },
	{ "dnfiveeighthblock", 0x2585 , 0 },
	{ "dnthreequarterblock", 0x2586 , 0 },
	{ "dnseveneighthblock", 0x2587 , 0 },
	{ "lfseveneighthblock", 0x2589 , 0 },
	{ "lfthreequarterblock", 0x258a , 0 },
	{ "lffiveeighthblock", 0x258b , 0 },
	{ "lfthreeeighthblock", 0x258d , 0 },
	{ "lfquarterblock", 0x258e , 0 },
	{ "lfeighthblock", 0x258f , 0 },
	{ "upeighthblock", 0x2594 , 0 },
	{ "rteighthblock", 0x2595 , 0 },
	{ "box", 0x25a1 , 0 },
	{ "boxrounded", 0x25a2 , 0 },
	{ "boxnested", 0x25a3 , 0 },
	{ "boxhorizhatch", 0x25a4 , 0 },
	{ "boxverthatch", 0x25a5 , 0 },
	{ "boxcrosshatch", 0x25a6 , 0 },
	{ "boxleftdiaghatch", 0x25a7 , 0 },
	{ "boxrtdiaghatch", 0x25a8 , 0 },
	{ "boxcrossdiaghatch", 0x25a9 , 0 },
	{ "smallboxfilled", 0x25aa , 0 },
	{ "smallbox", 0x25ab , 0 },
	{ "rectangle", 0x25ad , 0 },
	{ "filledvertrect", 0x25ae , 0 },
	{ "vertrectangle", 0x25af , 0 },
	{ "filledparallelogram", 0x25b0 , 0 },
	{ "parallelogram", 0x25b1 , 0 },
	{ "triangle", 0x25b3 , 0 },
	{ "smalltrianglesld", 0x25b4 , 0 },
	{ "smalltriangle", 0x25b5 , 0 },
	{ "trianglerightsld1", 0x25b6 , 0 },
	{ "triangleright1", 0x25b7 , 0 },
	{ "smalltrianglerightsld", 0x25b8 , 0 },
	{ "smalltriangleright", 0x25b9 , 0 },
	{ "triagrtopen", 0x25bb , 0 },
	{ "triangleinv", 0x25bd , 0 },
	{ "smalltriangleinvsld", 0x25be , 0 },
	{ "smalltriangleinv", 0x25bf , 0 },
	{ "triangleleftsld1", 0x25c0 , 0 },
	{ "triangleleft1", 0x25c1 , 0 },
	{ "smalltriangleleftsld", 0x25c2 , 0 },
	{ "smalltriangleleft", 0x25c3 , 0 },
	{ "triaglfopen", 0x25c5 , 0 },
	{ "diamondrhombsolid", 0x25c6 , 0 },
	{ "diamondrhomb", 0x25c7 , 0 },
	{ "diamondrhombnested", 0x25c8 , 0 },
	{ "circledash", 0x25cc , 0 },
	{ "circleverthatch", 0x25cd , 0 },
	{ "circlesolid", 0x25cf , 0 },
	{ "circleleftsld", 0x25d0 , 0 },
	{ "circlerightsld", 0x25d1 , 0 },
	{ "circlebottomsld", 0x25d2 , 0 },
	{ "circletopsld", 0x25d3 , 0 },
	{ "circlenesld", 0x25d4 , 0 },
	{ "circlenwopen", 0x25d5 , 0 },
	{ "semicircleleftsld", 0x25d6 , 0 },
	{ "semicirclelertsld", 0x25d7 , 0 },
	{ "invsemicircleup", 0x25da , 0 },
	{ "invsemicircledn", 0x25db , 0 },
	{ "nwquadarc", 0x25dc , 0 },
	{ "nequadarc", 0x25dd , 0 },
	{ "sequadarc", 0x25de , 0 },
	{ "swquadarc", 0x25df , 0 },
	{ "toparc", 0x25e0 , 0 },
	{ "bottomarc", 0x25e1 , 0 },
	{ "trianglesesld", 0x25e2 , 0 },
	{ "triangleswsld", 0x25e3 , 0 },
	{ "tranglenwsld", 0x25e4 , 0 },
	{ "trianglenesld", 0x25e5 , 0 },
	{ "squareleftsld", 0x25e7 , 0 },
	{ "squarerightsld", 0x25e8 , 0 },
	{ "squarenwsld", 0x25e9 , 0 },
	{ "squaresesld", 0x25ea , 0 },
	{ "squarevertbisect", 0x25eb , 0 },
	{ "triangledot", 0x25ec , 0 },
	{ "triangleleftsld", 0x25ed , 0 },
	{ "trianglerightsld", 0x25ee , 0 },
	{ "heartopen", 0x2661 , 0 },
	{ "diamondopen", 0x2662 , 0 },
	{ "spadeopen", 0x2664 , 0 },
	{ "Omegaiotasubleniscircumflex", 0x1fae , 0 },
	{ "Omegaleniscircumflex", 0x1f6e , 0 },
	{ "Upsilonaspercircumflex", 0x1f5f , 0 },
	{ "dieresiscircumflex", 0x1fc1 , 0 },
	{ "leniscircumflex", 0x1fcf , 0 },
	{ "aspercircumflex", 0x1fdf , 0 },
	{ "alphaleniscircumflex", 0x1f06 , 0 },
	{ "alphaaspercircumflex", 0x1f07 , 0 },
	{ "etaleniscircumflex", 0x1f26 , 0 },
	{ "etaaspercircumflex", 0x1f27 , 0 },
	{ "iotaleniscircumflex", 0x1f36 , 0 },
	{ "iotaaspercircumflex", 0x1f37 , 0 },
	{ "upsilonleniscircumflex", 0x1f56 , 0 },
	{ "upsilonaspercircumflex", 0x1f57 , 0 },
	{ "omegaleniscircumflex", 0x1f66 , 0 },
	{ "omegaaspercircumflex", 0x1f67 , 0 },
	{ "alphaiotasubleniscircumflex", 0x1f86 , 0 },
	{ "alphaiotasubaspercircumflex", 0x1f87 , 0 },
	{ "etaiotasubleniscircumflex", 0x1f96 , 0 },
	{ "etaiotasubaspercircumflex", 0x1f97 , 0 },
	{ "omegaiotasubleniscircumflex", 0x1fa6 , 0 },
	{ "omegaiotasubaspercircumflex", 0x1fa7 , 0 },
	{ "alphacircumflex", 0x1fb6 , 0 },
	{ "alphaiotasubcircumflex", 0x1fb7 , 0 },
	{ "etacircumflex", 0x1fc6 , 0 },
	{ "etaiotasubcircumflex", 0x1fc7 , 0 },
	{ "iotacircumflex", 0x1fd6 , 0 },
	{ "iotadieresiscircumflex", 0x1fd7 , 0 },
	{ "upsiloncircumflex", 0x1fe6 , 0 },
	{ "omegacircumflex", 0x1ff6 , 0 },
	{ "omegaiotasubcircumflex", 0x1ff7 , 0 },
	{ "upsilondieresiscircumflex", 0x1fe7 , 0 },
	{ "dialytika", 0x0308 , 0 },
	{ "koronis", 0x1fbd , 0 },
	{ "prosgegrammeni", 0x1fbe , 0 },
	{ "psili", 0x1fbf , 0 },
	{ "perispomeni", 0x1fc0 , 0 },
	{ "varia", 0x1fef , 0 },
	{ "oxia", 0x1ffd , 0 },
	{ "dasia", 0x1ffe , 0 },
	{ "Alphasmall", 0xf500 , 0 },
	{ "Betasmall", 0xf501 , 0 },
	{ "Gammasmall", 0xf502 , 0 },
	{ "Deltasmall", 0xf503 , 0 },
	{ "Epsilonsmall", 0xf504 , 0 },
	{ "Zetasmall", 0xf505 , 0 },
	{ "Etasmall", 0xf506 , 0 },
	{ "Thetasmall", 0xf507 , 0 },
	{ "Iotasmall", 0xf508 , 0 },
	{ "Kappasmall", 0xf509 , 0 },
	{ "Lambdasmall", 0xf50a , 0 },
	{ "Musmall", 0xf50b , 0 },
	{ "Nusmall", 0xf50c , 0 },
	{ "Xismall", 0xf50d , 0 },
	{ "Omicronsmall", 0xf50e , 0 },
	{ "Pismall", 0xf50f , 0 },
	{ "Rhosmall", 0xf510 , 0 },
	{ "Sigmasmall", 0xf512 , 0 },
	{ "Tausmall", 0xf513 , 0 },
	{ "Upsilonsmall", 0xf514 , 0 },
	{ "Phismall", 0xf515 , 0 },
	{ "Chismall", 0xf516 , 0 },
	{ "Psismall", 0xf517 , 0 },
	{ "Omegasmall", 0xf518 , 0 },
	{ "Iotadieresissmall", 0xf519 , 0 },
	{ "Upsilondieresissmall", 0xf51a , 0 },
	{ "uni2A0B.lgdisplay", 0xea57 , 0 },
	{ "uni2A0B.uplgdisplay", 0xea57 , 0 },
	{ "uni2A15.lgdisplay", 0xe376 , 0 },
	{ "uni2A15.uplgdisplay", 0xe376 , 0 },
	{ "uni2A16.lgdisplay", 0xe377 , 0 },
	{ "uni2A16.uplgdisplay", 0xe377 , 0 },
	{ "uni2A10.lgdisplay", 0xe395 , 0 },
	{ "uni2A10.uplgdisplay", 0xe395 , 0 },
	{ "uni2A12.lgdisplay", 0xe397 , 0 },
	{ "uni2A12.uplgdisplay", 0xe397 , 0 },
	{ "uni2A13.lgdisplay", 0xe398 , 0 },
	{ "uni2A13.uplgdisplay", 0xe398 , 0 },
	{ "uni2A14.lgdisplay", 0xe399 , 0 },
	{ "uni2A14.uplgdisplay", 0xe399 , 0 },
	{ "uni2A17.lgdisplay", 0xe39a , 0 },
	{ "uni2A17.uplgdisplay", 0xe39a , 0 },
	{ "uni2A11.uplgdisplay", 0xe39b , 0 },
	{ "uni2A11.lgdisplay", 0xe39b , 0 },
	{ "uni2A0F.lgdisplay", 0xe3d3 , 0 },
	{ "uni2A0F.uplgdisplay", 0xe3d3 , 0 },
/* From Richard Kinch's TeX list of glyph aliases */
	{ "fscript", 0x192 , 0 },
	{ "fraction1", 0x2215 , 0 },
	{ "negationslash", 0x2215 , 0 },
	{ "circleR", 0xae , 0 },
	{ "circlecopyrt", 0xa9 , 0 },
	{ "smile", 0x263a , 0 },
	{ "Ifractur", 0x2111 , 0 },
	{ "Rfractur", 0x211C , 0 },
	{ "Omegainv", 0x2127 , 0 },
	{ "mho", 0x2127 , 0 },
	{ "alephmath", 0x2135 , 0 },
	{ "beth", 0x2136 , 0 },
	{ "bethmath", 0x2136 , 0 },
	{ "gimelmath", 0x2137 , 0 },
	{ "daleth", 0x2138 , 0 },
	{ "daletmath", 0x2138 , 0 },
	{ "arrowbothv", 0x2195 , 0 },
	{ "prime1", 0x2032 , 0 },
	{ "primerev1", 0x2035 , 0 },
	{ "primereverse", 0x2035 , 0 },
	{ "followsequal1", 0x227d , 0 },
	{ "similarequal", 0x2243 , 0 },
	{ "square", 0x25a1 , 0 },
	{ "squaresolid", 0x25a0 , 0 },
	{ "squaresmallsolid", 0x25aa , 0 },
	{ "diamondsolid", 0x25c6 , 0 },
	{ "clockwise", 0x21bb , 0 },
	{ "anticlockwise", 0x21ba , 0 },
	{ "forces", 0x22a9 , 0 },
	{ "forcesbar", 0x22aa , 0 },
	{ "satisfies", 0x22a8 , 0 },
	{ "dblarrowdwn", 0x21ca , 0 },
	{ "shiftleft", 0x2196 , 0 },
	{ "shiftright", 0x2197 , 0 },
	{ "squiggleright", 0x21dd , 0 },
	{ "squiggleleft", 0x21dc , 0 },
	{ "squiggleleftright", 0x21ad , 0 },
	{ "curlyleft", 0x21ab , 0 },
	{ "curlyright", 0x21ac , 0 },
	{ "followsorequal", 0x227d , 0 },
	{ "equalsdots", 0x2251 , 0 },
	{ "defines", 0x225c , 0 },
	{ "ng", 0x014b , 0 },
	{ "Ng", 0x014a , 0 },
	{ "visiblespace", 0x2420 , 0 },
	{ "dslash", 0x0111 , 0 },
	{ "tie1", 0x2040 , 0 },
	{ "arrowdblbothv", 0x21d5 , 0 },
	{ "precedesequal1", 0x227c , 0 },
	{ "greaterorsimilar", 0x2273 , 0 },
	{ "precedesorequal", 0x227e , 0 },
	{ "lessorsimilar", 0x2272 , 0 },
	{ "equalorgreater", 0x22dd , 0 },
	{ "lessorequalslant", 0x2264 , 0 },
	{ "equaldotrightleft", 0x2253 , 0 },
	{ "equaldotleftright", 0x2252 , 0 },
	{ "followsorcurly", 0x227d , 0 },
	{ "greaterorequalslant", 0x2265 , 0 },
	{ "trianglerightequal", 0x22b5 , 0 },
	{ "triangleleftequal", 0x22b4 , 0 },
	{ "triangledownsld", 0x25bc , 0 },
	{ "arrowaxisright", 0x2192 , 0 },
	{ "arrowaxisleft", 0x2190 , 0 },
	{ "trianglesolid", 0x25b2 , 0 },
	{ "greaterlessequal", 0x22db , 0 },
	{ "orunderscore", 0x22bb , 0 },
	{ "frown", 0x2322 , 0 },
	{ "uprise", 0x22cf , 0 },
	{ "downfall", 0x22ce , 0 },
	{ "subsetdblequal", 0x2286 , 0 },
	{ "supersetdblequal", 0x2287 , 0 },
	{ "Finv", 0x2132 , 0 },
	{ "notarrowboth", 0x21ae , 0 },
	{ "archleftdown", 0x21b6 , 0 },
	{ "archrightdown", 0x21b7 , 0 },
	{ "notdblarrowleft", 0x21cd , 0 },
	{ "notdblarrowboth", 0x21ce , 0 },
	{ "notdblarrowright", 0x21cf , 0 },
	{ "epsiloninv", 0x220a , 0 },
	{ "equalorsimilar", 0x2242 , 0 },
	{ "notprecedes", 0x2280 , 0 },
	{ "notsubseteql", 0x2288 , 0 },
	{ "notsuperseteql", 0x2289 , 0 },
	{ "subsetnoteql", 0x228a , 0 },
	{ "supersetnoteql", 0x228b , 0 },
	{ "notturnstile", 0x22ac , 0 },
	{ "notforcesextra", 0x22af , 0 },
	{ "dividemultiply", 0x22c7 , 0 },
	{ "notprecedesoreql", 0x22e0 , 0 },
	{ "notfollowsoreql", 0x22e1 , 0 },
	{ "lessornotsimilar", 0x22e6 , 0 },
	{ "greaterornotsimilar", 0x22e7 , 0 },
	{ "precedeornoteqvlnt", 0x22e8 , 0 },
	{ "followornoteqvlnt", 0x22e9 , 0 },
	{ "nottriangeqlleft", 0x22ec , 0 },
	{ "nottriangeqlright", 0x22ed , 0 },
	{ "angbracketleft", 0x3008 , 0 },
	{ "angbracketright", 0x3009 , 0 },
	{ "check", 0x2713 , 0 },
	{ "circleS", 0x24c8 , 0 },
	{ "rightanglenw", 0x250c , 0 },
	{ "rightanglene", 0x2510 , 0 },
	{ "rightanglesw", 0x2514 , 0 },
	{ "rightanglese", 0x2518 , 0 },
	{ "Yen", 0x00a5 , 0 },
	{ "permill", 0x2030 , 0 },
	{ "recipe", 0x211e , 0 },
	{ "pertenmill", 0x2031 , 0 },
	{ "lnot", 0x00ac , 0 },
	{ "circleP", 0x2117 , 0 },
	{ "surd", 0x221a , 0 },
	{ "asciigrave", 0x60 , 0 },
	{ "asciigrave", 0x60 , 0 },
	{ "asciiacute", 0xb4 , 0 },
	{ "asciidieresis", 0xa8 , 0 },
	{ "pilcrow", 0xb6 , 0 },
/* Useful names to use to find glyphs */
	{ "circumflexcomb", 0x302 , 0 },
	{ "macroncomb", 0x304 , 0 },
	{ "brevecomb", 0x306 , 0 },
	{ "diaeresiscomb", 0x308 , 0 },
	{ "caroncomb", 0x30c , 0 },
	{ "vrachycomb", 0x306 , 0 },
	{ "psilicomb", 0x313 , 0 },
	{ "dasiacomb", 0x314 , 0 },
	{ "psilivaria", 0x1fcd , 0 },
	{ "psilioxia", 0x1fce , 0 },
	{ "psiliperispomeni", 0x1fcf , 0 },
	{ "dasiavaria", 0x1fdd , 0 },
	{ "dasiaoxia", 0x1fde , 0 },
	{ "dasiaperispomeni", 0x1fdf , 0 },
	{ "dialytikavaria", 0x1fed , 0 },
	{ "dialytikaoxia", 0x1fee , 0 },
	{ "dialytikaperispomeni", 0x1fc1 , 0 },
/* Adobe spells diaeresis the American way, as dieresis */
	{ "diaeresis", 0xa8 , 0 },
	{ "Adiaeresis", 0xC4 , 0 },
	{ "Ediaeresis", 0xCB , 0 },
	{ "Idiaeresis", 0xCF , 0 },
	{ "Odiaeresis", 0xD6 , 0 },
	{ "Udiaeresis", 0xDC , 0 },
	{ "adiaeresis", 0xE4 , 0 },
	{ "ediaeresis", 0xEB , 0 },
	{ "idiaeresis", 0xEF , 0 },
	{ "odiaeresis", 0xF6 , 0 },
	{ "udiaeresis", 0xFC , 0 },
	{ "Ydiaeresis", 0x178 , 0 },
	{ "diaeresistonos", 0x385 , 0 },
	{ "iotadiaeresistonos", 0x390 , 0 },
	{ "Iotadiaeresis", 0x3AA , 0 },
	{ "Upsilondiaeresis", 0x3AB , 0 },
	{ "upsilondiaeresistonos", 0x3B0 , 0 },
	{ "iotadiaeresis", 0x3CA , 0 },
	{ "upsilondiaeresis", 0x3CB , 0 },
	{ "Wdiaeresis", 0x1e84 , 0 },
	{ "wdiaeresis", 0x1e85 , 0 },
	{ "ETH", 0xd0 , 0 },
	{ "THORN", 0xde , 0 },
	{ "ssharp", 0xdf , 0 },
	{ "Ooblique", 0xd8 , 0 },
	{ "notsign", 0xac , 0 },
/* Sun has used "masculine" for ordmasculine */
	{ NULL, 0, 0 },
};
